/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-srt.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-plugin.h"

#include "ogmjob-queue.h"
#include "ogmjob-exec.h"

#include <unistd.h>
#include <string.h>
#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

static gint ogmrip_srt_run (OGMJobSpawn *spawn);

static gdouble
ogmrip_pgm_watch (OGMJobExec *exec, const gchar *buffer, OGMRipSubp *subp)
{
  guint files;

  if (sscanf (buffer, "%u files generated", &files) == 1)
  {
    g_object_set_data (G_OBJECT (subp), "__ogmrip_srt_index__", GUINT_TO_POINTER (0));
    g_object_set_data (G_OBJECT (subp), "__ogmrip_srt_files__", GUINT_TO_POINTER (files));
  }

  return -1.0;
}

static gdouble
ogmrip_ocr_watch (OGMJobExec *exec, const gchar *buffer, OGMRipSubp *subp)
{
  if (strncmp (buffer, "Elapsed time:", 13) == 0)
  {
    guint index, files;

    index = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (subp), "__ogmrip_srt_index__"));
    files = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (subp), "__ogmrip_srt_files__"));

    g_object_set_data (G_OBJECT (subp), "__ogmrip_srt_index__", GUINT_TO_POINTER (index + 1));

    return 0.98 + 0.02 * (index + 1) / (gdouble) files;
  }

  return -1.0;
}

static gchar **
ogmrip_pgm_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("subp2pgm"));

  if (ogmrip_subp_get_forced_only (subp))
    g_ptr_array_add (argv, g_strdup ("--forced"));

  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

#ifdef HAVE_GOCR_SUPPORT
static gchar **
ogmrip_ocr_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("gocr"));
  g_ptr_array_add (argv, g_strdup ("-v"));
  g_ptr_array_add (argv, g_strdup ("1"));
  g_ptr_array_add (argv, g_strdup ("-f"));

  switch (ogmrip_subp_get_charset (subp))
  {
    case OGMRIP_CHARSET_UTF8:
      g_ptr_array_add (argv, g_strdup ("UTF8"));
      break;
    case OGMRIP_CHARSET_ISO8859_1:
      g_ptr_array_add (argv, g_strdup ("ISO8859_1"));
      break;
    case OGMRIP_CHARSET_ASCII:
      g_ptr_array_add (argv, g_strdup ("ASCII"));
      break;
  }

  g_ptr_array_add (argv, g_strdup ("-m"));
  g_ptr_array_add (argv, g_strdup ("4"));
  g_ptr_array_add (argv, g_strdup ("-m"));
  g_ptr_array_add (argv, g_strdup ("64"));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strconcat (input, ".txt", NULL));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}
#else
static gchar **
ogmrip_ocr_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("ocrad"));
  g_ptr_array_add (argv, g_strdup ("-v"));
  g_ptr_array_add (argv, g_strdup ("-f"));
  g_ptr_array_add (argv, g_strdup ("-F"));

  switch (ogmrip_subp_get_charset (subp))
  {
    case OGMRIP_CHARSET_UTF8:
      g_ptr_array_add (argv, g_strdup ("utf8"));
      break;
    case OGMRIP_CHARSET_ISO8859_1:
    case OGMRIP_CHARSET_ASCII:
      g_ptr_array_add (argv, g_strdup ("byte"));
      break;
  }

  g_ptr_array_add (argv, g_strdup ("-l"));
  g_ptr_array_add (argv, g_strdup ("0"));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strconcat (input, ".txt", NULL));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}
#endif

static gchar **
ogmrip_srt_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (subp));
  g_return_val_if_fail (output != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("srtutil"));
  g_ptr_array_add (argv, g_strdup ("-s"));
/*
  g_ptr_array_add (argv, g_strdup ("-w"));
*/
  if (ogmrip_subp_get_eol_style (OGMRIP_SUBP (subp)))
  {
    g_ptr_array_add (argv, g_strdup ("-f"));
    g_ptr_array_add (argv, g_strdup ("dos"));
  }

  g_ptr_array_add (argv, g_strdup ("-i"));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar **
ogmrip_vobsub_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  argv = ogmrip_mencoder_vobsub_command (subp, NULL, output);
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipSrt, ogmrip_srt, OGMRIP_TYPE_SUBP)

static void
ogmrip_srt_class_init (OGMRipSrtClass *klass)
{
  OGMJobSpawnClass *spawn_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  spawn_class->run = ogmrip_srt_run;
}

static void
ogmrip_srt_init (OGMRipSrt *srt)
{
}

static gint
ogmrip_srt_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gint result, fd;
  gchar **argv;

  GDir *dir;
  GPatternSpec *pspec;
  gchar *pattern, *str;
  const gchar *name;

  gchar *tmp_file, *srtx_file;

  result = OGMJOB_RESULT_ERROR;

  fd = g_file_open_tmp ("sub.XXXXXX", &tmp_file, NULL);
  if (fd < 0)
    return OGMJOB_RESULT_ERROR;
  g_unlink (tmp_file);
  close (fd);

  srtx_file = g_strconcat (tmp_file, ".srtx", NULL);

  argv = ogmrip_vobsub_command (OGMRIP_SUBP (spawn), NULL, tmp_file);
  if (argv)
  {
    child = ogmjob_exec_newv (argv);
    ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mencoder_vobsub_watch, spawn, TRUE, FALSE, FALSE);
    ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
    g_object_unref (child);

    result = OGMJOB_SPAWN_CLASS (ogmrip_srt_parent_class)->run (spawn);

    ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);
  }

  if (result == OGMJOB_RESULT_COMPLETED)
  {
    result = OGMJOB_RESULT_ERROR;

    argv = ogmrip_pgm_command (OGMRIP_SUBP (spawn), tmp_file, NULL);
    if (argv)
    {
      child = ogmjob_exec_newv (argv);
      ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_pgm_watch, spawn, TRUE, FALSE, FALSE);
      result = ogmjob_spawn_run (child, NULL);
      g_object_unref (child);
    }
  }

  if (result == OGMJOB_RESULT_COMPLETED)
  {
    result = OGMJOB_RESULT_ERROR;

    dir = g_dir_open (g_get_tmp_dir (), 0, NULL);
    if (dir)
    {
      gchar *basename;

      basename = g_path_get_basename (tmp_file);
      pattern = g_strconcat (basename, "*.pgm", NULL);
      pspec = g_pattern_spec_new (pattern);
      g_free (basename);
      g_free (pattern);

      while ((name = g_dir_read_name (dir)))
      {
        if (g_pattern_match (pspec, strlen (name), name, NULL))
        {
          str = g_build_filename (g_get_tmp_dir (), name, NULL);
          argv = ogmrip_ocr_command (OGMRIP_SUBP (spawn), str, NULL);

          if (argv)
          {
            child = ogmjob_exec_newv (argv);
            ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_ocr_watch, spawn, FALSE, TRUE, TRUE);
            result = ogmjob_spawn_run (child, NULL);
            g_object_unref (child);

            if (result != OGMJOB_RESULT_COMPLETED)
              break;
          }

          g_unlink (str);
          g_free (str);
        }
      }

      g_pattern_spec_free (pspec);
      g_dir_close (dir);
    }
  }

  if (result == OGMJOB_RESULT_COMPLETED)
  {
    result = OGMJOB_RESULT_ERROR;

    argv = ogmrip_srt_command (OGMRIP_SUBP (spawn), srtx_file, NULL);
    if (argv)
    {
      child = ogmjob_exec_newv (argv);
      result = ogmjob_spawn_run (child, NULL);
      g_object_unref (child);
    }
  }

  if (g_file_test (srtx_file, G_FILE_TEST_EXISTS))
    g_unlink (srtx_file);
  g_free (srtx_file);

  srtx_file = g_strconcat (tmp_file, ".idx", NULL);
  if (g_file_test (srtx_file, G_FILE_TEST_EXISTS))
    g_unlink (srtx_file);
  g_free (srtx_file);

  srtx_file = g_strconcat (tmp_file, ".sub", NULL);
  if (g_file_test (srtx_file, G_FILE_TEST_EXISTS))
    g_unlink (srtx_file);
  g_free (srtx_file);

  dir = g_dir_open (g_get_tmp_dir (), 0, NULL);
  if (dir)
  {
    gchar *basename;

    basename = g_path_get_basename (tmp_file);
    pattern = g_strconcat (basename, "*.pgm*", NULL);
    pspec = g_pattern_spec_new (pattern);
    g_free (basename);
    g_free (pattern);

    while ((name = g_dir_read_name (dir)))
    {
      if (g_pattern_match (pspec, strlen (name), name, NULL))
      {
        str = g_build_filename (g_get_tmp_dir (), name, NULL);
        g_unlink (str);
        g_free (str);
      }
    }

    g_pattern_spec_free (pspec);
    g_dir_close (dir);
  }

  g_free (tmp_file);

  return result;
}

/**
 * ogmrip_srt_new:
 * @subp: An #OGMDvdSubpStream
 * @output: The output file
 *
 * Creates a new #OGMRipSrt.
 *
 * Returns: The new #OGMRipSrt
 */
OGMJobSpawn *
ogmrip_srt_new (OGMDvdSubpStream *subp, const gchar *output)
{
  g_return_val_if_fail (subp != NULL, NULL);
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_SRT, "stream", subp, "output", output, NULL);
}

static OGMRipPluginContainer srt_plugin =
{
  NULL,
  G_TYPE_NONE,
  "srt",
  N_("SRT text"),
  OGMRIP_FORMAT_SRT,
  TRUE
};

OGMRipPluginContainer *
ogmrip_init_plugin (void)
{
  srt_plugin.type = OGMRIP_TYPE_SRT;

  return &srt_plugin;
}

