/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmdvd.h"

#include "ogmrip-fs.h"
#include "ogmrip-pref.h"
#include "ogmrip-gconf.h"
#include "ogmrip-helper.h"
#include "ogmrip-plugin.h"
#include "ogmrip-preferences.h"

#include <string.h>
#include <glib/gi18n.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-pref.glade"
#define OGMRIP_GLADE_ROOT "notebook"

#define OGMRIP_PREF_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_PREF_DIALOG, OGMRipPrefDialogPriv))

struct _OGMRipPrefDialogPriv
{
  GtkWidget *container_combo;
  GtkWidget *passes_spin;
  GtkWidget *threads_spin;

  GtkWidget *text_table;

  GtkWidget *video_table;
  GSList *video_radio_list;

  GtkWidget *audio_table;
  GSList *audio_radio_list;

  GtkWidget *subp_table;
  GSList *subp_radio_list;
};

static void ogmrip_pref_dialog_finalize (GObject *gobject);

extern gchar *ogmdvd_languages[][3];
extern guint  ogmdvd_nlanguages;

/*
 * Toggle
 */

static void
ogmrip_pref_dialog_connect_toggle (GladeXML *xml, const gchar *name, const gchar *key, gboolean def)
{
  GtkWidget *widget;
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_bool (key, def);

  widget = glade_xml_get_widget (xml, name);
  if (widget)
    ogmrip_preferences_connect_toggle (widget, key);
}

/*
 * Spin
 */

static void
ogmrip_pref_dialog_connect_spin (GladeXML *xml, const gchar *name, const gchar *key, gint def)
{
  GtkWidget *widget;
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (key, def);

  widget = glade_xml_get_widget (xml, name);
  if (widget)
    ogmrip_preferences_connect_spin (widget, key);
}

/*
 * Combo
 */

static void
ogmrip_pref_dialog_connect_combo (GladeXML *xml, const gchar *name, const gchar *key, gint def)
{
  GtkWidget *widget;
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (key, def);

  widget = glade_xml_get_widget (xml, name);
  if (widget)
    ogmrip_preferences_connect_combo (widget, key);
}

/*
 * Lang
 */

static GConfValue *
ogmrip_pref_dialog_lang_get_value (GtkWidget *combo)
{
  GConfValue *value;
  const gchar *lang;
  gint i;

  value = gconf_value_new (GCONF_VALUE_INT);

  i = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  if (i > 0 && i + 1 < ogmdvd_nlanguages)
  {
    lang = ogmdvd_languages[i + 1][OGMDVD_LANGUAGE_ISO639_1];
    gconf_value_set_int (value, (lang[0] << 8) | lang[1]);
  }

  return value;
}

static gboolean
ogmrip_pref_dialog_lang_set_value (GtkWidget *combo, GConfValue *value)
{
  gint code;

  if (!value)
    return FALSE;

  code = gconf_value_get_int (value);
  if (!code)
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
  else
  {
    const gchar *lang;
    gint i;

    lang = ogmdvd_get_language_iso639_1 (code);

    for (i = 2; i < ogmdvd_nlanguages; i++)
      if (strcmp (ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1], lang) == 0)
        break;

    if (i == ogmdvd_nlanguages)
      return FALSE;

    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i - 1);
  }

  return TRUE;
}

static void
ogmrip_pref_dialog_connect_lang (GtkWidget *widget, const gchar *key, gint def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (key, def);

  if (widget)
    ogmrip_preferences_connect_combo_custom (widget, key,
        ogmrip_pref_dialog_lang_get_value,
        ogmrip_pref_dialog_lang_set_value);
}

/*
 * Chooser
 */

static void
ogmrip_pref_dialog_connect_output_chooser (GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (key, def);

  if (widget)
    ogmrip_preferences_connect_chooser (widget, key);
}

static GConfValue *
ogmrip_pref_dialog_tmp_chooser_get_value (GtkWidget *chooser)
{
  GConfValue *value;
  gchar *str;

  value = gconf_value_new (GCONF_VALUE_STRING);

  str = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
  if (!str)
    return NULL;

  gconf_value_set_string (value, str);
  g_free (str);
                
  return value;
}

static gboolean
ogmrip_pref_dialog_tmp_chooser_set_value (GtkWidget *chooser, GConfValue *value)
{
  const gchar *new_path;
  gchar *old_path;

  if (!value)
    return FALSE;

  new_path = gconf_value_get_string (value);
  if (!g_file_test (new_path, G_FILE_TEST_IS_DIR))
    return FALSE;

  ogmrip_fs_set_tmp_dir (new_path);

  old_path = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER (chooser));
  if (!old_path || !g_str_equal (new_path, old_path))
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser), new_path);

  return TRUE;
}

static void
ogmrip_pref_dialog_connect_tmp_chooser (GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (key, def);

  if (widget)
    ogmrip_preferences_connect_chooser_custom (widget, key,
        ogmrip_pref_dialog_tmp_chooser_get_value,
        ogmrip_pref_dialog_tmp_chooser_set_value);
}

/*
 * Container
 */

static void
ogmrip_pref_dialog_container_set_active (GtkWidget *combo, const gchar *name)
{
  GtkTreeModel *model;
  GtkTreeIter iter;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
  if (gtk_tree_model_iter_children (model, &iter, NULL))
  {
    gchar *str;
    do
    {
      gtk_tree_model_get (model, &iter, 1, &str, -1);
      if (g_str_equal (str, name))
      {
        gtk_combo_box_set_active_iter (GTK_COMBO_BOX (combo), &iter);
        g_free (str);
        break;
      }
      g_free (str);
    }
    while (gtk_tree_model_iter_next (model, &iter));
  }

}

static GConfValue *
ogmrip_pref_dialog_container_get_value (GtkWidget *combo)
{
  GConfValue *value;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *name;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
  if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo), &iter))
    return NULL;

  gtk_tree_model_get (model, &iter, 1, &name, -1);

  value = gconf_value_new (GCONF_VALUE_STRING);
  gconf_value_set_string (value, name);
  g_free (name);

  return value;
}

static gboolean
ogmrip_pref_dialog_container_set_value (GtkWidget *combo, GConfValue *value)
{
  GType container;
  const gchar *name;

  if (!value)
    return FALSE;

  name = gconf_value_get_string (value);
  container = ogmrip_gconf_get_container_type (name);

  if (container == G_TYPE_NONE)
    return FALSE;

  ogmrip_pref_dialog_container_set_active (combo, name);

  return TRUE;
}

static void
ogmrip_pref_dialog_connect_container (GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (key, def);

  if (widget)
    ogmrip_preferences_connect_combo_custom (widget, key,
        ogmrip_pref_dialog_container_get_value,
        ogmrip_pref_dialog_container_set_value);
}

static void
ogmrip_pref_dialog_codec_select_first (GSList *list)
{
  GSList *link;

  for (link = list; link; link = link->next)
    if (GTK_WIDGET_IS_SENSITIVE (link->data))
      break;

  if (link)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (link->data), TRUE);
}

static void
ogmrip_pref_dialog_check_container (GConfValue *value, OGMRipPrefDialog *dialog)
{
  GType container;
  const gchar *name = NULL;

  if (value)
    name = gconf_value_get_string (value);
  container = ogmrip_gconf_get_container_type (name);

  if (container != G_TYPE_NONE)
  {
    GType codec;
    GSList *link;
    gint i, select_other;
    gboolean can_contain;

    for (i = 0, select_other = 0, link = dialog->priv->video_radio_list; link; i++, link = link->next)
    {
      codec = ogmrip_plugin_get_nth_video_codec (i);
      can_contain = container != G_TYPE_NONE && ogmrip_plugin_can_contain_video (container, codec);
      gtk_widget_set_sensitive (GTK_WIDGET (link->data), can_contain);
      select_other = select_other || (!can_contain && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (link->data)));
    }

    if (select_other)
      ogmrip_pref_dialog_codec_select_first (dialog->priv->video_radio_list);

    for (i = 0, select_other = 0, link = dialog->priv->audio_radio_list; link; i++, link = link->next)
    {
      codec = ogmrip_plugin_get_nth_audio_codec (i);
      can_contain = container != G_TYPE_NONE && ogmrip_plugin_can_contain_audio (container, codec);
      gtk_widget_set_sensitive (GTK_WIDGET (link->data), can_contain);
      select_other = select_other || (!can_contain && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (link->data)));
    }

    if (select_other)
      ogmrip_pref_dialog_codec_select_first (dialog->priv->audio_radio_list);

    for (i = 0, select_other = 0, link = dialog->priv->subp_radio_list; link; i++, link = link->next)
    {
      codec = ogmrip_plugin_get_nth_subp_codec (i);
      can_contain = container != G_TYPE_NONE && ogmrip_plugin_can_contain_subp (container, codec);
      gtk_widget_set_sensitive (GTK_WIDGET (link->data), can_contain);
      select_other = select_other || (!can_contain && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (link->data)));
    }

    if (select_other)
      ogmrip_pref_dialog_codec_select_first (dialog->priv->subp_radio_list);
  }
}

/*
 * Codec
 */

static void
ogmrip_pref_dialog_codec_set_active (GSList *group, const gchar *name)
{
  gchar *str;

  while (group)
  {
    str = g_object_get_data (G_OBJECT (group->data), "__name__");
    if (g_str_equal (str, name))
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (group->data), TRUE);
    group = group->next;
  }
}

static GConfValue *
ogmrip_pref_dialog_codec_get_value (GtkWidget *codec)
{
  GConfValue *value;
  gchar *name;

  value = gconf_value_new (GCONF_VALUE_STRING);
  name = g_object_get_data (G_OBJECT (codec), "__name__");
  gconf_value_set_string (value, name);

  return value;
}

/*
 * Video Codec
 */

static gboolean
ogmrip_pref_dialog_video_codec_set_value (GtkWidget *codec, GConfValue *value)
{
  GType type, container;
  const gchar *name;

  if (!value)
    return FALSE;

  name = gconf_value_get_string (value);
  type = ogmrip_gconf_get_video_codec_type (name);

  if (type == G_TYPE_NONE)
    return FALSE;

  container = ogmrip_gconf_get_container_type (NULL);
  if (!ogmrip_plugin_can_contain_video (container, type))
    return FALSE;

  ogmrip_pref_dialog_codec_set_active (
      gtk_radio_button_get_group (GTK_RADIO_BUTTON (codec)), name);

  return TRUE;
}

static void
ogmrip_pref_dialog_connect_video_codec (GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (key, def);

  if (widget)
    ogmrip_preferences_connect_radio_custom (widget, key,
        ogmrip_pref_dialog_codec_get_value,
        ogmrip_pref_dialog_video_codec_set_value);
}

static void
ogmrip_pref_dialog_check_video_codec (GConfValue *value, OGMRipPrefDialog *dialog)
{
  GType codec;
  const gchar *name = NULL;

  if (value)
    name = gconf_value_get_string (value);
  codec = ogmrip_gconf_get_video_codec_type (name);

  if (codec != G_TYPE_NONE)
  {
    gint value;

    value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->passes_spin));
    gtk_spin_button_set_range (GTK_SPIN_BUTTON (dialog->priv->passes_spin), 1.0,
        MIN (100, ogmrip_plugin_get_video_codec_passes (codec)));
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->passes_spin), value);

    value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->priv->threads_spin));
    gtk_spin_button_set_range (GTK_SPIN_BUTTON (dialog->priv->threads_spin), 1.0,
        MIN (100, ogmrip_plugin_get_video_codec_threads (codec)));
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->priv->threads_spin), value);
  }
}


/*
 * Audio Codec
 */

static gboolean
ogmrip_pref_dialog_audio_codec_set_value (GtkWidget *codec, GConfValue *value)
{
  GType type, container;
  const gchar *name;

  if (!value)
    return FALSE;

  name = gconf_value_get_string (value);
  type = ogmrip_gconf_get_audio_codec_type (name);

  if (type == G_TYPE_NONE)
    return FALSE;

  container = ogmrip_gconf_get_container_type (NULL);
  if (!ogmrip_plugin_can_contain_audio (container, type))
    return FALSE;

  ogmrip_pref_dialog_codec_set_active (
      gtk_radio_button_get_group (GTK_RADIO_BUTTON (codec)), name);

  return TRUE;
}

static void
ogmrip_pref_dialog_connect_audio_codec (GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (key, def);

  if (widget)
    ogmrip_preferences_connect_radio_custom (widget, key,
        ogmrip_pref_dialog_codec_get_value,
        ogmrip_pref_dialog_audio_codec_set_value);
}

static void
ogmrip_pref_dialog_check_audio_codec (GConfValue *value, OGMRipPrefDialog *dialog)
{
}

/*
 * Subp Codec
 */

static gboolean
ogmrip_pref_dialog_subp_codec_set_value (GtkWidget *codec, GConfValue *value)
{
  GType type, container;
  const gchar *name;

  if (!value)
    return FALSE;

  name = gconf_value_get_string (value);
  type = ogmrip_gconf_get_subp_codec_type (name);

  if (type == G_TYPE_NONE)
    return FALSE;

  container = ogmrip_gconf_get_container_type (NULL);
  if (!ogmrip_plugin_can_contain_subp (container, type))
    return FALSE;

  ogmrip_pref_dialog_codec_set_active (
      gtk_radio_button_get_group (GTK_RADIO_BUTTON (codec)), name);

  return TRUE;
}

static void
ogmrip_pref_dialog_connect_subp_codec (GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (key, def);

  if (widget)
    ogmrip_preferences_connect_radio_custom (widget, key,
        ogmrip_pref_dialog_codec_get_value,
        ogmrip_pref_dialog_subp_codec_set_value);
}

static void
ogmrip_pref_dialog_check_subp_codec (GConfValue *value, OGMRipPrefDialog *dialog)
{
  GType codec;
  const gchar *name = NULL;

  if (value)
    name = gconf_value_get_string (value);
  codec = ogmrip_gconf_get_subp_codec_type (name);

  if (codec != G_TYPE_NONE)
    gtk_widget_set_sensitive (dialog->priv->text_table, ogmrip_plugin_get_subp_codec_text (codec));
}

/*
 * Dialog
 */

G_DEFINE_TYPE (OGMRipPrefDialog, ogmrip_pref_dialog, GTK_TYPE_DIALOG);

static void
ogmrip_pref_dialog_append_container (GType type, const gchar *name, const gchar *description, gpointer model)
{
  GtkTreeIter iter;

  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, description, 1, name, -1);
}

static void
ogmrip_pref_dialog_add_containers (OGMRipPrefDialog *dialog)
{
  GtkCellRenderer *cell;
  GtkListStore *store;
  gint n_containers;

  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (dialog->priv->container_combo), GTK_TREE_MODEL (store));
  g_object_unref (store);

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (dialog->priv->container_combo), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (dialog->priv->container_combo), cell, "text", 0, NULL);

  n_containers = ogmrip_plugin_get_n_containers ();
  gtk_widget_set_sensitive (dialog->priv->container_combo, n_containers > 0);

  ogmrip_plugin_foreach_container (ogmrip_pref_dialog_append_container, store);
}

static GtkWidget *
ogmrip_pref_dialog_append_codec (GtkTable *table, const gchar *name, const gchar *description)
{
  static GSList *group = NULL;
  static guint row = 0, col = 0;
  GtkWidget *radio;

  if (!table->children)
  {
    group = NULL;
    col = row = 0;
  }

  radio = gtk_radio_button_new_with_label (group, description);
  g_object_set_data (G_OBJECT (radio), "__name__", (gpointer) name);

  gtk_table_attach (table, radio, col, col + 1, row, row + 1, GTK_FILL, 0, 0, 0);
  gtk_widget_show (radio);

  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));

  col = (col + 1) % 2;
  if (!col)
    row ++;

  return radio;
}

static void
ogmrip_pref_dialog_append_video_codec (GType type, const gchar *name, const gchar *description, OGMRipPrefDialog *dialog)
{
  GtkWidget *radio;

  radio = ogmrip_pref_dialog_append_codec (GTK_TABLE (dialog->priv->video_table), name, description);
  dialog->priv->video_radio_list = g_slist_append (dialog->priv->video_radio_list, radio);
}

static void
ogmrip_pref_dialog_add_video_codecs (OGMRipPrefDialog *dialog)
{
  gint n_codecs;

  n_codecs = ogmrip_plugin_get_n_video_codecs ();
  gtk_widget_set_sensitive (dialog->priv->video_table, n_codecs > 0);

  if (n_codecs > 0)
    gtk_table_resize (GTK_TABLE (dialog->priv->video_table), (n_codecs + 1) / 2, 2);

  ogmrip_plugin_foreach_video_codec ((OGMRipPluginFunc) ogmrip_pref_dialog_append_video_codec, dialog);
}

static void
ogmrip_pref_dialog_append_audio_codec (GType type, const gchar *name, const gchar *description, OGMRipPrefDialog *dialog)
{
  GtkWidget *radio;

  radio = ogmrip_pref_dialog_append_codec (GTK_TABLE (dialog->priv->audio_table), name, description);
  dialog->priv->audio_radio_list = g_slist_append (dialog->priv->audio_radio_list, radio);
}

static void
ogmrip_pref_dialog_add_audio_codecs (OGMRipPrefDialog *dialog)
{
  gint n_codecs;

  n_codecs = ogmrip_plugin_get_n_audio_codecs ();
  gtk_widget_set_sensitive (dialog->priv->audio_table, n_codecs > 0);

  if (n_codecs > 0)
    gtk_table_resize (GTK_TABLE (dialog->priv->audio_table), (n_codecs + 1) / 2, 2);

  ogmrip_plugin_foreach_audio_codec ((OGMRipPluginFunc) ogmrip_pref_dialog_append_audio_codec, dialog);
}

static void
ogmrip_pref_dialog_append_subp_codec (GType type, const gchar *name, const gchar *description, OGMRipPrefDialog *dialog)
{
  GtkWidget *radio;

  radio = ogmrip_pref_dialog_append_codec (GTK_TABLE (dialog->priv->subp_table), name, description);
  dialog->priv->subp_radio_list = g_slist_append (dialog->priv->subp_radio_list, radio);
}

static void
ogmrip_pref_dialog_add_subp_codecs (OGMRipPrefDialog *dialog)
{
  gint n_codecs;

  n_codecs = ogmrip_plugin_get_n_subp_codecs ();
  gtk_widget_set_sensitive (dialog->priv->subp_table, n_codecs > 0);

  if (n_codecs > 0)
    gtk_table_resize (GTK_TABLE (dialog->priv->subp_table), (n_codecs + 1) / 2, 2);

  ogmrip_plugin_foreach_subp_codec ((OGMRipPluginFunc) ogmrip_pref_dialog_append_subp_codec, dialog);
}

static void
ogmrip_pref_dialog_add_languages (GtkWidget *widget)
{
  gchar *str;
  gint i;

  for (i = 2; i < ogmdvd_nlanguages; i++)
  {
    str = g_strdup_printf ("%s (%s)", ogmdvd_languages[i][OGMDVD_LANGUAGE_NAME], 
        ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1]);
    gtk_combo_box_append_text (GTK_COMBO_BOX (widget), str);
    g_free (str);
  }
}

static void
ogmrip_pref_dialog_class_init (OGMRipPrefDialogClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->finalize = ogmrip_pref_dialog_finalize;

  g_type_class_add_private (klass, sizeof (OGMRipPrefDialogPriv));
}

static void
ogmrip_pref_dialog_init (OGMRipPrefDialog *dialog)
{
  GSList *child;
  GtkWidget *widget;
  GladeXML *xml;

  dialog->priv = OGMRIP_PREF_DIALOG_GET_PRIVATE (dialog);

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
      GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
      NULL);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Preferences"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);

  widget = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, TRUE, TRUE, 0);
  gtk_widget_show (widget);

  widget = glade_xml_get_widget (xml, "output-dir-chooser");
  ogmrip_pref_dialog_connect_output_chooser (widget, OGMRIP_GCONF_OUTPUT_DIR, OGMRIP_DEFAULT_OUTPUT_DIR);

  gtk_file_chooser_set_action (GTK_FILE_CHOOSER (widget), 
      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

  widget = glade_xml_get_widget (xml, "tmp-dir-chooser");
  ogmrip_pref_dialog_connect_tmp_chooser (widget, OGMRIP_GCONF_TMP_DIR, OGMRIP_DEFAULT_TMP_DIR);

  gtk_file_chooser_set_action (GTK_FILE_CHOOSER (widget), 
      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

  dialog->priv->container_combo = glade_xml_get_widget (xml, "container-combo");
  ogmrip_pref_dialog_add_containers (dialog);

  dialog->priv->video_table = glade_xml_get_widget (xml, "video-table");
  ogmrip_pref_dialog_add_video_codecs (dialog);

  dialog->priv->audio_table = glade_xml_get_widget (xml, "audio-table");
  ogmrip_pref_dialog_add_audio_codecs (dialog);

  dialog->priv->subp_table = glade_xml_get_widget (xml, "subp-table");
  ogmrip_pref_dialog_add_subp_codecs (dialog);

  /*
   * General
   */

  ogmrip_pref_dialog_connect_combo (xml, "filename-combo", OGMRIP_GCONF_FILENAME, OGMRIP_DEFAULT_FILENAME);
  ogmrip_pref_dialog_connect_combo (xml, "fourcc-combo", OGMRIP_GCONF_FOURCC, OGMRIP_DEFAULT_FOURCC);
  ogmrip_pref_dialog_connect_spin (xml, "tnumber-spin", OGMRIP_GCONF_TNUMBER, OGMRIP_DEFAULT_TNUMBER);
  ogmrip_pref_dialog_connect_spin (xml, "tsize-spin", OGMRIP_GCONF_TSIZE, OGMRIP_DEFAULT_TSIZE);
  ogmrip_pref_dialog_connect_toggle (xml, "ensure-sync-check", OGMRIP_GCONF_ENSURE_SYNC, OGMRIP_DEFAULT_ENSURE_SYNC);

  ogmrip_pref_dialog_connect_container (dialog->priv->container_combo, OGMRIP_GCONF_CONTAINER, OGMRIP_DEFAULT_CONTAINER);

  widget = glade_xml_get_widget (xml, "chapter-lang-combo");
  ogmrip_pref_dialog_add_languages (widget);
  ogmrip_pref_dialog_connect_lang (widget, OGMRIP_GCONF_CHAPTER_LANG, OGMRIP_DEFAULT_CHAPTER_LANG);

  /*
   * Video
   */

  for (child = dialog->priv->video_radio_list; child; child = child->next)
    ogmrip_pref_dialog_connect_video_codec (child->data, OGMRIP_GCONF_VIDEO_CODEC, OGMRIP_DEFAULT_VIDEO_CODEC);

  ogmrip_pref_dialog_connect_spin (xml, "passes-spin", OGMRIP_GCONF_VIDEO_PASSES, OGMRIP_DEFAULT_VIDEO_PASSES);
  ogmrip_pref_dialog_connect_spin (xml, "threads-spin", OGMRIP_GCONF_VIDEO_THREADS, OGMRIP_DEFAULT_VIDEO_THREADS);
  ogmrip_pref_dialog_connect_toggle (xml, "trellis-check", OGMRIP_GCONF_VIDEO_TRELLIS, OGMRIP_DEFAULT_VIDEO_TRELLIS);
  ogmrip_pref_dialog_connect_toggle (xml, "qpel-check", OGMRIP_GCONF_VIDEO_QPEL, OGMRIP_DEFAULT_VIDEO_QPEL);
  ogmrip_pref_dialog_connect_toggle (xml, "turbo-check", OGMRIP_GCONF_VIDEO_TURBO, OGMRIP_DEFAULT_VIDEO_TURBO);
  ogmrip_pref_dialog_connect_toggle (xml, "denoise-check", OGMRIP_GCONF_VIDEO_DENOISE, OGMRIP_DEFAULT_VIDEO_DENOISE);
  ogmrip_pref_dialog_connect_toggle (xml, "deblock-check", OGMRIP_GCONF_VIDEO_DEBLOCK, OGMRIP_DEFAULT_VIDEO_DEBLOCK);
  ogmrip_pref_dialog_connect_toggle (xml, "dering-check", OGMRIP_GCONF_VIDEO_DERING, OGMRIP_DEFAULT_VIDEO_DERING);
  ogmrip_pref_dialog_connect_combo (xml, "quality-combo", OGMRIP_GCONF_VIDEO_QUALITY, OGMRIP_DEFAULT_VIDEO_QUALITY);
  ogmrip_pref_dialog_connect_combo (xml, "scaler-combo", OGMRIP_GCONF_VIDEO_SCALER, OGMRIP_DEFAULT_VIDEO_SCALER);

  /*
   * Audio
   */

  for (child = dialog->priv->audio_radio_list; child; child = child->next)
    ogmrip_pref_dialog_connect_audio_codec (child->data, OGMRIP_GCONF_AUDIO_CODEC, OGMRIP_DEFAULT_AUDIO_CODEC);

  ogmrip_pref_dialog_connect_spin (xml, "quality-spin", OGMRIP_GCONF_AUDIO_QUALITY, OGMRIP_DEFAULT_AUDIO_QUALITY);
  ogmrip_pref_dialog_connect_toggle (xml, "normalize-check", OGMRIP_GCONF_AUDIO_NORMALIZE, OGMRIP_DEFAULT_AUDIO_NORMALIZE);
  ogmrip_pref_dialog_connect_combo (xml, "srate-combo", OGMRIP_GCONF_AUDIO_SRATE, OGMRIP_DEFAULT_AUDIO_SRATE);
  ogmrip_pref_dialog_connect_combo (xml, "channels-combo", OGMRIP_GCONF_AUDIO_CHANNELS, OGMRIP_DEFAULT_AUDIO_CHANNELS);

  widget = glade_xml_get_widget (xml, "pref-audio-combo");
  ogmrip_pref_dialog_add_languages (widget);
  ogmrip_pref_dialog_connect_lang (widget, OGMRIP_GCONF_PREF_AUDIO, OGMRIP_DEFAULT_PREF_AUDIO);

  /*
   * Subp
   */

  for (child = dialog->priv->subp_radio_list; child; child = child->next)
    ogmrip_pref_dialog_connect_subp_codec (child->data, OGMRIP_GCONF_SUBP_CODEC, OGMRIP_DEFAULT_SUBP_CODEC);

  ogmrip_pref_dialog_connect_toggle (xml, "spell-check", OGMRIP_GCONF_SPELL_CHECK, OGMRIP_DEFAULT_SPELL_CHECK);
  ogmrip_pref_dialog_connect_toggle (xml, "forced-subs-check", OGMRIP_GCONF_FORCED_SUBS, OGMRIP_DEFAULT_FORCED_SUBS);
  ogmrip_pref_dialog_connect_combo (xml, "charset-combo", OGMRIP_GCONF_SUBP_CHARSET, OGMRIP_DEFAULT_SUBP_CHARSET);
  ogmrip_pref_dialog_connect_combo (xml, "eol-combo", OGMRIP_GCONF_SUBP_EOL, OGMRIP_DEFAULT_SUBP_EOL);

  widget = glade_xml_get_widget (xml, "pref-subp-combo");
  ogmrip_pref_dialog_add_languages (widget);
  ogmrip_pref_dialog_connect_lang (widget, OGMRIP_GCONF_PREF_SUBP, OGMRIP_DEFAULT_PREF_SUBP);

  /*
   * Advanced
   */

  ogmrip_pref_dialog_connect_toggle (xml, "copy-dvd-check", OGMRIP_GCONF_COPY_DVD, OGMRIP_DEFAULT_COPY_DVD);
  ogmrip_pref_dialog_connect_combo (xml, "after-enc-combo", OGMRIP_GCONF_AFTER_ENC, OGMRIP_DEFAULT_AFTER_ENC);
  ogmrip_pref_dialog_connect_toggle (xml, "keep-tmp-check", OGMRIP_GCONF_KEEP_TMP, OGMRIP_DEFAULT_KEEP_TMP);
  ogmrip_pref_dialog_connect_toggle (xml, "log-check", OGMRIP_GCONF_LOG_OUTPUT, OGMRIP_DEFAULT_LOG_OUTPUT);

  dialog->priv->text_table = glade_xml_get_widget (xml, "text-table");
  dialog->priv->passes_spin = glade_xml_get_widget (xml, "passes-spin");
  dialog->priv->threads_spin = glade_xml_get_widget (xml, "threads-spin");

  widget = glade_xml_get_widget (xml, "spell-check");
#if HAVE_ENCHANT_SUPPORT
  gtk_widget_set_sensitive (widget, TRUE);
#else
  gtk_widget_set_sensitive (widget, FALSE);
#endif

  ogmrip_preferences_add_notify_while_alive (OGMRIP_GCONF_CONTAINER,
      (GFunc) ogmrip_pref_dialog_check_container, dialog, dialog);
  ogmrip_pref_dialog_check_container (NULL, dialog);

  ogmrip_preferences_add_notify_while_alive (OGMRIP_GCONF_VIDEO_CODEC,
      (GFunc) ogmrip_pref_dialog_check_video_codec, dialog, dialog);
  ogmrip_pref_dialog_check_video_codec (NULL, dialog);

  ogmrip_preferences_add_notify_while_alive (OGMRIP_GCONF_AUDIO_CODEC,
      (GFunc) ogmrip_pref_dialog_check_audio_codec, dialog, dialog);
  ogmrip_pref_dialog_check_audio_codec (NULL, dialog);

  ogmrip_preferences_add_notify_while_alive (OGMRIP_GCONF_SUBP_CODEC,
      (GFunc) ogmrip_pref_dialog_check_subp_codec, dialog, dialog);
  ogmrip_pref_dialog_check_subp_codec (NULL, dialog);

  g_object_unref (xml);
}

static void
ogmrip_pref_dialog_finalize (GObject *gobject)
{
  OGMRipPrefDialog *dialog = OGMRIP_PREF_DIALOG (gobject);

  g_slist_free (dialog->priv->video_radio_list);
  dialog->priv->video_radio_list = NULL;

  g_slist_free (dialog->priv->audio_radio_list);
  dialog->priv->audio_radio_list = NULL;

  g_slist_free (dialog->priv->subp_radio_list);
  dialog->priv->subp_radio_list = NULL;

  G_OBJECT_CLASS (ogmrip_pref_dialog_parent_class)->finalize (gobject);
}

GtkWidget *
ogmrip_pref_dialog_new (void)
{
  return g_object_new (OGMRIP_TYPE_PREF_DIALOG, NULL);
}

