/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2008 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/**
 * SECTION:ogmrip-options-plugin
 * @title: Plugins System
 * @short_description: Functions for manipulating the plugins
 * @include: ogmrip-options-plugin.h
 */

#include "ogmrip-options-plugin.h"
#include "ogmrip-container.h"
#include "ogmrip-codec.h"

#include <string.h>

typedef struct
{
  GModule *module;
  GType dialog;
  GType type;
} OGMRipOptionsPlugin;

typedef OGMRipOptionsPlugin * (* OGMRipOptionsPluginInit) (void);

static GSList *plugins = NULL;

static GSList *
ogmrip_options_plugin_load (const gchar *dirname)
{
  GSList *slist = NULL;

  GModule *module;
  GPatternSpec *pspec;
  GDir *dir;

  OGMRipOptionsPlugin *plugin;
  OGMRipOptionsPluginInit init;
  gpointer ptr;

  const gchar *filename;
  gchar *fullname;
  gint len;

  len = strlen (dirname);

  pspec = g_pattern_spec_new ("*.so");

  dir = g_dir_open (dirname, 0, NULL);
  if (dir)
  {
    while ((filename = g_dir_read_name (dir)))
    {
      init = NULL;

      if (!g_pattern_match_string (pspec, filename))
        continue;

      fullname = g_build_filename (dirname, filename, NULL);
      module = g_module_open (fullname, G_MODULE_BIND_LAZY);
      g_free (fullname);

      if (!module)
      {
        g_warning ("Cannot open module %s", filename);
        continue;
      }

      if (!g_module_symbol (module, "ogmrip_init_options_plugin", &ptr))
      {
        g_warning ("Cannot find initialization function in module %s", filename);
        g_module_close (module);
        continue;
      }

      init = (OGMRipOptionsPluginInit) ptr;

      if (!init)
      {
        g_warning ("Invalid initialization function for module %s", filename);
        g_module_close (module);
        continue;
      }

      plugin = (* init) ();
      if (!plugin)
      {
        g_warning ("Failed to initialize module %s", filename);
        g_module_close (module);
        continue;
      }

      plugin->module = module;
      slist = g_slist_append (slist, plugin);
    }
    g_dir_close (dir);
  }

  g_pattern_spec_free (pspec);

  return slist;
}

/**
 * ogmrip_options_plugin_init:
 *
 * Initializes the plugin system.
 */
void
ogmrip_options_plugin_init (void)
{
  if (!plugins)
    plugins = ogmrip_options_plugin_load (OGMRIP_LIB_DIR "/ogmrip/options-plugins");
}

static void
ogmrip_options_plugin_close_module (OGMRipOptionsPlugin *plugin)
{
  g_module_close (plugin->module);
}

/**
 * ogmrip_options_plugin_uninit:
 *
 * Uninitializes the plugin system.
 */
void
ogmrip_options_plugin_uninit (void)
{
  g_slist_foreach (plugins, (GFunc) ogmrip_options_plugin_close_module, NULL);
}

static OGMRipOptionsPlugin *
ogmrip_options_plugin_find_by_type (GType type)
{
  OGMRipOptionsPlugin *plugin;
  GSList *link;

  for (link = plugins; link; link = link->next)
  {
    plugin = link->data;

    if (plugin && g_type_is_a (type, plugin->type))
        return plugin;

    if (plugin && plugin->type == type)
      return plugin;
  }

  return NULL;
}

/**
 * ogmrip_options_plugin_exists:
 * @type: The type of a codec or a container
 *
 * Checks wether a plugin exists for the codec or container.
 *
 * Returns: TRUE or FALSE
 */
gboolean
ogmrip_options_plugin_exists (GType type)
{
  g_return_val_if_fail (g_type_is_a (type, OGMRIP_TYPE_CONTAINER) ||
      g_type_is_a (type, OGMRIP_TYPE_CODEC), FALSE);

  if (ogmrip_options_plugin_find_by_type (type))
    return TRUE;

  return FALSE;
}

static GtkWidget *
ogmrip_options_plugin_dialog_new (GType type, const gchar *profile)
{
  OGMRipOptionsPlugin *plugin;

  plugin = ogmrip_options_plugin_find_by_type (type);
  if (!plugin)
    return NULL; 

  return g_object_new (plugin->dialog, "key", profile, NULL);
}

/**
 * ogmrip_container_options_plugin_dialog_new:
 * @type: The type of a container
 * @profile: The current profile
 *
 * Creates a new #GtkDialog to configure the container.
 *
 * Returns: a new #GtkDialog
 */
GtkWidget *
ogmrip_container_options_plugin_dialog_new (GType type, const gchar *profile)
{
  g_return_val_if_fail (g_type_is_a (type, OGMRIP_TYPE_CONTAINER), NULL);

  return ogmrip_options_plugin_dialog_new (type, profile);
}

/**
 * ogmrip_container_options_plugin_set_options:
 * @container: The #OGMRipContainer to set the options to
 *
 * Propagates the options to @container.
 */
void
ogmrip_container_options_plugin_set_options (OGMRipContainer *container)
{
  OGMRipContainerOptionsPlugin *plugin;
  GType type;

  g_return_if_fail (OGMRIP_IS_CONTAINER (container));

  type = G_TYPE_FROM_INSTANCE (container);
  plugin = (OGMRipContainerOptionsPlugin* ) ogmrip_options_plugin_find_by_type (type);
  if (plugin && plugin->set_options)
    (* plugin->set_options) (container);
}

/**
 * ogmrip_video_options_plugin_dialog_new:
 * @type: The type of a video codec
 * @profile: The current profile
 *
 * Creates a new #GtkDialog to configure the codec.
 *
 * Returns: a new #GtkDialog
 */
GtkWidget *
ogmrip_video_options_plugin_dialog_new (GType type, const gchar *profile)
{
  g_return_val_if_fail (g_type_is_a (type, OGMRIP_TYPE_VIDEO_CODEC), NULL);

  return ogmrip_options_plugin_dialog_new (type, profile);
}

/**
 * ogmrip_video_options_plugin_set_options:
 * @codec: The #OGMRipVideoCodec to set the options to
 *
 * Propagates the options to @codec.
 */
void
ogmrip_video_options_plugin_set_options (OGMRipVideoCodec *codec)
{
  OGMRipVideoOptionsPlugin *plugin;
  GType type;

  g_return_if_fail (OGMRIP_IS_VIDEO_CODEC (codec));

  type = G_TYPE_FROM_INSTANCE (codec);
  plugin = (OGMRipVideoOptionsPlugin* ) ogmrip_options_plugin_find_by_type (type);
  if (plugin && plugin->set_options)
    (* plugin->set_options) (codec);
}

/**
 * ogmrip_audio_options_plugin_dialog_new:
 * @type: The type of a audio codec
 * @profile: The current profile
 *
 * Creates a new #GtkDialog to configure the codec.
 *
 * Returns: a new #GtkDialog
 */
GtkWidget *
ogmrip_audio_options_plugin_dialog_new (GType type, const gchar *profile)
{
  g_return_val_if_fail (g_type_is_a (type, OGMRIP_TYPE_AUDIO_CODEC), NULL);

  return ogmrip_options_plugin_dialog_new (type, profile);
}

/**
 * ogmrip_audio_options_plugin_set_options:
 * @codec: The #OGMRipAudioCodec to set the options to
 *
 * Propagates the options to @codec.
 */
void
ogmrip_audio_options_plugin_set_options (OGMRipAudioCodec *codec)
{
  OGMRipAudioOptionsPlugin *plugin;
  GType type;

  g_return_if_fail (OGMRIP_IS_AUDIO_CODEC (codec));

  type = G_TYPE_FROM_INSTANCE (codec);
  plugin = (OGMRipAudioOptionsPlugin* ) ogmrip_options_plugin_find_by_type (type);
  if (plugin && plugin->set_options)
    (* plugin->set_options) (codec);
}

/**
 * ogmrip_subp_options_plugin_dialog_new:
 * @type: The type of a subp codec
 * @profile: The current profile
 *
 * Creates a new #GtkDialog to configure the codec.
 *
 * Returns: a new #GtkDialog
 */
GtkWidget *
ogmrip_subp_options_plugin_dialog_new (GType type, const gchar *profile)
{
  g_return_val_if_fail (g_type_is_a (type, OGMRIP_TYPE_SUBP_CODEC), NULL);

  return ogmrip_options_plugin_dialog_new (type, profile);
}

/**
 * ogmrip_subp_options_plugin_set_options:
 * @codec: The #OGMRipSubpCodec to set the options to
 *
 * Propagates the options to @codec.
 */
void
ogmrip_subp_options_plugin_set_options (OGMRipSubpCodec *codec)
{
  OGMRipSubpOptionsPlugin *plugin;
  GType type;

  g_return_if_fail (OGMRIP_IS_SUBP_CODEC (codec));

  type = G_TYPE_FROM_INSTANCE (codec);
  plugin = (OGMRipSubpOptionsPlugin* ) ogmrip_options_plugin_find_by_type (type);
  if (plugin && plugin->set_options)
    (* plugin->set_options) (codec);
}

