//******************************************************************************
//
// File:    Files.java
// Package: edu.rit.io
// Unit:    Class edu.rit.io.Files
//
// This Java source file is copyright (C) 2007 by Alan Kaminsky. All rights
// reserved. For further information, contact the author, Alan Kaminsky, at
// ark@cs.rit.edu.
//
// This Java source file is part of the Parallel Java Library ("PJ"). PJ is free
// software; you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation; either
// version 3 of the License, or (at your option) any later version.
//
// PJ is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
// A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
// A copy of the GNU General Public License is provided in the file gpl.txt. You
// may also obtain a copy of the GNU General Public License on the World Wide
// Web at http://www.gnu.org/licenses/gpl.html.
//
//******************************************************************************

package edu.rit.io;

import java.io.File;

/**
 * Class Files provides static methods for various file related operations.
 *
 * @author  Alan Kaminsky
 * @version 26-Nov-2007
 */
public class Files
	{

// Prevent construction.

	private Files()
		{
		}

// Exported operations.

	/**
	 * Append the given rank to the given file. The rank goes before the file
	 * extension if any. For example,
	 * <TT>Files.fileForRank(new&nbsp;File("image.pjg"),2)</TT> returns a File
	 * whose name is <TT>"image_2.pjg"</TT>;
	 * <TT>Files.fileForRank(new&nbsp;File("image"),2)</TT> returns a File whose
	 * name is <TT>"image_2"</TT>.
	 *
	 * @param  file  File.
	 * @param  rank  Rank.
	 *
	 * @return  File with rank appended.
	 */
	public static File fileForRank
		(File file,
		 int rank)
		{
		return new File (fileNameForRank (file.getPath(), rank));
		}

	/**
	 * Append the given rank to the given file name. The rank goes before the
	 * file extension if any. For example,
	 * <TT>Files.fileNameForRank("image.pjg",2)</TT> returns
	 * <TT>"image_2.pjg"</TT>; <TT>Files.fileNameForRank("image",2)</TT> returns
	 * <TT>"image_2"</TT>.
	 *
	 * @param  filename  File name.
	 * @param  rank      Rank.
	 *
	 * @return  File name with rank appended.
	 */
	public static String fileNameForRank
		(String filename,
		 int rank)
		{
		int i = filename.lastIndexOf ('.');
		return
			i == -1 ?
				filename + "_" + rank :
				filename.substring(0,i) + "_" + rank + filename.substring(i);
		}

// Unit test main program.

//	/**
//	 * Unit test main program.
//	 * <P>
//	 * Usage: java edu.rit.io.Files <I>filename</I> <I>rank</I>
//	 */
//	public static void main
//		(String[] args)
//		throws Exception
//		{
//		String filename = args[0];
//		int rank = Integer.parseInt (args[1]);
//		File file = new File (filename);
//		System.out.println
//			("Files.fileForRank (file, rank) = " +
//			  Files.fileForRank (file, rank));
//		System.out.println
//			("Files.fileNameForRank (filename, rank) = " +
//			  Files.fileNameForRank (filename, rank));
//		}

	}
