/****************************************************************************
** ImagePreview class
**
** Created: Tue Feb 02 22:06:51 2004
**      by: Varol Okan using Kate
**
** This class is the encapsulation of the ImagePreview from the FileSelection
** Dialog.
** It is also used in the ImageDialog.
**
****************************************************************************/

#include <qobject.h>
#include <qlistview.h>
#include <qpopupmenu.h>

#include "global.h"
#include "xml_dvd.h"
#include "menupreview.h"
#include "dialogbutton.h"
#include "menuobject.h"
#include "buttonobject.h"
#include "qdvdauthor.h"
#include "sourcefileentry.h"

ButtonObject::ButtonObject (QWidget *pParent) //, const char *pName, WFlags flags)
	: MenuObject (pParent)
{
	m_qsObjectType = QString (BUTTON_OBJECT);
	m_iButtonState = STATE_NORMAL;
	m_pSourceFileEntry = NULL;
}

ButtonObject::~ButtonObject ()
{
	// Please note that the m_listNormalState etc objects are not destroyed here
	// As this is merely a holding place for the Objects.
	// Creation and destruction is done in the Parent (MenuPreview / ButtonPreview etc.)
}

MenuObject *ButtonObject::clone()
{
	uint t;
	ButtonObject *pNewObject = new ButtonObject (MenuObject::parent());
	for (t=0;t<m_listNormalState.count();t++)
		pNewObject->appendNormal (m_listNormalState[t]->clone());
	for (t=0;t<m_listSelectedState.count();t++)
		pNewObject->appendSelected (m_listSelectedState[t]->clone());
	for (t=0;t<m_listHighlightedState.count();t++)
		pNewObject->appendHighlighted (m_listHighlightedState[t]->clone());

	pNewObject->setSourceFileEntry(sourceFileEntry());
	pNewObject->setAction(m_qsAction);
	pNewObject->setName  (m_qsName);
	pNewObject->setNext (NEXT_BUTTON_UP, m_qsUp);
	pNewObject->setNext (NEXT_BUTTON_DOWN, m_qsDown);
	pNewObject->setNext (NEXT_BUTTON_RIGHT, m_qsRight);
	pNewObject->setNext (NEXT_BUTTON_LEFT, m_qsLeft);
	
	return pNewObject;
}

ButtonObject &ButtonObject::operator = (ButtonObject &theOther)
{
	uint t;
	clear ();
	setSourceFileEntry(theOther.sourceFileEntry());
	m_qsAction = theOther.action();
	m_qsName   = theOther.name();
	m_qsUp     = theOther.next(NEXT_BUTTON_UP);
	m_qsDown   = theOther.next(NEXT_BUTTON_DOWN);
	m_qsRight  = theOther.next(NEXT_BUTTON_RIGHT);
	m_qsLeft   = theOther.next(NEXT_BUTTON_LEFT);
	// The following lists hold the information for the different button states.
	for (t=0;t<theOther.getNormalCount();t++)
		appendNormal(theOther.getNormal(t));
	for (t=0;t<theOther.getSelectedCount();t++)
		appendSelected(theOther.getSelected(t));
	for (t=0;t<theOther.getHighlightedCount();t++)
		appendHighlighted(theOther.getHighlighted(t));
	return *this;
}

void ButtonObject::clear ()
{
	uint t;

	m_iButtonState = STATE_NORMAL;
	m_qsAction = QString ("");
	m_qsUp     = QString ("");
	m_qsDown   = QString ("");
	m_qsRight  = QString ("");
	m_qsLeft   = QString ("");
	m_pSourceFileEntry = NULL;

	for (t=0;t<m_listNormalState.count();t++)
		delete m_listNormalState[t];
	for (t=0;t<m_listSelectedState.count();t++)
		delete m_listSelectedState[t];
	for (t=0;t<m_listHighlightedState.count();t++)
		delete m_listHighlightedState[t];
	m_listNormalState.clear();
	m_listSelectedState.clear();
	m_listHighlightedState.clear();
}

void ButtonObject::drawContents (QPainter *pPainter)
{
	drawContents(pPainter, BACKGROUND_IMG);
//	m_listNormalState[0]->drawContents(pPainter);
//	m_boundingRect = m_listNormalState[0]->boundingRect();
//	m_rect = m_listNormalState[0]->rect();
}

void ButtonObject::drawContents (QPainter *pPainter, uint iWhichState)
{
	switch (iWhichState)	{
	case BACKGROUND_IMG:
		m_listNormalState[0]->drawContents(pPainter);
		m_boundingRect = m_listNormalState[0]->boundingRect();
		m_rect = m_listNormalState[0]->rect();
	break;
	case HIGHLIGHTED_MASK:
		m_listHighlightedState[0]->drawContents(pPainter);
	break;
	case SELECTED_MASK:
		m_listSelectedState[0]->drawContents(pPainter);
	break;
	}
}

bool ButtonObject::mouseMoveEvent (QMouseEvent *pEvent)
{
	static QPoint m_currentMousePos = pEvent->pos();
	// This will move the textObject around and around and around we go ...
	QRect theRect = rect();
	int iX, iY, iWidth, iHeight;
	iWidth  = rect().width ();
	iHeight = rect().height();
	iX = rect().x() - (m_currentMousePos.x() - pEvent->pos().x());
	iY = rect().y() - (m_currentMousePos.y() - pEvent->pos().y());
	theRect.setX(iX);
	theRect.setY(iY);
	theRect.setWidth(iWidth);
	theRect.setHeight(iHeight);

	setRect(theRect);
	m_currentMousePos = pEvent->pos();
	return false;
}

bool ButtonObject::mousePressEvent (QMouseEvent *pEvent)
{
	if (pEvent->button() == Qt::RightButton)	{
		QPopupMenu *pMenu = new QPopupMenu(m_pParent);
		pMenu->insertItem ( tr ("Edit ..."), (QObject *)this, SLOT(slotEdit()));
		pMenu->insertItem ( tr ("Unbutton"), (QObject *)this, SLOT(slotUnbutton()));
		pMenu->insertItem ( tr ("Delete")  , (QObject *)this, SLOT(slotDelete()));
		pMenu->insertSeparator();
		pMenu->insertItem ( tr ("Add Frame"), (QObject *)this, SLOT(slotAddFrame()));
		pMenu->insertItem ( tr ("Add Image"), (QObject *)this, SLOT(slotAddImage()));
		pMenu->insertItem ( tr ("Add Movie"), (QObject *)this, SLOT(slotAddMovie()));
		int iReturn = pMenu->exec(pEvent->globalPos());
		// Here we mark that the user called a menu item thus we don't want the
		// base classes to continue with the mouse event
		if (iReturn != -1)
			return true;
//printf ("Clicked inside TextButton\n");
	}
	m_bDrawRect = true;
	return false;
}

bool ButtonObject::mouseReleaseEvent (QMouseEvent *)
{
	return false;
}

bool ButtonObject::mouseDoubleClickEvent(QMouseEvent *)
{
	// Call MenuPreview::slotCreateButtonDialog ()
	emit (signalCreateButtonDialog (this));
	return false;
}

void ButtonObject::appendNormal (MenuObject *pObject)
{
	m_listNormalState.append (pObject);
}

void ButtonObject::appendSelected (MenuObject *pObject)
{
	m_listSelectedState.append (pObject);
}

void ButtonObject::appendHighlighted (MenuObject *pObject)
{
	m_listHighlightedState.append (pObject);
}

uint ButtonObject::getNormalCount ()
{
	return m_listNormalState.count();
}

MenuObject *ButtonObject::getNormal (int iNr)
{
	if (iNr > (int)m_listNormalState.count())
		return NULL;
	return m_listNormalState[iNr];
}

uint ButtonObject::getSelectedCount ()
{
	return m_listSelectedState.count();
}

MenuObject *ButtonObject::getSelected (int iNr)
{
	if (iNr > (int)m_listSelectedState.count())
		return NULL;
	return m_listSelectedState[iNr];
}

uint ButtonObject::getHighlightedCount ()
{
	return m_listHighlightedState.count();
}

MenuObject *ButtonObject::getHighlighted (int iNr)
{
	if (iNr > (int)m_listHighlightedState.count())
		return NULL;
	return m_listHighlightedState[iNr];
}

void ButtonObject::deleteMasks()
{
	// This function handles only the case of the cancelation of the Button creation
	// From the DialogButton - class.
	// I.e. the user just created the Button and decides to cancel out.
	// In case the User cancels the ButtonDialog when the ButtonObject already 
	// existed before, nothing ought to be done here.
	uint t;
	// we can leave the m_listNormalState - objects as they reflect the original Objects.
	// However the m_listSelected / Highlighted should be deleted ...
	for (t=0;t<m_listSelectedState.count();t++)
		delete m_listSelectedState[t];
	for (t=0;t<m_listHighlightedState.count();t++)
		delete m_listHighlightedState[t];
	m_listSelectedState.clear();
	m_listHighlightedState.clear();
}

bool ButtonObject::readProjectFile  (QDomNode &theNode)
{
	// Okay, here we retain the stored data from the xml file.
	QDomElement theElement = theNode.toElement();
	QDomAttr attribute;

	attribute = theElement.attributeNode ( BUTTON_OBJECT_NAME );
	if (!attribute.isNull())
		m_qsName = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_ACTION );
	if (!attribute.isNull())
		m_qsAction = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_UP );
	if (!attribute.isNull())
		m_qsUp = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_DOWN );
	if (!attribute.isNull())
		m_qsDown = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_LEFT );
	if (!attribute.isNull())
		m_qsLeft = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_RIGHT );
	if (!attribute.isNull())
		m_qsRight = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_SOURCE_ENTRY);
	if (!attribute.isNull())
		m_qsSourceDisplayName = attribute.value();
	// Please note that the nortmal / selected / highlighted objects are read
	// in by ButtonObject::readObjects(...);
//printf ("ButtonObject::readProjectFile <%s> \n", (const char *)m_qsSourceDisplayName);
	return true;
}

bool ButtonObject::readObjects (QDomNode &theNode, MenuPreview *pPreview)
{
	QDomElement buttonElement;
	QDomNode buttonNode = theNode.firstChild();
	QDomNode childNode;
	MenuObject *pNewObject = NULL;
	while (!buttonNode.isNull())	{
		buttonElement   = buttonNode.toElement();
		QString tagName = buttonElement.tagName();
		childNode = buttonNode.firstChild();
		// Next we should create the new Object but if we see the
		// MenoObject tag it means it is the MenuObject vars of the button
		// itself, thus we don't want to create anything. (It would put out a warning).
		if (tagName != MENU_OBJECT)
			pNewObject = pPreview->readObject ( childNode );
		if (tagName == BUTTON_OBJECT_NORMAL)
			m_listNormalState.append ( pNewObject );

		else if (tagName == BUTTON_OBJECT_SELECTED)
			m_listSelectedState.append( pNewObject );

		else if (tagName == BUTTON_OBJECT_HIGHLIGHTED)
			m_listHighlightedState.append ( pNewObject );

		else if (tagName == MENU_OBJECT)
			MenuObject::readProjectFile( theNode );

		else
			printf ("Warning : ButtonObject::readObjects -=> wrong XML Node <%s>\nContinuing ...\n", (const char *)tagName);
		buttonNode = buttonNode.nextSibling ();
	}
	return true;
}

bool ButtonObject::writeProjectFile (QDomElement &theElement)
{
	uint t;
	QDomDocument xmlDoc = theElement.ownerDocument();
	QDomElement buttonNode = xmlDoc.createElement( BUTTON_OBJECT );	// <FrameObject>
	// Here we set the attributes of the <dvdauthor> tag
	if (!m_qsName.isNull())
		buttonNode.setAttribute( BUTTON_OBJECT_NAME,   m_qsName );
	if (!m_qsAction.isNull())
		buttonNode.setAttribute( BUTTON_OBJECT_ACTION, m_qsAction );
	if (!m_qsUp.isNull())
		buttonNode.setAttribute( BUTTON_OBJECT_UP,     m_qsUp );
	if (!m_qsDown.isNull())
		buttonNode.setAttribute( BUTTON_OBJECT_DOWN,   m_qsDown );
	if (!m_qsLeft.isNull())
		buttonNode.setAttribute( BUTTON_OBJECT_LEFT,   m_qsLeft );
	if (!m_qsRight.isNull())
		buttonNode.setAttribute( BUTTON_OBJECT_RIGHT,  m_qsRight );
	if (m_pSourceFileEntry)
		buttonNode.setAttribute( BUTTON_OBJECT_SOURCE_ENTRY,  m_pSourceFileEntry->qsDisplayName );
//printf ("ButtonObject::writeProjectFile <%s> - <%s> \n", (const char *)m_pSourceFileEntry->qsDisplayName, (const char *)m_qsAction );

	// And here we store the ButtonObjects children ...
	QDomElement normalNode = xmlDoc.createElement( BUTTON_OBJECT_NORMAL );
	for (t=0;t<m_listNormalState.count();t++)
		m_listNormalState[t]->writeProjectFile ( normalNode );

	QDomElement selectedNode = xmlDoc.createElement( BUTTON_OBJECT_SELECTED );
	for (t=0;t<m_listSelectedState.count();t++)
		m_listSelectedState[t]->writeProjectFile ( selectedNode );

	QDomElement highlightedNode = xmlDoc.createElement( BUTTON_OBJECT_HIGHLIGHTED );
	for (t=0;t<m_listHighlightedState.count();t++)
		m_listHighlightedState[t]->writeProjectFile ( highlightedNode );

	buttonNode.appendChild( normalNode );
	buttonNode.appendChild( selectedNode );
	buttonNode.appendChild( highlightedNode );
	
	theElement.appendChild( buttonNode );
	// And here we write the base class ...
	return MenuObject::writeProjectFile( buttonNode );
}

void ButtonObject::createStructure (QListViewItem *pParentItem)
{
	// Creating teh ListView structure of this Button.
	QListViewItem *pButtonItem;
	QListViewItem *pStateItem;

	QString qsAction;
	QStringList actionList;
	if (!m_qsAction.isEmpty())	{	// For new buttons this might be empty ...
		actionList = QStringList::split (QString (STRING_SEPARATOR), m_qsAction);
		qsAction = actionList[0] + QString (" ") +actionList[1] + QString (" ") +actionList[2];
		if (actionList.count() == 4)
			qsAction += QString (" ") +actionList[3];
	}
	pButtonItem = new QListViewItem(pParentItem, m_qsName, qsAction);
	pButtonItem->setExpandable (TRUE);

	uint t;
	for (t=0;t<m_listHighlightedState.count();t++)	{
		pStateItem = new QListViewItem(pButtonItem, QString ("Highlighted"), QString (""));
		pStateItem->setExpandable (TRUE);
		m_listHighlightedState[t]->createStructure (pStateItem);
	}
	for (t=0;t<m_listSelectedState.count();t++)	{
		pStateItem = new QListViewItem(pButtonItem, QString ("Selected"), QString (""));
		pStateItem->setExpandable (TRUE);
		m_listSelectedState[t]->createStructure (pStateItem);
	}
	for (t=0;t<m_listNormalState.count();t++)	{
		m_listNormalState[t]->createStructure (pButtonItem);
	}
}

void ButtonObject::replaceColor (QColor theColor, QColor oldColor)
{
	// We handle the actual color as a unique index. Thus we allow 
	// the user to utilise any of the 4 available colors for any purpose.
	uint t;
	
	// Does this make sense ??? I don't know yert ...
//	for (t=0;t<m_listNormalState.count();t++)	{
//		m_listNormalState[t]->replaceColor (theColor);
//	}
	for (t=0;t<m_listSelectedState.count();t++)	{
		m_listSelectedState[t]->replaceColor (theColor, oldColor);
	}
	for (t=0;t<m_listHighlightedState.count();t++)	{
		m_listHighlightedState[t]->replaceColor (theColor, oldColor);
	}
}

void ButtonObject::setAction(QString &qsAction)
{
	m_qsAction = qsAction;
}

QString &ButtonObject::action()
{
	return m_qsAction;
}

void ButtonObject::setSourceFileEntry(SourceFileEntry *pEntry)
{
	m_pSourceFileEntry = pEntry;
	if (pEntry)
		m_qsSourceDisplayName = pEntry->qsDisplayName;
	else
		m_qsSourceDisplayName = QString ();
}

QString &ButtonObject::sourceDisplayName ()
{
	return m_qsSourceDisplayName;
}

SourceFileEntry *ButtonObject::sourceFileEntry()
{
	return m_pSourceFileEntry;
}

void ButtonObject::setNext(uint iNext, QString &qsNext)
{
	switch (iNext)	{
	case NEXT_BUTTON_UP:
		m_qsUp = qsNext;
	break;
	case NEXT_BUTTON_DOWN:
		m_qsDown = qsNext;
	break;
	case NEXT_BUTTON_RIGHT:
		m_qsRight = qsNext;
	break;
	case NEXT_BUTTON_LEFT:
		m_qsLeft = qsNext;
	break;
	};
}

QString &ButtonObject::next(uint iNext)
{
	switch (iNext)	{
	case NEXT_BUTTON_UP:
		return m_qsUp;
	break;
	case NEXT_BUTTON_DOWN:
		return m_qsDown;
	break;
	case NEXT_BUTTON_RIGHT:
		return m_qsRight;
	break;
	case NEXT_BUTTON_LEFT:
		return m_qsLeft;
	break;
	};
	return m_qsUp;
}

void ButtonObject::slotEdit()
{
	mouseDoubleClickEvent(NULL);
}

void ButtonObject::slotUnbutton()
{
	// Here we are going to split the Button object into its three distinkt 
	// objects and add them to the Preview (MenuPreview that is).
	emit (signalUnbuttonMe(this));
	// The work is done in MenuPreview::slotUnbutton()
}

void ButtonObject::slotDelete()
{
	emit (signalDeleteMe(this));
}

void ButtonObject::slotAddFrame()
{

}

void ButtonObject::slotAddImage()
{

}

void ButtonObject::slotAddMovie() // movie object for the Menu that is ...
{

}


