/****************************************************************************
** Implementation of class XineInfo
**
** Created: Thu Jun 10 07:53:05 2004
**      by: Varol Okan using the kate editor
**
** This class collects all possible information about
** the current set stream.
**
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <qfileinfo.h>

#include "../../bugs_workaround.h"

#include "xineinfo.h"


/****************************************************************************
** CreateXineEngine class
**
** This class will create a xine_t object as a background task.
**
** Usually to create and initialize a xint_t - object takes a few 
** seconds. This is especially disturbing if you start a Dialog, or an
** application and it takes too long to start up.
**
** With this class the object will be created in the background and 
** initialized before returning it to its creator object.
**
*****************************************************************************/
#include <qapplication.h>
#include <qmessagebox.h>
#include <qdir.h>
CreateXineEngine::CreateXineEngine (xine_t **ppXinePointer, QString qsConfigFile)
{
	m_ppXineEngine = ppXinePointer;
	m_qsConfigFile = qsConfigFile;
	start ();
//	run ();
}

CreateXineEngine::~CreateXineEngine ()
{

}

void CreateXineEngine::run ()
{
	bool bSave = false;
	xine_t *pXineEngine = xine_new();

	if (!pXineEngine)	{
		QApplication::beep();
		QMessageBox::warning (NULL, QObject::tr ("Error creating Xine object."),
			QObject::tr ("I could not create the Xine object.\n"), 
			QMessageBox::Ok ,  QMessageBox::NoButton);
		return;
	}

	if (QFile::exists(m_qsConfigFile))
		xine_config_load (pXineEngine, (const char *)m_qsConfigFile);
	else
		bSave = true;

	xine_init(pXineEngine);

	if (bSave)	{
		printf ("No config file found, will create \n%s\n", (const char *) m_qsConfigFile);
		// First we check if the directory exists or if we should create it first ...
		QString qsDirPath = QDir::homeDirPath() + QString ("/.qdvdauthor");
		QDir theDir(qsDirPath);
		if (!theDir.exists())
			theDir.mkdir (qsDirPath);
		xine_config_save  (pXineEngine, (char *)((const char *)m_qsConfigFile));
	}
	// Here we set the original pointer to the new xine_t - object 
	*m_ppXineEngine = pXineEngine;
#ifdef XINE_THREADING_CRASH
	// FIXME:
	// remove this sleep() when I can figure out why QDVDAuthor is crashing when leaving this thread.
	// Note: it is only crashing on my dual PIII system not on any of the singe CPU machines.
	uint iForOneYear = 30758400;
	sleep (iForOneYear);
#endif
//printf ("Init xine over ...\n");
	// and here we commit suicide ...
//	delete this;
}




XineInfo::XineInfo ()
{
	initMe ();
	m_pXine = xine_new();
	xine_init(m_pXine);
	createXineStream ();
	m_bOwnXineEngine = true;
}

XineInfo::~XineInfo ()
{
	// Cleaning of the xine object.
	if (m_bOwnXineStream)
		xine_dispose(m_pXineStream);
	if (!m_pXine)
		return;
	xine_close_audio_driver(m_pXine, m_pAudioDriver);
	xine_close_video_driver(m_pXine, m_pVideoDriver);
	if (m_bOwnXineEngine)
		xine_exit(m_pXine);
}

XineInfo::XineInfo (xine_stream_t *pXineStream)
{
	initMe ();
	setXineStream (pXineStream);
}

XineInfo::XineInfo (xine_t *pXine)
{
	initMe ();
	m_pXine = pXine;
	createXineStream ();
}

void XineInfo::createXineStream ()
{
//printf ("CreateXineEngine::createXineStream (1) \n");
	if (!m_pXine)
		return;
	// opening xine output ports
	m_pVideoDriver = xine_open_video_driver(m_pXine,  "none", XINE_VISUAL_TYPE_NONE, NULL);
	m_pAudioDriver = xine_open_audio_driver(m_pXine , "none", NULL);

	// open a xine stream connected to these ports
	m_pXineStream = xine_stream_new(m_pXine, m_pAudioDriver, m_pVideoDriver);

	m_bOwnXineStream = true;
}

void XineInfo::initMe ()
{
	// First we initialize the variables ...
	m_pXineStream       = NULL;
	m_bHaveValues       = false;
	m_bHasAudio         = false;
	m_bHasVideo         = false;
	m_bAudioHandled     = false;
	m_bVideoHandled     = false;
	m_iSize             = 0;
	m_iResolutionWidth  = 0;
	m_iResolutionHeight = 0;
	m_fFPS              = 0.0f;
	m_iLength           = 0;
	m_iBits             = 0;
	m_iSample           = 0;
	m_iAudioBPS         = 0;
	m_iVideoBPS         = 0;
	m_pAudioDriver      = NULL;
	m_pVideoDriver      = NULL;
	m_bOwnXineEngine    = false;
	m_bOwnXineStream    = false;
	m_qsStatus = QObject::tr ("Not Ok");
}

void XineInfo::setXineStream (xine_stream_t *pXineStream)
{
//printf ("XineInfo::setXineStream\n");
	initMe ();
	m_pXineStream = pXineStream;
	queryValues ();
}

void XineInfo::setStream (QString qsStream)
{
//printf ("XineInfo::setStream <%s>\n", (const char *)qsStream);
	// Here we create a stream on the xine - object and dispose of it later on ...
	if ( (m_pXine == NULL) || (m_pAudioDriver == NULL) || (m_pVideoDriver == NULL) )
		return;
	// Next we check if the file exists ...
	QFileInfo fileInfo (qsStream);
	if (!fileInfo.exists())
		return;

	// Set the file size ...
	m_iSize = fileInfo.size();
	if (m_iSize < 1)
		m_qsSize = QString ("? B");
	if (m_iSize < 1024)
		m_qsSize = QString ("%1 B").arg(m_iSize);
	else if (m_iSize < 1024*1024)
		m_qsSize = QString ("%1kB").arg((float)m_iSize/1024);
	else if (m_iSize < 1024*1024*1024)
		m_qsSize = QString ("%1MB").arg((float)m_iSize/(1024*1024));
	else
		m_qsSize = QString ("%1GB").arg((float)m_iSize/(1024*1024*1024));

	// set the file name ...
	m_qsFileName = qsStream;

	if (!xine_open(m_pXineStream, (const char *)qsStream) || !xine_play(m_pXineStream, 0, 0)) {
		printf("Unable to open mrl\n");
		return;
	}

	queryValues ();
	xine_close(m_pXineStream);
}

void XineInfo::queryValues ()
{
//printf ("XineInfo::queryValues\n");
	int iReturn;
	// this is the main function, to query all values
	if (!m_pXineStream)
		return;
	m_bHasAudio = (bool)xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_HAS_AUDIO);
	m_bHasVideo = (bool)xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_HAS_VIDEO);

	m_bAudioHandled = (bool)xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_AUDIO_HANDLED);
	m_bVideoHandled = (bool)xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_VIDEO_HANDLED);
	
	// Here we check if the status is Ok (I.e. if audio avail and we can play the audio, then it is ok (same with video)
	if ( ( (m_bHasAudio) && (m_bAudioHandled) ) || ( (m_bHasVideo) && (m_bVideoHandled) ) )
		m_qsStatus = QObject::tr ("Ok");
	else
		m_qsStatus = QObject::tr ("Not Ok");

	if (m_bHasVideo)	{
		m_iResolutionWidth  = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_VIDEO_WIDTH);
		m_iResolutionHeight = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_VIDEO_HEIGHT);
		m_qsResolution = QString ("%1x%2").arg(m_iResolutionWidth).arg(m_iResolutionHeight);
		iReturn = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_FRAME_DURATION);
		m_fFPS = 90000.0 / (float)iReturn;
		m_qsFPS = QString ("%1").arg(m_fFPS);		// E.g. 25

		m_qsVideoFormat = getVideoFormat (m_iResolutionWidth, m_iResolutionHeight);

/*		iReturn = xine_get_param (m_pXineStream, XINE_PARAM_VO_ASPECT_RATIO);
		if (iReturn == XINE_VO_ASPECT_AUTO)
			m_qsRatio = QString ("auto");
		else if (iReturn == XINE_VO_ASPECT_SQUARE)
			m_qsRatio = QString ("1:1");
		else if (iReturn == XINE_VO_ASPECT_4_3)
			m_qsRatio = QString ("4:3");
		else if (iReturn == XINE_VO_ASPECT_ANAMORPHIC)
			m_qsRatio = QString ("16:9");
		else if (iReturn ==  XINE_VO_ASPECT_DVB)
			m_qsRatio = QString ("2.11:1");
		else
			m_qsRatio = QString ("<%1>").arg(iReturn);
*/
		iReturn = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_VIDEO_RATIO);
		if (iReturn == 1)
			m_qsRatio = QString ("1:1");
		else if (iReturn == (int)(16.0/9.0*10000))
			m_qsRatio = QString ("16:9");
		else if (iReturn == (int)(4.0/3.0*10000))
			m_qsRatio = QString ("4:3");
		else if (iReturn == (int)(2.11*10000))
			m_qsRatio = QString ("2.11:1");
		else if (iReturn == (int)(21.0/9.0*10000))
			m_qsRatio = QString ("21:9");
		else if (iReturn == (int)(29.0/9.0*10000))
			m_qsRatio = QString ("29:9");
		else
			m_qsRatio = QString ("<%1>").arg((float)iReturn/10000);

		iReturn = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_VIDEO_FOURCC);
		if (iReturn)
			m_qsVideoCodec = QString(QString ("<%1 = %2>").arg(iReturn).arg(get_fourcc_string (iReturn)));
		else
			m_qsVideoCodec = QString(xine_get_meta_info (m_pXineStream, XINE_META_INFO_VIDEOCODEC));
		
		m_iVideoBPS = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_VIDEO_BITRATE);
		if (m_iVideoBPS < 1)	{ 
			// Here we construct the video bitrate in case it is set to zero by xine.
			m_iVideoBPS = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_BITRATE);
			m_iAudioBPS = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_AUDIO_BITRATE);
			if ((m_iVideoBPS > 0) && (m_iAudioBPS > 0) )
				m_iVideoBPS -= m_iAudioBPS;
		}
		if (m_iVideoBPS < 1)
			m_qsVideoBPS = QString ("? bps");
		if (m_iVideoBPS < 1000)
			m_qsVideoBPS = QString ("%1 bps").arg(m_iVideoBPS);
		else if (m_iVideoBPS < 1000000)
			m_qsVideoBPS = QString ("%1kbps").arg((float)m_iVideoBPS/1000);
		else
			m_qsVideoBPS = QString ("%1Mbps").arg((float)m_iVideoBPS/1000000);
	}
	if (m_bHasAudio)	{
		int iPosStream, iPosTime, iLengthTime;
		iReturn = xine_get_pos_length (m_pXineStream, &iPosStream, &iPosTime, &iLengthTime);
		if (iReturn == 1)	{	// success
			m_iLength = iLengthTime;
			m_qsLength = msToTimeString (iLengthTime);
		}
//		else
//			printf ("XineInfo::queryValues - Could not get streamLength <%x><%d>\n", (void *)m_pXineStream, iLengthTime);

		m_iBits = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_AUDIO_BITS);
		iReturn = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_AUDIO_FOURCC);
		if (iReturn)
			m_qsAudioCodec = QString(QString ("<%1 = %2>").arg(iReturn).arg(get_fourcc_string (iReturn)));
		else
			m_qsAudioCodec = QString(xine_get_meta_info (m_pXineStream, XINE_META_INFO_AUDIOCODEC));

		iReturn = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_AUDIO_CHANNELS);
		if (iReturn == 0)
			m_qsAudioFormat = QString ("");
		else if (iReturn == 1)
			m_qsAudioFormat = QObject::tr ("mono");
		else if (iReturn == 2)
			m_qsAudioFormat = QObject::tr ("stereo");
		else if (iReturn == 4)
			m_qsAudioFormat = QObject::tr ("quad");
		else
			m_qsAudioFormat = QObject::tr ("%1 channels").arg(iReturn);

		m_iSample = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_AUDIO_SAMPLERATE);
		if (m_iSample < 1000)
			m_qsSample = QString ("%1 Hz").arg(m_iSample);
		else if (m_iSample < 1000000)
			m_qsSample = QString ("%1kHz").arg(m_iSample/1000);
		else
			m_qsSample = QString ("%1MHz").arg(m_iSample/1000000);

		m_iAudioBPS = xine_get_stream_info (m_pXineStream, XINE_STREAM_INFO_AUDIO_BITRATE);
		if (m_iAudioBPS < 1000)
			m_qsAudioBPS = QString ("%1 bps").arg(m_iAudioBPS);
		else if (m_iAudioBPS < 1000000)
			m_qsAudioBPS = QString ("%1kbps").arg((float)m_iAudioBPS/1000);
		else
			m_qsAudioBPS = QString ("%1Mbps").arg((float)m_iAudioBPS/1000000);
	}
	/*
uint t;
int iAudioArray[] = {XINE_STREAM_INFO_HAS_AUDIO, XINE_STREAM_INFO_IGNORE_AUDIO, XINE_STREAM_INFO_BITRATE, XINE_STREAM_INFO_AUDIO_CHANNELS, XINE_STREAM_INFO_AUDIO_BITS, XINE_STREAM_INFO_AUDIO_SAMPLERATE, XINE_STREAM_INFO_AUDIO_BITRATE, XINE_STREAM_INFO_AUDIO_FOURCC, XINE_STREAM_INFO_AUDIO_HANDLED, XINE_STREAM_INFO_MAX_AUDIO_CHANNEL, XINE_STREAM_INFO_AUDIO_MODE};

int iVideoArray[] = {XINE_STREAM_INFO_VIDEO_WIDTH, XINE_STREAM_INFO_VIDEO_HEIGHT, XINE_STREAM_INFO_VIDEO_RATIO, XINE_STREAM_INFO_VIDEO_CHANNELS, XINE_STREAM_INFO_VIDEO_STREAMS, XINE_STREAM_INFO_VIDEO_BITRATE, XINE_STREAM_INFO_VIDEO_FOURCC, XINE_STREAM_INFO_VIDEO_HANDLED, XINE_STREAM_INFO_FRAME_DURATION, XINE_STREAM_INFO_HAS_CHAPTERS, XINE_STREAM_INFO_HAS_VIDEO, XINE_STREAM_INFO_IGNORE_VIDEO, XINE_STREAM_INFO_VIDEO_HAS_STILL, XINE_STREAM_INFO_SKIPPED_FRAMES, XINE_STREAM_INFO_DISCARDED_FRAMES};

printf ("Audio - Video properties ...\n");
for (t=0;t<11;t++)
	printf ("<%d = %d> ", iAudioArray[t], xine_get_stream_info (m_pXineStream, iAudioArray[t]));
printf ("\n\n");
for (t=0;t<15;t++)
	printf ("<%d = %d> ", iVideoArray[t], xine_get_stream_info (m_pXineStream, iVideoArray[t]));
printf ("\n\n");
*/
}

// Function borrowed from kaffeine ...
QString XineInfo::msToTimeString(unsigned long int iMSec)
{
  int hours;
  int min;
  int sec;
  int my_msec=iMSec;
  QString tmp;
  QString t;

    // the extra 500 will ensure that we are rounding correctly.
    iMSec = (iMSec+500)/1000;  //sec
    hours = iMSec/3600;
    my_msec -= hours*3600*1000;
    t = t.setNum(hours);
    t.append(":");

    iMSec = iMSec - (hours*3600);
    min = iMSec / 60;
    my_msec -= min*60*1000;
    tmp = tmp.setNum(min);
    tmp = tmp.rightJustify(2, '0');
    t.append(tmp);
    t.append(":");

    sec = iMSec - (min*60);
    my_msec -= sec*1000;
    
    tmp = tmp.setNum(sec);
    tmp = tmp.rightJustify(2, '0');
    t.append(tmp);

   return t;
}

char * XineInfo::get_fourcc_string (int f)
{	// from gtk-xine.c rev 1.2
	static char fcc[5];
	memset(&fcc, 0, sizeof(fcc));
	
	/* should we take care about endinaness ? */
	fcc[0] = f      | 0xFFFFFF00;
	fcc[1] = f>>8   | 0xFFFFFF00;
	fcc[2] = f>>16  | 0xFFFFFF00;
	fcc[3] = f>>24  | 0xFFFFFF00;
	fcc[4] = 0;

	if (f <= 0xFFFF)
		sprintf (fcc, "0x%x", f);
		
	if  ( (fcc[0] == 'm') && (fcc[1] == 's') )
	{
//		if ((fcc[2] = 0x0) && (fcc[3] == 0x55)) Original
		if ((fcc[2] == 0x0) && (fcc[3] == 0x55))
			*(uint32_t *)fcc = 0x33706d2e; /* Force to '.mp3' */
	}
	return (char *)&fcc[0];
}

QString XineInfo::getVideoFormat (int iWidth, int iHeight)
{
	QString qsFormat = ("custom");
	if ( ((iWidth == 720) && (iHeight == 480) ) ||
	     ((iWidth == 704) && (iHeight == 480) ) ||
	     ((iWidth == 352) && (iHeight == 480) ) ||
	     ((iWidth == 352) && (iHeight == 240) ) )
		 qsFormat = QString ("ntsc");

	if ( ((iWidth == 720) && (iHeight == 576) ) ||
	     ((iWidth == 704) && (iHeight == 576) ) ||
	     ((iWidth == 352) && (iHeight == 576) ) ||
	     ((iWidth == 352) && (iHeight == 288) ) )
		 qsFormat = QString ("pal");

	return qsFormat;
}

xine_stream_t *XineInfo::getXineStream ()
{
	return m_pXineStream;
}

xine_t *XineInfo::getXine ()
{
	return m_pXine;
}

QString XineInfo::getFileName (bool bFullName)
{
	if (!bFullName)	{
		QFileInfo fileInfo(m_qsFileName);
		QString qsFileName = fileInfo.fileName();
		return qsFileName;
	}
	return m_qsFileName;
}

unsigned long int XineInfo::getLength ()
{
	// info in milli Seconds
	return m_iLength;
}

QString XineInfo::getLengthString ()
{
	return m_qsLength;
}

QString XineInfo::getRatio ()
{
	return m_qsRatio;
}

int XineInfo::getBits ()
{
	return m_iBits;
}

QString XineInfo::getFormat (bool bVideo)
{
	if (bVideo)
		return m_qsVideoFormat;
	return m_qsAudioFormat;
}

QString XineInfo::getCodec (bool bVideo)
{
	if (bVideo)
		return m_qsVideoCodec;
	return m_qsAudioCodec;
}

uint XineInfo::getSize ()
{
	// in bytes
	return m_iSize;
}

QString XineInfo::getSizeString ()
{
	return m_qsSize;
}

int XineInfo::getResolution (bool bWidth)
{
	if (bWidth)
		return m_iResolutionWidth;
	else 
		return m_iResolutionHeight;
}

QString XineInfo::getResolutionString ()
{
	return m_qsResolution;
}

float XineInfo::getFPS ()
{
	return m_fFPS;
}

QString XineInfo::getStatus ()
{
	return m_qsStatus;
}

int XineInfo::getSample ()
{
	return m_iSample;
}

QString XineInfo::getSampleString ()
{
	return m_qsSample;
}

int XineInfo::getBPS (bool bVideo)
{
	if (bVideo)
		return m_iVideoBPS;
	return m_iAudioBPS;	
}

QString XineInfo::getBPSString (bool bVideo)
{
	if (bVideo)
		return m_qsVideoBPS;
	return m_qsAudioBPS;
}

bool XineInfo::hasAudio ()
{
	return m_bHasAudio;
}

bool XineInfo::hasVideo ()
{
	return m_bHasVideo;
}

bool XineInfo::audioHandled ()
{
	return m_bAudioHandled;
}

bool XineInfo::videoHandled ()
{
	return m_bVideoHandled;
}

#include <qimage.h>
#include "convert.h"
QImage XineInfo::getScreenshot(QString qsMovieFile)
{
	uchar *pRgbPile = NULL;
	int iWidth, iHeight;
	double fScaleFactor;

	setStream (qsMovieFile);
	GetScreenshot (pRgbPile, iWidth, iHeight, fScaleFactor);
	
	if (!pRgbPile)
		return QImage();
	QImage screenShot(pRgbPile, iWidth, iHeight, 32, 0, 0, QImage::IgnoreEndian);
	if (fScaleFactor >= 1.0)
		iWidth = (int)((double) iWidth * fScaleFactor);
	else
		iHeight = (int) ((double) iHeight / fScaleFactor);

	screenShot = screenShot.smoothScale(iWidth, iHeight);
	delete []pRgbPile;	
	return screenShot;
}

void XineInfo::GetScreenshot(uchar*& rgb32BitData, int& videoWidth, int& videoHeight, double& scaleFactor) const
{

  uint8_t   *yuv = NULL, *y = NULL, *u = NULL, *v =NULL;
  
  int        width, height, ratio, format;
  double     desired_ratio, image_ratio;

  if (!xine_get_current_frame (m_pXineStream, &width, &height, &ratio, &format, NULL))
    return;

  yuv = new uint8_t[((width+8) * (height+1) * 2)];
  if (yuv == NULL)
    {
      printf ("Not enough memory to make screenshot!\n");
      return;
    }  

  xine_get_current_frame (m_pXineStream, &width, &height, &ratio, &format, yuv);

  videoWidth = width;
  videoHeight = height;

/*
 * convert to yv12 if necessary
 */

  switch (format) {
  case XINE_IMGFMT_YUY2:
    {
      uint8_t *yuy2 = yuv;

      yuv = new uint8_t[(width * height * 2)];
      if (yuv == NULL)
      {
//        errorOut("Not enough memory to make screenshot!");
        return;
      }  
      y = yuv;
      u = yuv + width * height;
      v = yuv + width * height * 5 / 4;

      yuy2Toyv12 (y, u, v, yuy2, width, height);

      delete [] yuy2;
    }
    break;
  case XINE_IMGFMT_YV12:
    y = yuv;
    u = yuv + width * height;
    v = yuv + width * height * 5 / 4;

    break;
  default:
    {
      printf ("Screenshot: Format %s not supported!\n", (char*)&format);
      delete [] yuv;
      return;
    }  
  }

  /*
   * convert to rgb
   */

  rgb32BitData = yv12ToRgb (y, u, v, width, height);


  image_ratio = (double) width / (double) height;


  switch (ratio) {
  case XINE_VO_ASPECT_ANAMORPHIC:  /* anamorphic  */
//  case XINE_VO_ASPECT_PAN_SCAN:  /* depreciated */
    desired_ratio = 16.0 /9.0;
    break;
  case XINE_VO_ASPECT_DVB:         /* 2.11:1 */
    desired_ratio = 2.11/1.0;
    break;
  case XINE_VO_ASPECT_SQUARE:      /* square pels */
//  case XINE_VO_ASPECT_DONT_TOUCH:  /* depreciated */
    desired_ratio = image_ratio;
    break; 
  default:
    printf ("Screenshot: Unknown aspect ratio: %d - using 4:3", ratio);
  case XINE_VO_ASPECT_4_3:         /* 4:3   */
    desired_ratio = 4.0 / 3.0;
    break;
  }

  scaleFactor = desired_ratio / image_ratio;

  delete [] yuv;
}


