/****************************************************************************
** DialogTextFont class
**
** Created: Tue Feb 18 22:48:00 2004
**      by: Varol Okan using Kate
**
**
****************************************************************************/

#include <qfont.h>
#include <qstringlist.h>
#include <qfontdatabase.h>
#include <qtextedit.h>
#include <qlistbox.h>
#include <qcombobox.h>
#include <qcolordialog.h>
#include <qpainter.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qslider.h>
#include <qlineedit.h>
#include <qlabel.h>

#include "global.h"
#include "dialogtextfont.h"
#include "dialoganimation.h"

// Here we initialize the static member variables ...
QFont   DialogTextFont::m_previousFont       = QFont   ();
QString DialogTextFont::m_previousText       = QString ();
QPoint  DialogTextFont::m_previousTextAlign  = QPoint  ( Qt::AlignHCenter, Qt::AlignVCenter );
QColor  DialogTextFont::m_previousBackground = QColor  ( TRANSPARENT_COLOR );
QColor  DialogTextFont::m_previousForeground = QColor  ( 0, 0, 0 );

DialogTextFont::DialogTextFont(QWidget * parent, const char * name, WFlags f)
	: uiDialogTextFont (parent, name, f)
{
	initMe();
}

DialogTextFont::~DialogTextFont()
{

}

void DialogTextFont::setFont(QFont &newFont)
{
	m_qfFont = newFont;
	// Here we set the fields ...
	// Underline / strikeout / pointSize / fontFamily / script / fontStyle
	int t, iIndex;
	for (iIndex=0;iIndex<(int)m_pListFonts->count();iIndex++)	{
		if (m_pListFonts->text(iIndex) == newFont.family())	{
			m_pListFonts->setCurrentItem(iIndex);
			break;
		}
	}
	// Next we try to set the style of the font.
	iIndex = 0;
	QFontDatabase database;
	QString qsStyle = database.styleString (newFont);
	for (t=0;t<m_pComboFontStyle->count();t++)	{
		if (m_pComboFontStyle->text(t) == qsStyle)	{
			iIndex = t;
			break;
		}
	}
	m_pComboFontStyle->setCurrentItem(iIndex);
	// Here we get the font size which comes closest to the newFont pointSize (or the same).
	int iAvailFontSize, iFontSize = newFont.pointSize();
	for (iIndex=0;iIndex<m_pComboFontSizes->count();iIndex++)	{
		iAvailFontSize = m_pComboFontSizes->text(iIndex).toInt();
		if (iAvailFontSize >= iFontSize)
			break;
	}
	m_pComboFontSizes->setCurrentItem(iIndex);

	if (newFont.underline())
		m_pButtonUnderline->setOn(true);
	if (newFont.strikeOut())
		m_pButtonStrikeout->setOn(true);

	m_pTextEdit->setFont(newFont);
}

QFont &DialogTextFont::getFont ()
{
	QFontDatabase database;

	QString qsFamily = m_pListFonts->text(m_pListFonts->currentItem());
	QString qsStyle  = m_pComboFontStyle->text(m_pComboFontStyle->currentItem());
	int iPointSize   = m_pComboFontSizes->currentText().toInt();
	m_qfFont =  database.font(qsFamily, qsStyle, iPointSize );
	m_qfFont.setUnderline (m_pButtonUnderline->isOn());
	m_qfFont.setStrikeOut (m_pButtonStrikeout->isOn());

	return m_qfFont;
}

void  DialogTextFont::setText (QString &newText)
{
	m_pTextEdit->setText(newText);
}

QString DialogTextFont::getText ()
{
	return m_pTextEdit->text();
}

void  DialogTextFont::setAnimation (QString &qsAnimation)
{
	m_qsAnimation = qsAnimation;
}

QString &DialogTextFont::getAnimation ()
{
	return m_qsAnimation;
}

void  DialogTextFont::setTextAlign (QPoint &newAlign)
{
	// First we re-set the buttons ...
	m_pButtonLeft->   setOn(false);
	m_pButtonRight->  setOn(false);
	m_pButtonCenter-> setOn(false);
	m_pButtonJustify->setOn(false);

	if (newAlign.x() == Qt::AlignHCenter)
		m_pButtonCenter->setOn(true);
	else if (newAlign.x() == Qt::AlignHCenter)
		m_pButtonCenter->setOn(true);
	else if (newAlign.x() == Qt::AlignRight)
		m_pButtonRight->setOn(true);
	else if (newAlign.x() == Qt::AlignJustify)
		m_pButtonJustify->setOn(true);

	if (newAlign.y() == Qt::AlignVCenter)
		m_pButtonVCenter->setOn(true);
	else
		m_pButtonVCenter->setOn(false);
	updateFont (2);
}

QPoint &DialogTextFont::getTextAlign ()
{
	static QPoint thePoint;
	thePoint.setX(0);
	thePoint.setY(0);
	if (m_pButtonLeft->isOn())
		thePoint.setX(Qt::AlignLeft);
	else if (m_pButtonCenter->isOn())
		thePoint.setX(Qt::AlignHCenter);
//	else if (m_pButtonCenter->isOn())
//		thePoint.setX(Qt::AlignHCenter);
	else if (m_pButtonRight->isOn())
		thePoint.setX(Qt::AlignRight);
	else if (m_pButtonJustify->isOn())
		thePoint.setX(Qt::AlignJustify);

	if (m_pButtonVCenter->isOn())
		thePoint.setY(Qt::AlignVCenter);
	else
		thePoint.setY(Qt::AlignTop);
	return thePoint;
}

bool DialogTextFont::getFit()
{
	return m_pButtonFit->isOn();
}

void DialogTextFont::setRect (QRect &rect)
{
	m_pEditX->setText(QString("%1").arg(rect.x()));
	m_pEditY->setText(QString("%1").arg(rect.y()));
	m_pEditWidth->setText(QString("%1").arg(rect.width()));
	m_pEditHeight->setText(QString("%1").arg(rect.height()));
}

QRect &DialogTextFont::getRect ()
{
	static QRect theRect;
	theRect = QRect (m_pEditX->text().toInt(), m_pEditY->text().toInt(), m_pEditWidth->text().toInt(), m_pEditHeight->text().toInt());
	return theRect;
}

void DialogTextFont::setTransparency (float fNewTransparency)
{
	if (fNewTransparency > 1.0f)
		fNewTransparency = 1.0f;
	if (fNewTransparency < 0.0f)
		fNewTransparency = 0.0f;
	m_fTransparency = fNewTransparency;
	m_pEditTransparency->setText (QString("%1").arg(fNewTransparency));
	m_pSliderTransparency->setValue ((int)(1000.0*fNewTransparency));
}

float DialogTextFont::getTransparency ()
{
	return m_fTransparency;
}

void DialogTextFont::updateButtons ()
{
	if (m_pButtonLeft->isOn())
		m_pButtonLeft->setPixmap (QPixmap ().fromMimeSource("text_cleft.png"));
	else
		m_pButtonLeft->setPixmap (QPixmap ().fromMimeSource("text_left.png"));

	if (m_pButtonCenter->isOn())
		m_pButtonCenter->setPixmap (QPixmap ().fromMimeSource("text_ccenter.png"));
	else
		m_pButtonCenter->setPixmap (QPixmap ().fromMimeSource("text_center.png"));

	if (m_pButtonRight->isOn())
		m_pButtonRight->setPixmap (QPixmap ().fromMimeSource("text_cright.png"));
	else
		m_pButtonRight->setPixmap (QPixmap ().fromMimeSource("text_right.png"));

	if (m_pButtonJustify->isOn())
		m_pButtonJustify->setPixmap (QPixmap ().fromMimeSource("text_cjustify.png"));
	else
		m_pButtonJustify->setPixmap (QPixmap ().fromMimeSource("text_justify.png"));

	if (m_pButtonVCenter->isOn())
		m_pButtonVCenter->setPixmap (QPixmap ().fromMimeSource("text_cvcenter.png"));
	else
		m_pButtonVCenter->setPixmap (QPixmap ().fromMimeSource("text_vcenter.png"));

	if (m_pButtonFit->isOn())
		m_pButtonFit->setPixmap (QPixmap ().fromMimeSource("text_cfit.png"));
	else
		m_pButtonFit->setPixmap (QPixmap ().fromMimeSource("text_fit.png"));

	if (m_pButtonUnderline->isOn())
		m_pButtonUnderline->setPixmap (QPixmap ().fromMimeSource("text_cunderline.png"));
	else
		m_pButtonUnderline->setPixmap (QPixmap ().fromMimeSource("text_underline.png"));

	if (m_pButtonStrikeout->isOn())
		m_pButtonStrikeout->setPixmap (QPixmap ().fromMimeSource("text_cstrikeout.png"));
	else
		m_pButtonStrikeout->setPixmap (QPixmap ().fromMimeSource("text_strikeout.png"));
}

void DialogTextFont::initMe()
{
	// Here we set all available options ...
	QFontDatabase database;
	setTransparency (0.0);
	QStringList listFonts = database.families();
	m_pListFonts->insertStringList(listFonts);
	// Next we search for a good starting font ...
	uint t, iIndex;
	for (iIndex=0;iIndex<m_pListFonts->count();iIndex++)	{
		if (m_pListFonts->text(iIndex) == QString ("Arial"))
			break;
		else if (m_pListFonts->text(iIndex) == QString ("Courier"))
			break;
		else if (m_pListFonts->text(iIndex) == QString ("Times New Roman"))
			break;
		else if (m_pListFonts->text(iIndex) == QString ("Times"))
			break;
	}
	QString fontName = m_pListFonts->text(iIndex);

	QStringList listStyles = database.styles (fontName);
	m_pComboFontStyle->insertStringList (listStyles);
	QString styleName = m_pComboFontStyle->text(0);
	m_pComboFontStyle->setCurrentItem(0);

	// Here we get the sizes of the font.
	QValueList<int> listSizes = database.pointSizes(fontName);
	if (listSizes.isEmpty())	{
		listSizes = database.smoothSizes(fontName, styleName);
		if (listSizes.isEmpty())	{
			listSizes = database.standardSizes();
		}
	}
	// So lets say we can set the fontFamily at least ...
	m_pListFonts->setCurrentItem(iIndex);
	// And here we take care of the FontSize ...
	iIndex = 0;
	for (t=0;t<listSizes.count();t++)	{
		m_pComboFontSizes->insertItem (QString ("%1").arg(listSizes[t]));
		if (listSizes[t] < 11)	// Lets get at least a nice big raedable size
			iIndex = t;
	}
	m_pComboFontSizes->setCurrentItem (iIndex);

	QString qsScripts("Unicode Latin Greek Cyrillic Armenian Georgian Runic Ogham SpacingModifiers CombiningMarks"
		"Hebrew Arabic Syriac Thaana"
		"Devanagari Bengali Gurmukhi Gujarati Oriya Tamil Telugu Kannada Malayalam Sinhala Thai Lao Tibetan Myanmar Khmer"
		"Han Hiragana Katakana Hangul Bopomofo Yi"
		"Ethiopic Cherokee CanadianAboriginal Mongolian"
		"CurrencySymbols LetterlikeSymbols NumberForms MathematicalOperators TechnicalSymbols GeometricSymbols MiscellaneousSymbols EnclosedAndSquare Braille");
	QStringList listScripts = QStringList::split (QString (" "), qsScripts);
	m_pComboScript->insertStringList(listScripts);
	m_pComboScript->setCurrentItem(1);

	// Here we handle the bad expansion of TextEdit. Keep it in line ...
	show ();

	QSize size = m_pTextEdit->size();
	m_pTextEdit->setFixedSize (size);
	//Finally we set the font of the TextEdit window ...
	setFont            ( m_previousFont       );
	setText            ( m_previousText       );
	setTextAlign       ( m_previousTextAlign  );
	setBackgroundColor ( m_previousBackground );
	setForegroundColor ( m_previousForeground );
	QPainter thePainter (m_pFrameBackgroundColor);
	thePainter.drawLine(0, 0, m_pFrameBackgroundColor->width(), m_pFrameFontColor->height());

	m_pTextEdit->setTextFormat(PlainText);

	// Finally we set up the callback routines ...
	connect (m_pListFonts, SIGNAL(highlighted (int)), this, SLOT(slotFontFamilyChanged(int)));
	connect (m_pComboFontStyle, SIGNAL(activated (int)), this, SLOT(slotFontStyleChanged(int)));
	connect (m_pComboFontSizes, SIGNAL(activated (int)), this, SLOT(slotFontSizeChanged(int)));
	connect (m_pComboFontSizes, SIGNAL(textChanged ( const QString & )), this, SLOT(slotFontSizeChanged( const QString & )));
	connect (m_pComboScript, SIGNAL(activated (int)), this, SLOT(slotScriptChanged(int)));

	connect (m_pButtonLeft, SIGNAL(toggled (bool)), this, SLOT(slotLeft(bool)));
	connect (m_pButtonCenter, SIGNAL(toggled (bool)), this, SLOT(slotCenter(bool)));
	connect (m_pButtonRight, SIGNAL(toggled (bool)), this, SLOT(slotRight(bool)));
	connect (m_pButtonJustify, SIGNAL(toggled (bool)), this, SLOT(slotJustify(bool)));
	connect (m_pButtonVCenter, SIGNAL(toggled (bool)), this, SLOT(slotVCenter(bool)));
	connect (m_pButtonFit, SIGNAL(toggled (bool)), this, SLOT(slotFit(bool)));
	connect (m_pButtonStrikeout, SIGNAL(toggled (bool)), this, SLOT(slotStrikeout(bool)));
	connect (m_pButtonUnderline, SIGNAL(toggled (bool)), this, SLOT(slotUnderline(bool)));

	connect (m_pButtonAnimation,  SIGNAL(clicked()), this, SLOT(slotAnimation()));
	connect (m_pButtonFontColor,  SIGNAL(clicked()), this, SLOT(slotForegroundColor()));
	connect (m_pButtonBackgroundColor, SIGNAL(clicked()), this, SLOT(slotBackgroundColor()));
	connect (m_pSliderTransparency, SIGNAL(valueChanged (int)), this, SLOT(slotTransparency(int)));

	connect (buttonOk, SIGNAL(clicked()), this, SLOT(slotOkay()));
	updateButtons();
}

void DialogTextFont::paintEvent (QPaintEvent *pPaint)
{
	// first we call the main function
	QWidget::paintEvent(pPaint);
	// and here we do  our little do ...
	QColor transparentColor = QColor(TRANSPARENT_COLOR);
	if (transparentColor.rgb() ==  m_backgroundColor.rgb())	{	//m_pFrameBackgroundColor->paletteForegroundColor())	{
		QPainter thePainter (m_pFrameBackgroundColor);
		thePainter.setPen(QPen(QColor(0, 0, 0), 2));
		thePainter.drawLine(6, 6, m_pFrameBackgroundColor->width()-6, m_pFrameFontColor->height()-6);
		thePainter.drawLine(6, m_pFrameFontColor->height()-6, m_pFrameBackgroundColor->width()-6, 6);
	}
}

void DialogTextFont::slotForegroundColor ()
{
	QColor newColor = QColorDialog::getColor(m_foregroundColor, this);
	setForegroundColor (newColor);
}

void DialogTextFont::slotBackgroundColor ()
{
	QColor transparentColor = QColor(TRANSPARENT_COLOR);
	QColor newColor = QColorDialog::getColor(transparentColor, this);
	setBackgroundColor (newColor);
}

void DialogTextFont::slotTransparency (int iNewValue)
{
	// Called from the slider
	if (iNewValue < 5)
		iNewValue = 0;
	if (iNewValue > 995)
		iNewValue = 1000;
	// [0.0 .. 1.0]=[0..999]=> 1.0=fully transparent.
	m_fTransparency = (float)((float)iNewValue/1000.0);
	m_pEditTransparency->setText (QString("%1").arg(m_fTransparency, 0, 'f', 2));
}

void DialogTextFont::slotFontFamilyChanged(int)
{
	updateFont(0);
}

void DialogTextFont::slotFontStyleChanged(int)
{
	updateFont(1);
}

void DialogTextFont::slotFontSizeChanged(int)
{
	updateFont(2);
}

void DialogTextFont::slotFontSizeChanged( const QString & )
{
	updateFont(2);
}

void DialogTextFont::slotScriptChanged(int)
{
	updateFont(0);
}

void DialogTextFont::slotLeft (bool)
{
	if (m_pButtonLeft->isOn())	{
		m_pButtonCenter->setOn(false);
		m_pButtonRight->setOn(false);
		m_pButtonJustify->setOn(false);
	}
	updateFont(2);
}

void DialogTextFont::slotCenter (bool)
{
	if (m_pButtonCenter->isOn())	{
		m_pButtonLeft->setOn(false);
		m_pButtonRight->setOn(false);
		m_pButtonJustify->setOn(false);
	}
	updateFont(2);
}

void DialogTextFont::slotRight (bool)
{
	if (m_pButtonRight->isOn())	{
		m_pButtonCenter->setOn(false);
		m_pButtonLeft->setOn(false);
		m_pButtonJustify->setOn(false);
	}
	updateFont(2);
}

void DialogTextFont::slotJustify (bool)
{
	if (m_pButtonJustify->isOn())	{
		m_pButtonCenter->setOn(false);
		m_pButtonRight->setOn(false);
		m_pButtonLeft->setOn(false);
	}
	updateFont(2);
}

void DialogTextFont::slotVCenter (bool)
{
	updateFont(2);
}

void DialogTextFont::slotFit (bool bFitForIt)
{
	// here we enable - disable the coordinates if the user wants to
	// fit the box to the text extension.
	m_pEditWidth-> setEnabled(!bFitForIt);
	m_pEditHeight->setEnabled(!bFitForIt);
	updateFont(2);
}

void DialogTextFont::slotUnderline (bool)
{
	updateFont(2);
}

void DialogTextFont::slotStrikeout (bool)
{
	updateFont(2);
}

void DialogTextFont::slotAnimation ()
{
	// This function gets the Animation Dialog
	// Within this dialo, you can create a script
	// (or hardcode) Attributes of the object.
	// Before rendering the menu the script will 
	// get executed and all data sent to stdout 
	// will be captured in the data file, wich 
	// will provide the data for the animation.
	QString qsEmpty;
	DialogAnimation dialog ( this );
	dialog.initMe ( m_qsAnimation, 300, 25.0, DialogAnimation::TypeText );
	dialog.setFontName ( m_qfFont.toString () );
	if ( dialog.exec () == QDialog::Accepted )
		m_qsAnimation = dialog.string ();

	if (m_qsAnimation.length () < 5)
		m_qsAnimation = qsEmpty;
}

void  DialogTextFont::updateFont(uint iSwitch)
{
	uint t;
	int iIndex;
	// First see what styles are avail ...
	QFontDatabase database;
	QString fontName  = m_pListFonts->text(m_pListFonts->currentItem());
	QString styleName = m_pComboFontStyle->text(m_pComboFontStyle->currentItem());
	int iFontSize     = m_pComboFontSizes->currentText ().toInt();
	switch (iSwitch)	{
	case 0:	{
		// The fontName has changed, so we need to get the styles avail for this new font.
		QStringList listStyles = database.styles (fontName);
		m_pComboFontStyle->clear();
		m_pComboFontStyle->insertStringList (listStyles);
		iIndex = findString(listStyles, styleName);
		if (iIndex == -1)
			iIndex = 0;
		styleName = m_pComboFontStyle->text(iIndex);
		m_pComboFontStyle->setCurrentItem(iIndex);
	}
	case 1:
		// The style has changed
		m_pComboFontSizes->clear();
		QValueList<int> listSizes = database.pointSizes(fontName);
		if (listSizes.isEmpty())	{
			listSizes = database.smoothSizes(fontName, styleName);
			if (listSizes.isEmpty())	{
				listSizes = database.standardSizes();
			}
		}
		iIndex = 0;
		for (t=0;t<listSizes.count();t++)	{
			m_pComboFontSizes->insertItem (QString ("%1").arg(listSizes[t]));
			if (listSizes[t] == iFontSize)
				iIndex = t;
		}
		m_pComboFontSizes->setCurrentItem (iIndex);
	}
	QString qsText = m_pTextEdit->text();
	int iAlignement;
	iAlignement = Qt::AlignLeft;
	if (m_pButtonCenter->isOn())
		iAlignement =  Qt::AlignHCenter;
	else if (m_pButtonRight->isOn())
		iAlignement = Qt::AlignRight;
	else if (m_pButtonJustify->isOn())
		iAlignement = Qt::AlignJustify;

	m_pTextEdit->setText(qsText);
	// and last we update the TextEdit accordingly
	if (m_pButtonVCenter->isOn())
		iAlignement |= Qt::AlignVCenter;
	else
		iAlignement |= Qt::AlignTop;

	m_pTextEdit->setAlignment ( iAlignement );
	updateButtons ();
	m_pTextEdit->setFont(getFont());
};

int DialogTextFont::findString (QStringList &list, QString string)
{
	for (uint t=0;t<list.count();t++)	{
		if (list[t] == string)
			return t;
	}
	return -1;
}

QColor &DialogTextFont::getBackgroundColor ()
{
	return m_backgroundColor;
}

QColor &DialogTextFont::getForegroundColor ()
{
	return m_foregroundColor;
}

void DialogTextFont::setBackgroundColor (QColor &newColor)
{
	// Check if the user pressed cancel ...
	if (!newColor.isValid())
		return;
	m_pFrameBackgroundColor->setPaletteBackgroundColor(newColor);
	update();
	m_pTextEdit->setPaletteBackgroundColor   (newColor);
	m_backgroundColor = newColor;
}

void DialogTextFont::setForegroundColor (QColor &newColor)
{
	m_pFrameFontColor->setPaletteBackgroundColor(newColor);
	m_pTextEdit->setPaletteForegroundColor (newColor);
	m_foregroundColor = newColor;
}

void DialogTextFont::slotOkay()
{
	// here we store the last settings in the static members so the next time we open this
	// dialog we will have the right settings ...
	m_previousFont = getFont ();
	m_previousText = getText ();
	m_previousTextAlign  = getTextAlign ();
	m_previousForeground = getForegroundColor ();
	m_previousBackground = getBackgroundColor ();

	accept ();
}


/* scripts ...
QFont::Latin - consists of most alphabets based on the original Latin alphabet.
QFont::Greek - covers ancient and modern Greek and Coptic.
QFont::Cyrillic - covers the Slavic and non-Slavic languages using cyrillic alphabets. 
QFont::Armenian - contains the Armenian alphabet used with the Armenian language. 
QFont::Georgian - covers at least the language Georgian. 
QFont::Runic - covers the known constituents of the Runic alphabets used by the early and medieval societies in the Germanic, Scandinavian, and Anglo-Saxon areas. 
QFont::Ogham - is an alphabetical script used to write a very early form of Irish. 
QFont::SpacingModifiers - are small signs indicating modifications to the preceeding letter. 
QFont::CombiningMarks - consist of diacritical marks not specific to a particular alphabet, diacritical marks used in combination with mathematical and technical symbols, and glyph encodings applied to multiple letterforms. 

Middle Eastern scripts (right to left): 

QFont::Hebrew - is used for writing Hebrew, Yiddish, and some other languages. 
QFont::Arabic - covers the Arabic language as well as Persian, Urdu, Kurdish and some others. 
QFont::Syriac - is used to write the active liturgical languages and dialects of several Middle Eastern and Southeast Indian communities. 
QFont::Thaana - is used to write the Maledivian Dhivehi language. 

South and Southeast Asian scripts (left to right with few historical exceptions): 

QFont::Devanagari - covers classical Sanskrit and modern Hindi as well as several other languages. 
QFont::Bengali - is a relative to Devanagari employed to write the Bengali language used in West Bengal/India and Bangladesh as well as several minority languages. 
QFont::Gurmukhi - is another Devanagari relative used to write Punjabi. 
QFont::Gujarati - is closely related to Devanagari and used to write the Gujarati language of the Gujarat state in India. 
QFont::Oriya - is used to write the Oriya language of Orissa state/India. 
QFont::Tamil - is used to write the Tamil language of Tamil Nadu state/India, Sri Lanka, Singapore and parts of Malaysia as well as some minority languages. 
QFont::Telugu - is used to write the Telugu language of Andhra Pradesh state/India and some minority languages. 
QFont::Kannada - is another South Indian script used to write the Kannada language of Karnataka state/India and some minority languages. 
QFont::Malayalam - is used to write the Malayalam language of Kerala state/India. 
QFont::Sinhala - is used for Sri Lanka's majority language Sinhala and is also employed to write Pali, Sanskrit, and Tamil. 
QFont::Thai - is used to write Thai and other Southeast Asian languages. 
QFont::Lao - is a language and script quite similar to Thai. 
QFont::Tibetan - is the script used to write Tibetan in several countries like Tibet, the bordering Indian regions and Nepal. It is also used in the Buddist philosophy and liturgy of the Mongolian cultural area.
QFont::Myanmar - is mainly used to write the Burmese language of Myanmar (former Burma). 
QFont::Khmer - is the official language of Kampuchea. 

East Asian scripts (traditionally top-down, right to left, modern often horizontal left to right): 

QFont::Han - consists of the CJK (Chinese, Japanese, Korean) idiographic characters.
QFont::Hiragana - is a cursive syllabary used to indicate phonetics and pronounciation of Japanese words. 
QFont::Katakana - is a non-cursive syllabic script used to write Japanese words with visual emphasis and non-Japanese words in a phonetical manner. 
QFont::Hangul - is a Korean script consisting of alphabetic components. 
QFont::Bopomofo - is a phonetic alphabet for Chinese (mainly Mandarin). 
QFont::Yi - (also called Cuan or Wei) is a syllabary used to write the Yi language of Southwestern China, Myanmar, Laos, and Vietnam. 

Additional scripts that do not fit well into the script categories above: 

QFont::Ethiopic - is a syllabary used by several Central East African languages. 
QFont::Cherokee - is a left-to-right syllabic script used to write the Cherokee language. 
QFont::CanadianAboriginal - consists of the syllabics used by some Canadian aboriginal societies. 
QFont::Mongolian - is the traditional (and recently reintroduced) script used to write Mongolian. 

Symbols: 

QFont::CurrencySymbols - contains currency symbols not encoded in other scripts. 
QFont::LetterlikeSymbols - consists of symbols derived from ordinary letters of an alphabetical script. 
QFont::NumberForms - are provided for compatibility with other existing character sets. 
QFont::MathematicalOperators - consists of encodings for operators, relations and other symbols like arrows used in a mathematical context. 
QFont::TechnicalSymbols - contains representations for control codes, the space symbol, APL symbols and other symbols mainly used in the context of electronic data processing. 
QFont::GeometricSymbols - covers block elements and geometric shapes. 
QFont::MiscellaneousSymbols - consists of a heterogeneous collection of symbols that do not fit any other Unicode character block, e.g. Dingbats. 
QFont::EnclosedAndSquare - is provided for compatibility with some East Asian standards. 
QFont::Braille - is an international writing system used by blind people. This script encodes the 256 eight-dot patterns with the 64 six-dot patterns as a subset. 

QFont::Unicode - includes all the above scripts.
*/


