/****************************************************************************
** DialogThumbnail
**
** Created: Thu Jan 18 22:48:00 2007
**      by: Varol Okan using XEmacs
**
****************************************************************************/

#include <qlabel.h>
#include <qslider.h>
#include <qdialog.h>
#include <qlistbox.h>
#include <qgroupbox.h>
#include <qfileinfo.h>
#include <qiconview.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qprogressbar.h>

#include "global.h"
#include "utils.h"
#include "dvdmenu.h"
#include "dialogfont.h"
#include "qdvdauthor.h"
#include "dialogframe.h"
#include "sourcefileentry.h"
#include "dialogthumbnail.h"
#include "dragndropcontainer.h"
#include "qplayer/mediacreator.h"

DialogThumbnail::TextAttr  DialogThumbnail::m_textAttr  = TextAttr  ( );
DialogThumbnail::FrameAttr DialogThumbnail::m_frameAttr = FrameAttr ( );

DialogThumbnail::ThumbnailRequest::ThumbnailRequest ( DialogThumbnail *pDialog, QIconViewItem *pItem, QString qsFile, long iOffset )
  : ExecuteJob ( pDialog )
{
  pImage          = new QImage;
  pOwnerItem      = pItem;
  pParent         = pDialog;
  iMSecondsOffset = iOffset;
  qsFileName      = qsFile;    
}

DialogThumbnail::ThumbnailRequest::~ThumbnailRequest ( )
{
  if ( pImage )
    delete pImage;
}

bool DialogThumbnail::ThumbnailRequest::response ( )
{  
  if ( ! pParent )
    return false;
  // Rather then copy data back and forth I simply steal the pointer to the QImage object
  QImage *pTempImage = pImage;
  pImage = NULL;
  pParent->initWork ( pOwnerItem, pTempImage );

  return true;
}

DialogThumbnail::WorkWaiting::WorkWaiting ( QIconViewItem *pViewItem, QImage *pImg )
{
  pItem  = pViewItem;
  pImage = pImg;
}

DialogThumbnail::WorkWaiting::~WorkWaiting ( )
{
  if ( pImage )
    delete pImage; 
};


DialogThumbnail::DialogThumbnail ( QDVDAuthor *pDVDAuthor, const char *pName, WFlags f )
  : uiDialogThumbnail ( pDVDAuthor, pName, f )
{
  m_pSourceFileInfo  = NULL;
  m_iCurrentChapter  = 0;
  m_iThumbnailWidth  = 120;
  m_iThumbnailHeight = 120;
  m_iThumbnailStretchMode = 0;
  m_iSliderWidthStart     = 0;
  m_iSliderHeightStart    = 0;

  if ( pDVDAuthor ) {
    DragNDropContainer *pDND = pDVDAuthor->getDragNDropContainer ( );
    DVDMenu *pDVDMenu = pDVDAuthor->getVMGMenu ( );
    m_pEditWidth ->setText ( QString ( "%1" ).arg ( pDND->sizeButton.width  ( ) ) );
    m_pEditHeight->setText ( QString ( "%1" ).arg ( pDND->sizeButton.height ( ) ) );
    m_textAttr.foregroundColor  = pDND->colorForeground.rgb ( ) & 0x00FFFFFF;
    m_textAttr.backgroundColor |= 0xFF000000;
    m_textAttr.font             = pDND->font;
    if ( pDVDMenu )
      m_frameAttr.color          = pDVDMenu->getColor ( 1 ).rgb ( ); // def highlighted color
    //give the default DND font a more appropriate size ...
    if ( m_textAttr.font.pixelSize ( ) != -1 )
      m_textAttr.font.setPixelSize ( 14 );
    else 
      m_textAttr.font.setPointSize ( 14 );
  }

  connect ( m_pButtonOk,        SIGNAL ( clicked ( ) ), this, SLOT ( hide        ( ) ) );
  connect ( m_pButtonFont,      SIGNAL ( clicked ( ) ), this, SLOT ( slotFont    ( ) ) );
  connect ( m_pButtonFrame,     SIGNAL ( clicked ( ) ), this, SLOT ( slotFrame   ( ) ) );
  connect ( m_pButtonRefresh,   SIGNAL ( clicked ( ) ), this, SLOT ( slotRefresh ( ) ) );
  connect ( m_pButtonReload,    SIGNAL ( clicked ( ) ), this, SLOT ( slotReload  ( ) ) );
  connect ( m_pSliderSize,      SIGNAL ( valueChanged ( int ) ), this, SLOT ( slotSizeChanged   ( int ) ) );
  connect ( m_pSliderSpace,     SIGNAL ( valueChanged ( int ) ), this, SLOT ( slotSpaceChanged  ( int ) ) );
  connect ( m_pComboTextType,   SIGNAL ( activated    ( int ) ), this, SLOT ( slotTextType      ( int ) ) );
  connect ( m_pSliderSize,      SIGNAL ( sliderPressed    ( ) ), this, SLOT ( slotSizePressed   ( ) ) );
  connect ( m_pSliderSize,      SIGNAL ( sliderReleased   ( ) ), this, SLOT ( slotSizeReleased  ( ) ) );
  connect ( m_pSliderSpace,     SIGNAL ( sliderReleased   ( ) ), this, SLOT ( slotSpaceReleased ( ) ) );
  connect ( m_pListBoxChapters, SIGNAL ( currentChanged   ( QListBoxItem * ) ), 
	    this, SLOT ( slotChapterChanged( QListBoxItem * ) ) );
}

DialogThumbnail::~DialogThumbnail ( )
{
  // remove any outstanding requests ...
  MediaCreator::unregisterFromMediaScanner ( this );
  MediaCreator::unregisterFromMediaScanner ( m_pSourceFileInfo );
}

void DialogThumbnail::initSourceFileInfo ( SourceFileInfo *pInfo, QString &qsAction )
{
  m_pListBoxChapters   ->clear ( );
  m_pIconViewThumbnails->clear ( );

  if ( ! pInfo )
    return;  

  QFileInfo fileInfo ( pInfo->qsFileName );
  setCaption ( QString ( "Thumbnail Dialog : " ) + fileInfo.fileName ( ) );
  m_pSourceFileInfo = pInfo;
  m_qsActionString  = qsAction;
  m_pListBoxChapters->insertStringList ( pInfo->listChapters );
  m_pProgressBar->setTotalSteps ( pInfo->listChapters.count( ) - 1 );

  slotRefresh ( );
}

void DialogThumbnail::timerEvent ( QTimerEvent *pEvent )
{
  if ( m_pSourceFileInfo && pEvent->timerId ( ) == MEDIASCANNER_EVENT + 4 ) {
    int t;
    WorkWaiting *pWork;
    QValueList<WorkWaiting *> tempList;
    // to decrease threading issues, we quickly remove the used Work objects from the member list.
    for ( t=0;t<(int)m_listOfWorkToDo.count ( ); t++ ) {
      pWork = m_listOfWorkToDo [ t ];
      m_listOfWorkToDo.remove ( pWork );
      tempList.append ( pWork );
    }
    for ( t=0;t<(int)tempList.count ( ); t++ ) {
      pWork = tempList [ t ];
      updateItem ( pWork->pItem, pWork->pImage );
      delete pWork;
      m_pProgressBar->setProgress ( m_iProgress ++ );
    }
    if ( (int)m_iProgress >= m_pProgressBar->totalSteps ( )-2 )
      m_pIconViewThumbnails->arrangeItemsInGrid ( TRUE );
  }

  uiDialogThumbnail::timerEvent ( pEvent );
}

void DialogThumbnail::initWork ( QIconViewItem *pItem, QImage *pImage )
{
  WorkWaiting *pNewWork = new WorkWaiting ( pItem, pImage );
  m_listOfWorkToDo.append ( pNewWork );
}

DialogThumbnail::TextAttr  *DialogThumbnail::textAttr  ( )
{
  return &m_textAttr;
}

DialogThumbnail::FrameAttr *DialogThumbnail::frameAttr ( )
{
  return &m_frameAttr;
}

QString DialogThumbnail::getChapter ( QIconViewItem *pItem )
{
  QString qsChapter;
  if ( pItem )  {
  	QStringList list = QStringList::split ( STRING_SEPARATOR, pItem->key ( ) );
  	qsChapter = list[ list.size ( ) - 1 ];
  }
  return  qsChapter;
}

bool DialogThumbnail::updateItem ( QIconViewItem *pItem, QImage *pImage )
{
  if ( ! pImage )
    return false;

  QString   qsFileName, qsChapter;
  Utils     theUtils;
  QFileInfo fileInfo ( m_pSourceFileInfo->qsFileName );
    
  // At this point we should add the image to the IconView ...
  qsChapter  = getChapter ( pItem );
  qsChapter  =  qsChapter.remove  ( ':' );
  qsChapter  =  qsChapter.replace ( '.', "_" );
  
  qsFileName  = theUtils.getTempPath ( fileInfo.baseName ( ) );
  qsFileName += QString ( "/thumb_%1.png" ).arg  ( qsChapter );
  pImage->save ( qsFileName, "PNG", 100 );

  if ( ! pItem )
    return false;

  QPixmap thePixmap;
  QImage::ScaleMode scaleMode = QImage::ScaleMin;
  if ( m_iThumbnailStretchMode ==  1 )
    scaleMode = QImage::ScaleMax;
  if ( m_iThumbnailStretchMode ==  2 )
    scaleMode = QImage::ScaleFree;  

  *pImage = pImage->smoothScale ( m_iThumbnailWidth, m_iThumbnailHeight, scaleMode );
  if ( thePixmap.convertFromImage ( *pImage ) )
    pItem->setPixmap ( thePixmap );

  return true;
}

bool DialogThumbnail::getBufferedThumbnail ( )
{
  QIconViewItem *pItem = NULL;

  while ( m_iCurrentChapter < m_pSourceFileInfo->listChapters.count ( ) ) {
    pItem = createCurrentItem ( false );
    if ( ! pItem )
      return true;

    m_pProgressBar->setProgress ( m_iProgress );
    m_iCurrentChapter ++;
  }
  return false;
}

void DialogThumbnail::slotFrame ( )
{
  DialogFrame frameDialog ( this );
  frameDialog.initMe      ( m_frameAttr.iStyle, 
			    m_frameAttr.iWidth, 
			    m_frameAttr.iJoin, 
			    m_frameAttr.color );
  if (  frameDialog.exec  (  ) == QDialog::Rejected )
    return;

  m_frameAttr.iStyle = frameDialog.style         ( );
  m_frameAttr.iJoin  = frameDialog.join          ( );
  m_frameAttr.iWidth = frameDialog.frameWidth    ( );
  m_frameAttr.color  = frameDialog.color ( ).rgb ( );
}

void DialogThumbnail::slotFont ( )
{
  DialogFont fontDialog ( this );
  QString qsText ( "No Text" );
  if ( m_pComboTextType->currentItem ( ) == 0 )
    qsText = "00:00:00.000";
  else if ( m_pComboTextType->currentItem ( ) == 1 )
    qsText = "00:00:00";
  else if ( m_pComboTextType->currentItem ( ) == 2 )
    qsText = "Chapter 1";

  fontDialog.initMe     ( qsText );
  fontDialog.setTheFont ( m_textAttr.font );
  fontDialog.setBackgroundColor ( m_textAttr.backgroundColor );
  fontDialog.setForegroundColor ( m_textAttr.foregroundColor );
  if ( fontDialog.exec ( ) == QDialog::Rejected )
    return;

  m_textAttr.qsText = ""; // of no interest or the ThumbnailDialog.
  m_textAttr.font   = fontDialog.getFont ( );
  m_textAttr.backgroundColor = fontDialog.backgroundColor ( );
  m_textAttr.foregroundColor = fontDialog.foregroundColor ( );
}

void DialogThumbnail::slotTextType ( int iNewTextType )
{
  QIconViewItem *pItem = m_pIconViewThumbnails->firstItem ( );
  while ( pItem ) {
    setItemText ( pItem, iNewTextType );
    pItem = pItem->nextItem ( );
  }
}

void DialogThumbnail::setItemText ( QIconViewItem *pItem, int iTextType )
{
  // we support 4 different tet types
  // 0 == Timestamp          E.g. "00:00:00.000"
  // 1 == Timestamp seconds  E.g. "00:00:00"
  // 2 == Chapter No         E.g. "Chapter 2"
  // 3 == None               E.g. ""
  QString qsText;
  QString qsChapter = getChapter ( pItem );
  switch ( iTextType ) {
  case 0:
    qsText = qsChapter;
  break;
  case 1:
    qsText = qsChapter.left ( qsChapter.length ( ) - 4 );
  break;
  case 2: {
    int t;
    for ( t=0;t<(int)m_pSourceFileInfo->listChapters.count ( ); t++ ) {
      if ( m_pSourceFileInfo->listChapters[ t ] == qsChapter ) {
	qsText = QString (  "Chapter %1" ).arg ( t + 1 );
	break;
      }
    }
  }
  break;
  };

  pItem->setText ( qsText );
}

void DialogThumbnail::slotSpaceReleased ( )
{
  int iValue = m_pSliderSpace->value ( );
    m_iSliderWidthStart  = 0;
  m_iSliderHeightStart = 0;
  m_pIconViewThumbnails->setSpacing  ( iValue );
  QSize oldSize = m_pIconViewThumbnails->size ( );
  m_pIconViewThumbnails->resize ( oldSize.width( ), oldSize.height ( )+1 );
  m_pIconViewThumbnails->resize ( oldSize.width( ), oldSize.height ( ) );  
}

void DialogThumbnail::slotSpaceChanged ( int iNewValue )
{
  m_pLabelSpace->setText ( QString ( "<p align=\"center\"><b>%1</b></p>" ).arg ( iNewValue ) );
}

void DialogThumbnail::slotSizeReleased ( )
{
  refresh ( false );
  m_iSliderWidthStart  = 0;
  m_iSliderHeightStart = 0;
  m_pSliderSize->setValue ( 0 );
}

void DialogThumbnail::slotSizePressed ( )
{
  // we need to keep the start values of Width / Height ...
  m_iSliderWidthStart  = m_pEditWidth-> text ( ).toInt ( );
  m_iSliderHeightStart = m_pEditHeight->text ( ).toInt ( );
}

void DialogThumbnail::slotSizeChanged ( int iNewSize )
{
  if ( ( m_iSliderWidthStart < 1 ) || ( m_iSliderHeightStart < 1 ) )
    return;
  // Values are between [ -100 .. +100 ] representing - 100% (0 size) and + 100% ( double the start size )
  float fMult   = ( (float)iNewSize / 100.0f ) + 1.0f;
  float fWidth  = fMult * (float)m_iSliderWidthStart;
  float fHeight = fMult * (float)m_iSliderHeightStart;
  m_pEditWidth ->setText ( QString ( "%1" ).arg ( (int)fWidth  ) );
  m_pEditHeight->setText ( QString ( "%1" ).arg ( (int)fHeight ) );
}

void DialogThumbnail::slotChapterChanged ( QListBoxItem *pListBoxItem )
{
  // Called when te user selects items in the ListBox on the left hand side.
  if ( ! pListBoxItem )
    return;

  int t;
  QString qsChapter = pListBoxItem->text ( );
  QIconViewItem *pItem = m_pIconViewThumbnails->firstItem ( );
  // First we ensure the current item is visible
  while ( pItem ) {
    if  ( getChapter ( pItem ) == qsChapter ) {
      m_pIconViewThumbnails->clearSelection    ( );
      m_pIconViewThumbnails->ensureItemVisible ( pItem );
      break;
    }
    pItem = pItem->nextItem ( );
  }

  // Next we loop through the ListBox and find each 
  // chapter that has been selected (multiple selection)
  for ( t=0;t<(int)m_pListBoxChapters->count ( ); t++ ) {
    pListBoxItem = m_pListBoxChapters->item ( t );
    if ( pListBoxItem && pListBoxItem->isSelected ( ) ) {
//printf ( "DialogThumbnail::chapterChanged <%p>\n", pListBoxItem );
      // Okay, and for this selected ListBox item, we find the 
      // associciated IconViewItem and do also sect it.
      pItem = m_pIconViewThumbnails->firstItem ( );
      // First we ensure the current item is visible
      while ( pItem ) {
	if  ( getChapter ( pItem ) == qsChapter ) {
	  m_pIconViewThumbnails->setSelected ( pItem, TRUE, TRUE );
	  pItem = NULL;
	}
	else
	  pItem = pItem->nextItem ( );
      }
    }
  }
}
/*
void DialogThumbnail::slotChapterChanged ( QListBoxItem *pListBoxItem )
{
// This one is for single selection only ...
  printf ( "DialogThumbnail::chapterChanged <%p>\n", pListBoxItem );
  if ( ! pListBoxItem )
    return;

  QString qsChapter = pListBoxItem->text ( );
  QIconViewItem *pItem = m_pIconViewThumbnails->firstItem ( );
  
  while ( pItem ) {
    if  ( getChapter ( pItem ) == qsChapter ) {
      m_pIconViewThumbnails->clearSelection    ( );
      m_pIconViewThumbnails->ensureItemVisible ( pItem );
      pItem->setSelected    ( true );
    }
    pItem = pItem->nextItem ( );
  }
}
*/

void DialogThumbnail::slotReload ( )
{
  // Here we request the whole enchilada again ...
  refresh ( true );
}

void DialogThumbnail::slotRefresh ( )
{
  refresh ( false );
}

void DialogThumbnail::refresh ( bool bForceReload )
{
  bool bOkay;

  m_pProgressBar->reset ( );
  m_pIconViewThumbnails->clear ( );
  m_iCurrentChapter = 0;
  m_iProgress       = 0;

  if ( ( ! m_pSourceFileInfo ) || ( m_pSourceFileInfo->listChapters.count ( ) < 1 ) )
    return;

  m_iProgress = m_pSourceFileInfo->listChapters.count ( );
  // First we should remove outstanding tasks ...
  MediaCreator::unregisterFromMediaScanner ( this );

  m_iThumbnailWidth = m_pEditWidth->text ( ).toInt ( &bOkay );
  if ( ! bOkay )
    m_iThumbnailWidth = 120;

  m_iThumbnailHeight = m_pEditHeight->text ( ).toInt ( &bOkay );
  if ( ! bOkay )
    m_iThumbnailHeight = 120;

  m_iThumbnailStretchMode = m_pComboStretch->currentItem ( );

  while ( m_iCurrentChapter < m_pSourceFileInfo->listChapters.count ( ) ) {
    if  ( bForceReload || getBufferedThumbnail ( ) ) {
      ThumbnailRequest *pRequest = NULL;
      //pInfo->pPreview = new QImage (QImage().fromMimeSource( "please_wait.jpg" ));

      Utils theUtils;  
      long  iMSecOffset = theUtils.getMsFromString ( m_pSourceFileInfo->listChapters[ m_iCurrentChapter ] );
      QIconViewItem *pItem = createCurrentItem ( true );

      pRequest = new ThumbnailRequest ( this, pItem, m_pSourceFileInfo->qsFileName, iMSecOffset );
      MediaCreator::registerWithMediaScanner ( pRequest );
      m_iProgress --;       // for every requeste thumbnail we decrease the progress.
      m_iCurrentChapter ++; // Also every requested thumbnail should increase the currentCHapter counter.
    }
  }
  m_pProgressBar->setProgress ( m_iProgress );
  int iTextType = m_pComboTextType->currentItem ( );
  slotTextType ( iTextType );
}

QIconViewItem *DialogThumbnail::createCurrentItem ( bool bPleaseWait )
{
  QIconViewItem *pItem = NULL;
  QString qsChapter;
  QImage  theImage;

  QImage::ScaleMode scaleMode = QImage::ScaleMin;
  if ( m_iThumbnailStretchMode ==  1 )
    scaleMode = QImage::ScaleMax;
  if ( m_iThumbnailStretchMode ==  2 )
    scaleMode = QImage::ScaleFree;
  
  if ( bPleaseWait )
    theImage = QImage ( QImage ( ).fromMimeSource ( "please_wait.jpg" ) );
  else {
    Utils     theUtils;
    QString   qsFileName, qsBaseName;
    QFileInfo fileInfo ( m_pSourceFileInfo->qsFileName );
    qsBaseName = theUtils.getTempPath ( fileInfo.baseName ( ) );
    qsChapter  = m_pSourceFileInfo->listChapters [ m_iCurrentChapter ];
    qsChapter  = qsChapter.remove  ( ':' );
    qsChapter  = qsChapter.replace ( '.', "_" );
    qsFileName = qsBaseName + QString ( "/thumb_%1.png" ).arg ( qsChapter );
    if ( ! theImage.load ( qsFileName ) )
      return NULL;

    if ( theImage.isNull ( ) )
      return NULL;
  }
  theImage = theImage.smoothScale ( m_iThumbnailWidth, m_iThumbnailHeight, scaleMode );

  qsChapter  = m_pSourceFileInfo->listChapters [ m_iCurrentChapter ];
  pItem = new QIconViewItem ( m_pIconViewThumbnails, qsChapter );
  if ( pItem ) {
    QPixmap thePixmap;
    if ( thePixmap.convertFromImage ( theImage ) ) {
      pItem->setPixmap ( thePixmap );
      pItem->setKey    ( m_qsActionString + qsChapter );
    }
  }
  return pItem;
}
