/****************************************************************************
** Class Utils implementation ...
**
**   Created : Wed Jun 17 07:53:05 2008
**        by : Varol Okan
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
** Here we have some misc functions which are needed by a few classes 
** but can not really be assigned to any of those classes.
**
****************************************************************************/

#include <QColor>
#include <QPainter>
#include <QFileInfo>

#include "utils.h"
#include "render.h"
#include "filter.h"
#include "slideshow.h"

namespace Render
{

Slideshow::Slideshow ( Render::Render *pParent )
{
  m_pParent = pParent;
  m_size = QSize ( 720, 480 ); // NTSC
}

Slideshow::Slideshow ( )
{
  m_pParent = NULL;
  m_size    = QSize ( 720, 480 ); // NTSC
}

Slideshow::~Slideshow ( )
{
}

void Slideshow::initMe ( CXmlSlideshow *pSlideshow, QString &qsTempPath )
{
  m_pSlideshow = pSlideshow;
  m_qsTempPath = qsTempPath;
  QString qsFileName = pSlideshow->slideshow_name;
  qsFileName.replace ( "/", "_" );
  m_qsFileName = m_qsTempPath + qsFileName + ".vob";
}

void Slideshow::setFileName ( QString &qsFileName )
{
  if ( qsFileName.isEmpty ( ) )  {
    QString qsFile = m_pSlideshow->slideshow_name;
    qsFile.replace ( "/", "_" );
    m_qsFileName = m_qsTempPath + qsFile + ".vob";
  }
  else
    m_qsFileName = qsFileName;
}

void Slideshow::displayProgress ( float fProgress )
{
  if ( m_pParent )
    m_pParent->sendProgress ( fProgress );
  else {
    printf ( "." );
    if ( fProgress >= 100.0f )
      printf ( "\n" );
    fflush ( stdout );
  }
}

QString Slideshow::getHashName ( QString qsInput )
{
  if ( m_pParent )
    return m_pParent->getHashName ( qsInput );
  return qsInput;
}

bool Slideshow::exec ( )
{
  if ( ! m_pSlideshow )
    return false;

  // The slideshow has to start with something
  int iX = m_pSlideshow->xres;
  int iY = m_pSlideshow->yres;
  if ( iX < 10 )
       iX = 720;
  if ( iY < 0 )
       iY = 480;
  m_size = QSize ( iX, iY );

  double fFPS = 29.97;
  Encoder::Encoder::enVideo enFormat = Encoder::Encoder::vfNTSC;
  if ( ( iX == 720) && ( iY == 576 ) )
    enFormat = Encoder::Encoder::vfPAL;
  else if ( ( iX == 704 ) && ( iY == 576 ) )
    enFormat = Encoder::Encoder::vfPAL;
  else if ( ( iX == 352 ) && ( iY == 576 ) )
    enFormat = Encoder::Encoder::vfPAL;
  else if ( ( iX == 352 ) && ( iY == 288 ) )
    enFormat = Encoder::Encoder::vfPAL;
  if ( enFormat == Encoder::Encoder::vfPAL )
    fFPS = 25.0;

  CXmlSlideshow::img_struct  *pXmlStart = createStartEnd (  true );
  CXmlSlideshow::img_struct  *pXmlEnd   = createStartEnd ( false );
  CXmlSlideshow::img_struct  *pXmlCurrent, *pXmlPrevious;
  CXmlSlideshow::time_object *pTimeObject;

  createBackground ( );

  Encoder::FFmpeg theEncoder;
  theEncoder.setAudioList ( m_pSlideshow->audio_list );
  theEncoder.initStream   ( m_qsFileName, enFormat, Encoder::Encoder::afAC3, 1000 );

  pXmlPrevious     = pXmlStart;
  uint t, iCount   = m_pSlideshow->count ( );
  uint iAudioCount = m_pSlideshow->audio_list.count ( );
  float fProgress  = 0.0f;
  float fCount2    = 2.0f * ( iCount + iAudioCount ) + 2.0f;
  int iProgress    = 0;
  int iStart       = 0;

  if ( ! m_pSlideshow->intro_page )  {
    // The user chose to skip the intro page with the slideshow title
    iStart        = 1;
    pXmlPrevious  = m_pSlideshow->getImg ( 0 );
    fCount2      -= 2.0f;
  }

  for  ( t=iStart; t<iCount+1; t++ )  {
    if ( m_pParent && m_pParent->killClient ( ) )  {
      // The user terminated this slideshow generation
      theEncoder.endStream ( );
      delete pXmlStart;
      delete pXmlEnd;
      return false;
    }
    if ( t<iCount )
      pTimeObject = m_pSlideshow->getTimeObject ( t );
    else
      pTimeObject = (CXmlSlideshow::time_object *)pXmlEnd;
    // The frame is to be displayed for a bit ...
    if ( pXmlPrevious->node_name == "img" )
      createFromImage ( pXmlPrevious, &theEncoder, fFPS );
    else if ( pXmlPrevious->node_name == "vid" )
      createFromVid   ( (CXmlSlideshow::vid_struct *)pXmlPrevious, &theEncoder, fFPS );
    else
      continue;

    pXmlCurrent = (CXmlSlideshow::img_struct *)pTimeObject;
    fProgress = 100.0f * ( (float)iProgress++ / fCount2 );
    displayProgress ( fProgress );
    if ( ( pXmlPrevious->node_name == "img" ) && 
         ( pXmlCurrent ->node_name == "img" ) )
      createIITransition ( pXmlPrevious, pXmlCurrent, &theEncoder, fFPS );
    else if ( ( pXmlPrevious->node_name == "vid" ) && 
              ( pXmlCurrent ->node_name == "img" ) )
      createVITransition ( (CXmlSlideshow::vid_struct *)pXmlPrevious, pXmlCurrent, &theEncoder, fFPS );
    else if ( ( pXmlPrevious->node_name == "img" ) && 
              ( pXmlCurrent ->node_name == "vid" ) )
      createIVTransition ( pXmlPrevious, (CXmlSlideshow::vid_struct *)pXmlCurrent, &theEncoder, fFPS );
    else if ( ( pXmlPrevious->node_name == "vid" ) && 
              ( pXmlCurrent ->node_name == "vid" ) )
      createVVTransition ( (CXmlSlideshow::vid_struct *)pXmlPrevious, (CXmlSlideshow::vid_struct *)pXmlCurrent, &theEncoder, fFPS );

    fProgress = 100.0f * ( (float)iProgress++ / fCount2 );
    displayProgress ( fProgress );

    pXmlPrevious = pXmlCurrent;
  }
  // The frame is to be displayed for a bit ...
//printf ( "Encoding               <%s>\n", (const char *)pXmlPrevious->src.toUtf8 ( ) );
  createFromImage  ( pXmlPrevious, &theEncoder, fFPS );
  fProgress = 100.0f * ( (float)iProgress++ / fCount2 );
  displayProgress ( fProgress );
//printf ( "Encoding Transition to <%s>\n", (const char *)pXmlEnd->src.toUtf8 ( ) );
  if ( pXmlPrevious->node_name == "img" )
    createIITransition ( pXmlPrevious, pXmlEnd, &theEncoder, fFPS );
  else
    createVITransition ( (CXmlSlideshow::vid_struct *)pXmlPrevious, pXmlEnd, &theEncoder, fFPS );

  displayProgress ( 100.0f );
//  displayProgress ( 100.0f );

  theEncoder.endStream  ( );
  delete pXmlStart;
  delete pXmlEnd;
  return true;
}

void Slideshow::createBackground ( )
{
  m_background = QImage ( m_size, QImage::Format_ARGB32_Premultiplied );
  QString qsBackground = m_pSlideshow->background;
  bool bTryImageBackground = true;
  if ( ! qsBackground.isEmpty ( ) )  {
    if ( qsBackground[0] != '/' ) {
      QColor theColor ( qsBackground );
      if ( theColor.isValid ( ) )  {
        m_background.fill   ( theColor.rgba ( ) );
        bTryImageBackground = false;
      }
    }
    if ( bTryImageBackground )  {
      QFileInfo fileInfo ( qsBackground );
      if ( fileInfo.exists ( ) )  {
        QPainter thePainter;
        QImage img ( qsBackground );
        img  = img.scaled ( m_size, Qt::IgnoreAspectRatio, Qt::SmoothTransformation );
        thePainter.begin  ( &m_background );
        thePainter.drawImage ( 0, 0, img, 0, 0 );
        thePainter.end ( );
      }
      else
        m_background.fill ( 0xFF000000 );
    }
  }
  else
    m_background.fill ( 0xFF000000 );
}

CXmlSlideshow::img_struct *Slideshow::createStartEnd ( bool bStart )
{
  // This frunction creates the first frame and the last frame.
  int iX = m_size.width  ( );
  int iY = m_size.height ( );

  QFont theFont       ( "Helvetica [Cronyx]", 24, QFont::Bold );
  QImage canvas       (    m_size, QImage::Format_ARGB32_Premultiplied );
  QPainter thePainter (   &canvas );
  thePainter.setPen   ( QColor ( "#AFAF15" ) ); // Yellow
  thePainter.setBrush ( QColor ( "#000000" ) );   // Qt::blue );
  thePainter.drawRect ( 0,0,iX,iY );
  thePainter.setFont  (   theFont );

  QString qsFileName = m_pSlideshow->slideshow_name;
  qsFileName.replace ( "/", "_" );
  if ( bStart )  {
    thePainter.drawText ( iX / 3.0, iY / 2.0 - 10, m_pSlideshow->slideshow_name );
    qsFileName = m_qsTempPath + qsFileName + "-start.jpg";
  }
  else
    qsFileName = m_qsTempPath + qsFileName + "-end.jpg";

  // Note this is the same algol as in Render::fileFromSocket
  QString qsNewFileName = getHashName ( qsFileName );
  // Store the hash file name so we can access
  canvas.save ( qsNewFileName, "JPEG", 100 );

  CXmlSlideshow::img_struct *pXml = new CXmlSlideshow::img_struct;
  pXml->src    = qsFileName;
  pXml->width  = iX;
  pXml->height = iY;
  return pXml;
}

void Slideshow::createFromImage ( CXmlSlideshow::img_struct *pXmlImage, Encoder::FFmpeg *pEncoder, double fFPS )
{
  if ( ! pXmlImage )
    return;

  QPainter thePainter;
  QString  qsFileName = getHashName ( pXmlImage->src );
  QImage   canvas   ( m_size, QImage::Format_ARGB32_Premultiplied );
  QImage   imgStart ( qsFileName );

  if ( pXmlImage->pMatrix )
       imgStart = imgStart.transformed ( *pXmlImage->pMatrix, Qt::SmoothTransformation );
  imgStart = imgStart.scaled ( m_size, Qt::KeepAspectRatio );

  int iX, iY;
  iX = (int)( ( m_size.width  ( ) - imgStart.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStart.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStart, 0, 0 );
  thePainter.end ( );

  float fDuration = pXmlImage->fDuration;
  if (  fDuration < 0.0f )
        fDuration = (float)m_pSlideshow->delay;
  if (  fDuration < 0.0f )
        fDuration = 5.0f;
  fDuration *= 1000.0f; // in mSeconds please.

  // And at the end we will add this to the VOB compliant video TS
  int iNumberOfImages = (int)( fFPS * fDuration / 1000.0f );
  pEncoder->addImage ( &canvas, iNumberOfImages );
}

void Slideshow::createFromVid ( CXmlSlideshow::vid_struct *pXmlVideo, Encoder::FFmpeg *pEncoder, double ) // fFPS )
{
  if ( ! pXmlVideo )
    return;

  QString qsFileName = getHashName ( pXmlVideo->src );
  // I do not think we'll be able to support Matrix operations. ( pXmlVideo->pMatrix )
  pEncoder->addVid ( qsFileName, m_size.width ( ), m_size.height ( ) );
}

// Image to Image Transition
void Slideshow::createIITransition ( CXmlSlideshow::img_struct *pStart, CXmlSlideshow::img_struct *pStop, Encoder::FFmpeg *pEncoder, double fFPS )
{
  if ( ! pStart || ! pStop )
    return;

  QPainter thePainter;
  QString  qsFileName = getHashName ( pStart->src );
  QImage   canvas   ( m_size, QImage::Format_ARGB32_Premultiplied );
  QImage   imgStart ( qsFileName );

  if ( pStart->pMatrix )
       imgStart = imgStart.transformed ( *pStart->pMatrix, Qt::SmoothTransformation );
  imgStart = imgStart.scaled ( m_size, Qt::KeepAspectRatio );

  int iX, iY;
  iX = (int)( ( m_size.width  ( ) - imgStart.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStart.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStart, 0, 0 );
  thePainter.end ( );
  imgStart = canvas;

  qsFileName = getHashName ( pStop->src );
  QImage imgStop ( qsFileName );

  if ( pStop->pMatrix )
       imgStop = imgStop.transformed ( *pStop->pMatrix, Qt::SmoothTransformation );
  imgStop = imgStop.scaled ( m_size, Qt::KeepAspectRatio );

  iX = (int)( ( m_size.width  ( ) - imgStop.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStop.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStop, 0, 0 );
  thePainter.end   ( );
  imgStop = canvas;

  float fDuration = (float)m_pSlideshow->filter_delay;
  if  ( fDuration < 0.0f )
        fDuration = pStop->fDuration;
  if  ( fDuration < 0.0f )
        fDuration = 3.0f;
  fDuration *= 1000.0f; // in mSeconds please.

  int iNumberOfImages = (int)( fFPS * fDuration / 1000.0f );
  if ( m_pParent && m_pParent->killClient ( ) )
    return;

  Filter *pFilter = Filter::create ( m_pSlideshow, pStop->pTransition );
  if ( pFilter  )  {
    if ( m_pSlideshow->imgBkgImg ( ) )
       pFilter->exec ( pEncoder, imgStart, imgStop, iNumberOfImages, Filter::VolumeFull, &m_background );
    else
       pFilter->exec ( pEncoder, imgStart, imgStop, iNumberOfImages, Filter::VolumeFull );
    delete pFilter;
  }
}

// Image to Vid transition
void Slideshow::createIVTransition ( CXmlSlideshow::img_struct *pStart, CXmlSlideshow::vid_struct *pStop, Encoder::FFmpeg *pEncoder, double fFPS )
{
  if ( ! pStart || ! pStop )
    return;
  // Here we have to 
  // o Get the first frame of the Video
  // o turn down the background music during the duration of the transition
  // o Transition over
  QPainter thePainter;
  QString  qsFileName = getHashName ( pStart->src );
  QImage   canvas   ( m_size, QImage::Format_ARGB32_Premultiplied );
  QImage   imgStart ( qsFileName );

  if ( pStart->pMatrix )
       imgStart = imgStart.transformed ( *pStart->pMatrix, Qt::SmoothTransformation );
  imgStart = imgStart.scaled ( m_size, Qt::KeepAspectRatio );

  int iX, iY;
  iX = (int)( ( m_size.width  ( ) - imgStart.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStart.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStart, 0, 0 );
  thePainter.end ( );
  imgStart = canvas;

  qsFileName = getHashName ( pStop->src );
  QImage imgStop;
  if ( pEncoder->initVid   ( qsFileName ) )
     imgStop = pEncoder->getVidFrame( 0.0 );
  if ( imgStop.isNull ( ) )
     imgStop = imgStart;

//  m_vid.setSWScale ( m_size.width ( ), m_size.height ( ) );
  // We need a image with transparency ...
  imgStop = imgStop.convertToFormat ( QImage::Format_ARGB32 );//_Premultiplied );

  if ( pStop->pMatrix )
       imgStop = imgStop.transformed ( *pStop->pMatrix, Qt::SmoothTransformation );

  imgStop = imgStop.scaled ( m_size, Qt::KeepAspectRatio );

  iX = (int)( ( m_size.width  ( ) - imgStop.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStop.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStop, 0, 0 );
  thePainter.end   ( );

  imgStop = canvas;

  float fDuration = (float)m_pSlideshow->filter_delay;
  if  ( fDuration < 0.0f )
        fDuration = pStop->fDuration;
  if  ( fDuration < 0.0f )
        fDuration = 3.0f;
  fDuration *= 1000.0f; // in mSeconds please.

  int iNumberOfImages = (int)( fFPS * fDuration / 1000.0f );
  if ( m_pParent && m_pParent->killClient ( ) )
    return;

  Filter *pFilter = Filter::create ( m_pSlideshow, pStop->pTransition );
  if ( pFilter  )  {
    if ( m_pSlideshow->imgBkgImg ( ) )
       pFilter->exec ( pEncoder, imgStart, imgStop, iNumberOfImages, Filter::VolumeFull, &m_background );
    else
       pFilter->exec ( pEncoder, imgStart, imgStop, iNumberOfImages, Filter::VolumeFull );
    delete pFilter;
  }
}

// Video To Image transition
void Slideshow::createVITransition ( CXmlSlideshow::vid_struct *pStart, CXmlSlideshow::img_struct *pStop, Encoder::FFmpeg *pEncoder, double fFPS )
{
  if ( ! pStart || ! pStop )
    return;
  // Here we have to 
  // o Get the last frame of the Video
  // o turn up the background music during the duration of the transition
  // o Transition over
  QPainter thePainter;
  QString  qsFileName = getHashName ( pStart->src );
  QImage   canvas   ( m_size, QImage::Format_ARGB32_Premultiplied );
  QImage   imgStart = pEncoder->getVidFrame ( -1.0 );
//  // In case something went bad ...
  if ( imgStart.isNull ( ) )
       imgStart = QImage ( qsFileName );

  // Don't forget to close the current Vid stream in the encoder.
  pEncoder->closeVid ( );

  if ( imgStart.isNull ( ) )
    return;

//  imgStart = QImage ( "/tmp/z.jpg" );
  if ( pStart->pMatrix )
       imgStart = imgStart.transformed ( *pStart->pMatrix, Qt::SmoothTransformation );
  imgStart = imgStart.scaled ( m_size, Qt::KeepAspectRatio );

  int iX, iY;
  iX = (int)( ( m_size.width  ( ) - imgStart.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStart.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStart, 0, 0 );
  thePainter.end ( );
  imgStart = canvas;

  qsFileName = getHashName ( pStop->src );
  QImage imgStop ( qsFileName );
  if ( imgStop.isNull ( ) )
       imgStop = imgStart;

  if ( pStop->pMatrix )
       imgStop = imgStop.transformed ( *pStop->pMatrix, Qt::SmoothTransformation );
  imgStop = imgStop.scaled ( m_size, Qt::KeepAspectRatio );

  iX = (int)( ( m_size.width  ( ) - imgStop.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStop.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStop, 0, 0 );
  thePainter.end   ( );
  imgStop = canvas;

  float fDuration = (float)m_pSlideshow->filter_delay;
  if  ( fDuration < 0.0f )
        fDuration = pStop->fDuration;
  if  ( fDuration < 0.0f )
        fDuration = 3.0f;
  fDuration *= 1000.0f; // in mSeconds please.

  int iNumberOfImages = (int)( fFPS * fDuration / 1000.0f );
  if ( m_pParent && m_pParent->killClient ( ) )
    return;

  Filter *pFilter = Filter::create ( m_pSlideshow, pStop->pTransition );
  if ( pFilter  )  {
    if ( m_pSlideshow->imgBkgImg ( ) )
       pFilter->exec ( pEncoder, imgStart, imgStop, iNumberOfImages, Filter::VolumeFull, &m_background );
    else
       pFilter->exec ( pEncoder, imgStart, imgStop, iNumberOfImages, Filter::VolumeFull );
    delete pFilter;
  }
}

// Vid to Vid transition
void Slideshow::createVVTransition ( CXmlSlideshow::vid_struct *pStart, CXmlSlideshow::vid_struct *pStop, Encoder::FFmpeg *pEncoder, double fFPS )
{
  if ( ! pStart || ! pStop )
    return;
  // Here we have to 
  // o Get the last frame of the Video
  // o turn up the background music during the duration of the transition
  // o Transition over
  QPainter thePainter;
  QString  qsFileName = getHashName ( pStart->src );
  QImage   canvas   ( m_size, QImage::Format_ARGB32_Premultiplied );
  QImage   imgStart = pEncoder->getVidFrame ( -1.0 );
//  // In case something went bad ...
  if ( imgStart.isNull ( ) )
       imgStart = QImage ( qsFileName );

  // Don't forget to close the current Vid stream in the encoder.
  pEncoder->closeVid ( );

  if ( imgStart.isNull ( ) )
    return;

//  imgStart = QImage ( "/tmp/z.jpg" );
  if ( pStart->pMatrix )
       imgStart = imgStart.transformed ( *pStart->pMatrix, Qt::SmoothTransformation );
  imgStart = imgStart.scaled ( m_size, Qt::KeepAspectRatio );

  int iX, iY;
  iX = (int)( ( m_size.width  ( ) - imgStart.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStart.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStart, 0, 0 );
  thePainter.end ( );
  imgStart = canvas;

  qsFileName = getHashName ( pStop->src );
  QImage imgStop;
  if ( pEncoder->initVid   ( qsFileName ) )
     imgStop = pEncoder->getVidFrame( 0.0 );
  if ( imgStop.isNull ( ) )
     imgStop = imgStart;

  // We need a image with transparency ...
  imgStop = imgStop.convertToFormat ( QImage::Format_ARGB32 );//_Premultiplied );
  imgStop = imgStop.scaled ( m_size, Qt::KeepAspectRatio );

  iX = (int)( ( m_size.width  ( ) - imgStop.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStop.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStop, 0, 0 );
  thePainter.end   ( );
  imgStop = canvas;

  float fDuration = (float)m_pSlideshow->filter_delay;
  if  ( fDuration < 0.0f )
        fDuration = pStop->fDuration;
  if  ( fDuration < 0.0f )
        fDuration = 3.0f;
  fDuration *= 1000.0f; // in mSeconds please.

  int iNumberOfImages = (int)( fFPS * fDuration / 1000.0f );
  if ( m_pParent && m_pParent->killClient ( ) )
    return;

  Filter *pFilter = Filter::create ( m_pSlideshow, pStop->pTransition );
  if ( pFilter  )  {
    if ( m_pSlideshow->imgBkgImg ( ) )
       pFilter->exec ( pEncoder, imgStart, imgStop, iNumberOfImages, Filter::VolumeFull, &m_background );
    else
       pFilter->exec ( pEncoder, imgStart, imgStop, iNumberOfImages, Filter::VolumeFull );
    delete pFilter;
  }
}

}; // end of namespace Render
