#include "rpggame.h"

VARP(firemethod, 0, 1, 1);
VARP(friendlyfire, 0, 0, 1);

void projectile::init(rpgchar *d, equipment *w, equipment *a, int flags, float speed)
{
	if(DEBUG_PROJ)
		conoutf(CON_DEBUG, "DEBUG: Creating projectile; Owner %p item %p flags %i speed %f ...", d, w, flags, speed);

	dir = vec(
		RAD * (d->yaw + (flags & INIT_FUZZY ? (rnd(6001) - 3000) / 100.0f : 0 )),
		RAD * (d->pitch + (flags & INIT_FUZZY ? (rnd(3001) - 1500) / 100.0f : 0))
	);
	o = d->o;

	//we now need to move the projectile outside the player's bounding box
	vec test = vec(dir).mul(max(d->eyeheight, d->radius) * 4); test.add(o);

	float mult = 0;
	game::intersect(d, test, o, mult);

	mult = (1 - mult ) + 0.0001;
	mult *= test.dist(o);

	//finally move it just outside the bounding box
	o.add(vec(dir).mul(mult));

	//initialise the other stuff
	owner = d;
	emitpos = o;
	lastemit = lastmillis;

	if(DEBUG_PROJ)
		conoutf(CON_DEBUG, "DEBUG: owner BB collision point: %f %f %f ...", o.x, o.y, o.z);

	if(w)
	{
		item = w->base;
		use = w->use;
		use_weapon *wep = (use_weapon *) game::items[w->base]->uses[w->use];

		fx = wep->effect;
		dist = wep->range;
		time = wep->lifetime;
		pflags = wep->flags;
		elasticity = wep->elasticity;
		radius = wep->radius;
		dir.mul(wep->speed);


		if(DEBUG_PROJ)
			conoutf(CON_DEBUG, "DEBUG: weapon was provided... fx: %i dist: %i time: %i pflags: %i elasticity %f radius: %i speed %f", fx, dist, time, pflags, elasticity, radius, dir.magnitude());
	}
	if(a)
	{
		ammo = a->base;
		ause = a->use;
		use_weapon *wep = (use_weapon *) game::items[a->base]->uses[a->use];

		fx = wep->effect; //visuals take precedence
		dist += wep->range;
		time += wep->lifetime;
		pflags |= wep->flags;
		elasticity = max(elasticity, wep->elasticity);
		radius += wep->radius;
		dir.mul(wep->speed);

		if(DEBUG_PROJ)
			conoutf(CON_DEBUG, "DEBUG: ammo was provided... fx: %i dist: %i time: %i pflags: %i elasticity %f radius: %i speed %f", fx, dist, time, pflags, elasticity, radius, dir.magnitude());
	}

	if(!w && !a)
	{
		dir.mul(speed);
	}

	if(firemethod || flags & INIT_ADJUST)
	{ //add the actor's velocity, but adjust for it so it doesn't deviate from the intended target
		vec vel(d->vel);
		vel.add(d->falling);
		vel.div(100.0f);
		vel.add(dir);

		dir.normalize();
		dir.mul(vel.magnitude());

		if(DEBUG_PROJ)
			conoutf(CON_DEBUG, "DEBUG: compensating for owner velocity, dir: (%f, %f, %f) speed: %f", dir.x, dir.y, dir.z, dir.magnitude());
	}
	else
	{
		//just add the actor's velocity, no adjusting
		vec vel(d->vel);
		vel.add(d->falling);
		vel.div(100.0f);
		dir.add(vel);

		if(DEBUG_PROJ)
			conoutf(CON_DEBUG, "DEBUG: adding owner velocity, dir: (%f, %f, %f) speed: %f", dir.x, dir.y, dir.z, dir.magnitude());
	}

}

VARP(maxricochets, 1, 5, 100);
vec normal;

bool projectile::update()
{
	if(deleted)
		return false;

	rpgent *victim = NULL;

	if(pflags & P_TIME)
		time -= curtime;

	if(!(pflags & P_STATIONARY))
	{
		float distance = 100.0f * dir.magnitude() * curtime / 1000.0f;
		int tries = 0;

		while(!deleted && !victim && distance > 0 && tries < maxricochets)
		{
			victim = travel(distance);

			if(!victim && distance > 0 && pflags & P_RICOCHET)
			{
				//offset slightly
				o.add(vec(normal).mul(0.1));

				//reflect around normal NOTE: reflect maintains magnitude
				dir.reflect(normal);
				dir.mul(elasticity);
				distance *= elasticity;

				//freeze the projectile when it gets too slow
				if(
					(!gravity && dir.magnitude() <= 1e-3f) ||
					(gravity && dir.magnitude2() <= 1e-3f && abs(dir.z) <= 0.1 &&
						1 >= raycube(o, vec(0, 0, -gravity).normalize(), 2, RAY_ALPHAPOLY|RAY_CLIPMAT)
					)
				)
				{
					if(DEBUG_PROJ)
						conoutf(CON_DEBUG, "DEBUG: Freezing slow projectile %p", this);

					if((pflags & P_VOLATILE || pflags & P_DIST) && !(pflags & P_TIME || pflags & P_PROXIMITY))
					{
						if(DEBUG_PROJ)
							conoutf(CON_DEBUG, "DEBUG: adjusting frozen projectile to enable self destruct %p", this);

						if(pflags & P_VOLATILE)
							deleted = true;
						else
							dist = 0;
					}
					else if(pflags == P_RICOCHET)
					{
						if(DEBUG_PROJ)
							conoutf(CON_DEBUG, "DEBUG: adding proximity trigger to frozen pure ricochet projectile %p", this);
						pflags |= P_PROXIMITY;
					}

					pflags |= P_STATIONARY;
					break;
				}
			}
			else if(victim || distance > 0)
				deleted = true;

			tries++;
		}

		if(gravity && 1 <= raycube(o, vec(0, 0, -gravity).normalize(), 2, RAY_ALPHAPOLY|RAY_CLIPMAT))
			dir.z -= (curtime * gravity) / 100000.0f;
	}

	//entity is outside world, delete
	if(o.x < 0 || o.x >= getworldsize() ||
	   o.y < 0 || o.y >= getworldsize() ||
	   o.z < 0 || o.z >= getworldsize()
	)
	{
		if(DEBUG_PROJ)
			conoutf(CON_DEBUG, "DEBUG: projectile %p out of world - deleting", this);
		return false;
	}

	//time has run out and the projecitle hence has to either explode or fizzle out
	if(!victim && ((pflags & P_TIME && time <= 0) || (pflags & P_DIST && dist <= 0)))
	{
		//just fizzle out - it's not volatile and no one was hurt
		if(!(pflags & P_VOLATILE))
		{
			if(DEBUG_PROJ)
				conoutf(CON_DEBUG, "DEBUG: Deleting expired non-volatile projectile %p", this);

			return false;
		}

		if(DEBUG_PROJ)
			conoutf(CON_DEBUG, "DEBUG: volatile projectile expired %p", this);

		deleted = true;
	}
	else if(!victim && pflags & P_PROXIMITY)
	{
		loopv(game::curmap->objs)
		{
			rpgent *d = game::curmap->objs[i];
			if(d == owner)
				continue;

			if( radius >= o.dist_to_bb<vec>(d->feetpos(), d->o) - d->radius )
			{
				if(DEBUG_PROJ)
					conoutf(CON_DEBUG, "DEBUG: creature %p entered vicinity of %p", d, this);

				deleted = true;
				victim= d;
				break;
			}
		}
	}

	if(deleted)
	{
		drawdeath();

		if(game::items.inrange(item) && game::items[item]->uses.inrange(use))
		{
			if(DEBUG_PROJ)
				conoutf(CON_DEBUG, "DEBUG: projectile %p has an associated item, applying to victim(s)...", this);
			use_weapon *wep = (use_weapon *) game::items[item]->uses[use];
			use_weapon *amm = (use_weapon *) ((ammo  >= 0 && ause >= 0) ? game::items[ammo]->uses[ause] : NULL);

			switch(ammo >= 0 ? amm->target : wep->target)
			{
				case T_SINGLE:
					if(victim)
					{
						if(DEBUG_PROJ) conoutf(CON_DEBUG, "DEBUG: conferring status effect to victim %p", victim);
						victim->hit(owner, wep, amm, charge, vec(dir).normalize());
					}
					break;
				case T_MULTI:
					loopv(game::curmap->objs)
					{
						rpgent *d = game::curmap->objs[i];
						if(d == owner && !friendlyfire) continue;

						if(radius >= o.dist_to_bb<vec>(d->feetpos(), d->o)  - d->radius)
						{
							if(DEBUG_PROJ)
								conoutf(CON_DEBUG, "Conferring group status effect to %p", d);
							vec kick = vec(d->o).sub(o).normalize();
							d->hit(owner, wep, amm, charge, kick);
						}
					}
					break;
				case T_AREA:
				{
					areaeffect *wepeffect = game::curmap->aeffects.add(new areaeffect());
					areaeffect *ammeffect = game::curmap->aeffects.add(new areaeffect());
					wepeffect->owner = ammeffect->owner = owner;
					wepeffect->o = ammeffect->o = o;
					wepeffect->mul = ammeffect->mul = charge;

					wepeffect->duration = wep->duration;
					wepeffect->group = wep->status;
					wepeffect->fx = wep->effect;
					wepeffect->radius = wep->radius;

					ammeffect->duration = amm->duration;
					ammeffect->group = amm->status;
					ammeffect->fx = amm->effect;
					ammeffect->radius = amm->radius;
				}
			}
		}
	}

	return true;
}

rpgent *projectile::travel(float &distance)
{
	vec pos;
	float geomdist = raycubepos(o, vec(dir).normalize(), pos, distance, RAY_CLIPMAT|RAY_ALPHAPOLY);

	extern vec hitsurface;
	normal = hitsurface;

	float entdist = 0;
	rpgent *vic = game::intersectclosest(o, pos, friendlyfire ? NULL : owner, entdist, 2);

	if(!vic || entdist > 1 || vic == owner)
	{
		vic = NULL;

		if(pflags & P_DIST)
			dist -= geomdist;
		distance -= geomdist;
	}
	else
	{
		if(DEBUG_PROJ)
			conoutf(CON_DEBUG, "DEBUG: Victim %p", vic);
		pos.sub(o).mul(entdist).add(o);
		if(pflags & P_DIST)
			dist -= o.dist(pos);
		distance -= o.dist(pos);
	}

	o = pos;

	return vic;
}

VARP(projallowflare, 0, 1, 1);

void projectile::render(bool mainpass)
{
	if(!game::effects.inrange(fx))
	{
		conoutf(CON_ERROR, "ERROR: effect definition %i doesn't exist - deleting", fx);
		deleted = true;
		return;
	}

	effect *e = game::effects[fx];
	if(e->mdl)
	{
		float yaw, pitch;
		if(e->flags & FX_SPIN)
			vectoyawpitch(vec((lastmillis % 360) * RAD, (lastmillis % (360 * 5)) * RAD / 5.0f), yaw, pitch);
		else
			vectoyawpitch(dir, yaw, pitch);

		rendermodel(&light, e->mdl, ANIM_MAPMODEL|ANIM_LOOP, o, yaw, pitch, 0, MDL_SHADOW);
	}
	if(mainpass)
	{
		if(!e->mdl)
			rpgeffect::drawsplash(e, o, 0, charge, rpgeffect::PROJ, 0);

		if(e->flags & FX_TRAIL && !(pflags&P_STATIONARY))
		{
			//bool drawline(effect *e, vec &from, vec &to, float size, int type)
			rpgeffect::drawline(e, emitpos, o, 1, rpgeffect::TRAIL);
		}

		if(projallowflare && e->flags & (FX_FIXEDFLARE|FX_FLARE))
		{
			regularlensflare(o, e->traillightcol.x * 256, e->traillightcol.y * 256, e->traillightcol.z * 256, e->flags&FX_FIXEDFLARE, false, e->traillightradius * (e->flags&FX_FIXEDFLARE ? .5f : 5.0f));
		}
	}
}

void projectile::drawdeath()
{
	if(!game::effects.inrange(fx))
	{
		conoutf(CON_ERROR, "ERROR: effect definition %i doesn't exist", fx);
		return;
	}
	effect *e = game::effects[fx];

	rpgeffect::drawsphere(e, o, radius, 1, rpgeffect::DEATH, 0);
}

void projectile::dynlight()
{
	if(!game::effects.inrange(fx))
	{
		conoutf(CON_ERROR, "ERROR: effect definition %i doesn't exist", fx);
		return;
	}
	effect *e = game::effects[fx];
	if(!e->flags & FX_DYNLIGHT)
		return;

	if(deleted)
	{
		adddynlight(o, e->deathlightradius, e->deathlightcol, e->deathlightfade, e->deathlightfade / 2, e->deathlightflags, e->deathlightinitradius, e->deathlightinitcol);
	}
	else
	{
		adddynlight(o, e->traillightradius, e->traillightcol);
	}
}
