#Module for XAS input
#
#Main Routine returns a data structure with:
#
# self.type = file type:
#           SB = SSRL binary
#           SA = SSRL ASCII
#           DQ = DND Quick-Scan
#           DA = DND ASCII
#           DM = DND Multi-element
#           GC = Generic
# self.rtc = clock normalizing counts
# self.eV 
# self.i0
# self.i1
# self.i2  -- if not present, array of ones
# self.iF  -- will be array if multielement is used
# self.ICR -- data for Ge13's, else array of ones

from Numeric import *
from string import *
from struct import *
import Pmw
from Tkinter import *
import tkSimpleDialog
import tkMessageBox
import os
import xdrlib
import binascii

###################################
##
## SSRL Binary
##
###################################

class ssrlbin:
    global reloadps

    def __init__(self,fn,master,ieee=0):
        global reloadps
        #make dictionary for columns
        fid=open(fn,"rb")
        self.type='SB'
        title=fid.read(40)
        date=fid.read(40)
        if date[0] not in printable:
            self.rtc=array([1.0,1.0,1.0,1.0])
            self.eV=array([1.0,2.0,3.0,4.0])
            self.i0=array([1.0,1.0,1.0,1.0])
            self.i1=array([1.0,1.0,1.0,1.0])
            self.i2=array([1.0,1.0,1.0,1.0])
            self.ICR=array([1.0,1.0,1.0,1.0])
            self.iF=array([1.0,1.0,1.0,1.0])
            self.comment='DID NOT READ - CORRUPT FILE'
            tkMessageBox.showwarning("Open File"," WARNING!\n SSRL binary file is corrupt!")
            return            
        buf=fid.read(40)
        parse=split(buf)
        npts=int(parse[1])
        ncols=int(parse[3])
        scaler=fid.read(40)
        scaler=upper(scaler)
        region=fid.read(40)
        environ=fid.read(80)
        lsinfo=fid.read(40)
        self.comment=fid.read(80*6)
        offs=fromstring(fid.read(4*ncols),Float32)
        wts=fromstring(fid.read(4*ncols),Float32)
        lbs=[]
        i=0
        while i<ncols:
            buf=fid.read(20)
            lbs.append(buf[0:18])
            i=i+1
        #form detector assignments
        detind={}
        ffl=0
        ffst=0
        ffscax=[[0,0],[0,0]]
        for typ in lbs:
            if (count(lower(typ),'time')>0 and count(lower(typ),'lifetime')==0) or count(lower(typ),'clock')>0 or lower(typ[0:3])=='rtc':
                detind.update({'rtc':lbs.index(typ)})
            if count(lower(typ),'requested energy')>0 or count(lower(typ),'requested position')>0:
                detind.update({'ev':lbs.index(typ)})
            if lower(typ[0:3])=='i0 ': detind.update({'i0':lbs.index(typ)})
            if count(lower(typ),'i1')>0: detind.update({'i1':lbs.index(typ)})
            if count(lower(typ),'i2')>0: detind.update({'i2':lbs.index(typ)})
            if count(lower(typ),'lytle')>0 or upper(typ)==ljust('FF',18):
                detind.update({'ifst':lbs.index(typ)})
                detind.update({'ife':lbs.index(typ)})
                ffl=lbs.index(typ)
            if (upper(typ[0:3])=='FF1' or count(upper(typ),'SCA')>0) and ffst==0:
                detind.update({'ifst':lbs.index(typ)})
                ffst=1
            if (upper(typ[0:2])=='FF' or count(upper(typ),'SCA')>0) and ffst==1:
                detind.update({'ife':lbs.index(typ)+1})
            if upper(typ[0:4])=='ICR1' and ffst==1 or upper(typ[0:5])=='ICR:1' and ffst==1:
                detind.update({'icrst':lbs.index(typ)})
                ffst=2
            if upper(typ[0:3])=='ICR' and ffst==2:
                detind.update({'icre':lbs.index(typ)+1})
            if count(upper(typ),'SCA2')>0:
                if ffscax[0][0]==0:ffscax[0][0]=lbs.index(typ)
                ffscax[0][1]=lbs.index(typ)+1
            if count(upper(typ),'SCA3')>0:
                if ffscax[1][0]==0:ffscax[1][0]=lbs.index(typ)
                ffscax[1][1]=lbs.index(typ)+1                
        fid.seek(0,1)
        np=fromstring(fid.read(4*4),Int)
        i=0
        d=zeros((npts,ncols),Float32)
        while i<npts:
            j=0
            while j<ncols:
                buf=fid.read(4)
                if not ieee:
                    new=fromstring(buf[2]+buf[3]+buf[0]+buf[1],Float32)/4  #2301
                else:
                    #new=fromstring(buf[0]+buf[1]+buf[2]+buf[3],Float32)/4
                    new=unpack('f',buf)[0]
                ind=j+ncols*i
                put(d,ind,new)
                j=j+1
            i=i+1
        fid.close()
        #assign arrays
        #first the ones that should always be in all files
        self.eV=d[:,detind['ev']]
        self.rtc=d[:,detind['rtc']]
        self.i0=d[:,detind['i0']]
        #now the optional ones -- fill with ones if not present
        try:
            self.i1=d[:,detind['i1']]
        except:
            self.i1=ones((len(self.i0)),Float32)
        try:    
            self.i2=d[:,detind['i2']]
        except:
            self.i2=ones((len(self.i0)),Float32)
        #and then iF and ICR
        if detind.has_key('ifst'):
            try:
                if (reloadps[1]=='Lytle' or reloadps[1]==None) and reloadps[0]==self.type:
                    detind.update({'ifst':ffl})
                    detind.update({'ife':ffl})
                elif reloadps[1]=='Ge' and reloadps[0]==self.type:
                    if reloadps[2]=='SCA2' and ffscax[0][0]!=0:
                        detind.update({'ifst':ffscax[0][0]})
                        detind.update({'ife':ffscax[0][1]})
                    if reloadps[2]=='SCA3' and ffscax[1][0]!=0:
                        detind.update({'ifst':ffscax[1][0]})
                        detind.update({'ife':ffscax[1][1]})
                else:
                    raise ValueError, ""
            except:
                reloadps=[]
                rsel='Ge'
                select=''
                if ffl!=0 and ffst!=0:
                    #file has BOTH Lytle and Ge channels
                    #ask which...
                    roidialog=Pmw.Dialog(parent=master,title="Detector Selection",buttons=('OK','Apply All'))
                    roiselbox=Pmw.RadioSelect(parent=roidialog.interior(),labelpos='n',label_text="Please select the fluo. channel to load")
                    rbuts=('Lytle','Ge')
                    for rb in rbuts:
                        roiselbox.add(rb)
                    roiselbox.pack(side=TOP,fill='both',expand=1,padx=5,pady=5)
                    select=roidialog.activate()
                    rsel=roiselbox.getvalue()
                    if rsel=='Lytle' or rsel==None:
                        detind.update({'ifst':ffl})
                        detind.update({'ife':ffl})
                    #SCA selection for Ge
                if rsel=='Ge' and ffscax[0][0]!=0:
                    scadialog=Pmw.Dialog(parent=master,title="SCA Selection",buttons=('OK',))
                    scaselbox=Pmw.RadioSelect(parent=scadialog.interior(),labelpos='n',label_text="Please select SCA channel to load")
                    sbuts=('SCA1','SCA2')
                    if ffscax[1][0]!=0:sbuts=('SCA1','SCA2','SCA3')
                    for sb in sbuts:
                        scaselbox.add(sb)
                    scaselbox.pack(side=TOP,fill='both',expand=1,padx=5,pady=5)
                    scaselect=scadialog.activate()
                    ssel=scaselbox.getvalue()
                    if ssel=='SCA2':
                        detind.update({'ifst':ffscax[0][0]})
                        detind.update({'ife':ffscax[0][1]})
                    if ssel=='SCA3':
                        detind.update({'ifst':ffscax[1][0]})
                        detind.update({'ife':ffscax[1][1]})                        
                if select=='Apply All':
                    try:
                        reloadps=[self.type,rsel,ssel]
                    except:
                        reloadps=[self.type,rsel,'SCA1']             
            if detind['ifst']==detind['ife']:
                #lytle/single channel
                self.iF=d[:,detind['ifst']]
                self.ICR=d[:,detind['ifst']]
            else:
                #multi-element
                self.iF=d[:,detind['ifst']:detind['ife']]
                if  detind.has_key('icrst'):
                    self.ICR=d[:,detind['icrst']:detind['icre']]
                else:
                    self.ICR=self.iF
        else:
            self.iF=ones((len(self.i0)),Float32)
            self.ICR=ones((len(self.i0)),Float32)
                        
###################################
##
## SSRL ASCII
##
###################################
    
class ssrlascii:
    global reloadps

    def __init__(self,fn,master):
        global reloadps
        fid=open(fn,"r")
        self.type='SA'
        title=fid.readline()
        date=fid.readline()
        buf=fid.readline()
        parse=split(buf)
        npts=int(parse[1])
        ncols=int(parse[3])
        scaler=fid.readline()
        scaler=upper(scaler)
        region=fid.readline()
        environ=fid.readline()
        lsinfo=fid.readline()
        self.comment=[]
        j=range(6)
        for i in j: 
            buf=fid.readline()
            buf2=buf[:79]
            self.comment.append(buf2)
        buf=fid.readline()
        buf=fid.readline()
        buf=fid.readline()
        wts=split(buf)
        buf=fid.readline()
        buf=fid.readline()
        offs=split(buf)
        buf=fid.readline()
        lbs=[]
        i=0
        while i<ncols:
            buf=fid.readline()
            lbs.append(buf[0:18])
            i=i+1
        #form detector assignments
        detind={}
        ffl=0
        ffst=0
        ffscax=[[0,0],[0,0]]
        for typ in lbs:
            if (count(lower(typ),'time')>0 and count(lower(typ),'lifetime')==0) or count(lower(typ),'clock')>0 or lower(typ[0:3])=='rtc':
                detind.update({'rtc':lbs.index(typ)})
            if count(lower(typ),'requested energy')>0 or count(lower(typ),'requested position')>0:
                detind.update({'ev':lbs.index(typ)})
            if lower(typ[0:3])=='i0 ': detind.update({'i0':lbs.index(typ)})
            if count(lower(typ),'i1')>0: detind.update({'i1':lbs.index(typ)})
            if count(lower(typ),'i2')>0: detind.update({'i2':lbs.index(typ)})
            if count(lower(typ),'lytle')>0 or upper(typ)==ljust('FF',18):
                detind.update({'ifst':lbs.index(typ)})
                detind.update({'ife':lbs.index(typ)})
                ffl=lbs.index(typ)
            if (upper(typ[0:3])=='FF1' or count(upper(typ),'SCA')>0) and ffst==0:
                detind.update({'ifst':lbs.index(typ)})
                ffst=1
            if (upper(typ[0:2])=='FF' or count(upper(typ),'SCA')>0) and ffst==1:
                detind.update({'ife':lbs.index(typ)+1})
            if upper(typ[0:4])=='ICR1' and ffst==1 or upper(typ[0:5])=='ICR:1' and ffst==1:
                detind.update({'icrst':lbs.index(typ)})
                ffst=2
            if upper(typ[0:3])=='ICR' and ffst==2:
                detind.update({'icre':lbs.index(typ)+1})
            if count(upper(typ),'SCA2')>0:
                if ffscax[0][0]==0:ffscax[0][0]=lbs.index(typ)
                ffscax[0][1]=lbs.index(typ)+1
            if count(upper(typ),'SCA3')>0:
                if ffscax[1][0]==0:ffscax[1][0]=lbs.index(typ)
                ffscax[1][1]=lbs.index(typ)+1    
        buf=fid.readline()
        d=zeros((npts,ncols),Float32)        
        j=range(npts)
        k=range(ncols)
        for i in j:
            buf=fid.readline()
            parse=split(buf)
            for m in k:
                ind=m+ncols*i
                put(d,ind,atof(parse[m]))                
        fid.close()            
        #assign arrays
        #first the ones that should always be in all files
        self.eV=d[:,detind['ev']]
        self.rtc=d[:,detind['rtc']]
        self.i0=d[:,detind['i0']]
        #now the optional ones -- fill with ones if not present
        try:
            self.i1=d[:,detind['i1']]
        except:
            self.i1=ones((len(self.i0)),Float32)
        try:    
            self.i2=d[:,detind['i2']]
        except:
            self.i2=ones((len(self.i0)),Float32)
        #and then iF and ICR
        if detind.has_key('ifst'):
            try:
                if (reloadps[1]=='Lytle' or reloadps[1]==None) and reloadps[0]==self.type:
                    detind.update({'ifst':ffl})
                    detind.update({'ife':ffl})
                elif reloadps[1]=='Ge' and reloadps[0]==self.type:
                    if reloadps[2]=='SCA2' and ffscax[0][0]!=0:
                        detind.update({'ifst':ffscax[0][0]})
                        detind.update({'ife':ffscax[0][1]})
                    if reloadps[2]=='SCA3' and ffscax[1][0]!=0:
                        detind.update({'ifst':ffscax[1][0]})
                        detind.update({'ife':ffscax[1][1]})
                else:
                    raise ValueError, ""
            except:
                reloadps=[]
                rsel='Ge'
                select=''                
                if ffl!=0 and ffst!=0:
                    #file has BOTH Lytle and Ge channels
                    #ask which...
                    roidialog=Pmw.Dialog(parent=master,title="Detector Selection",buttons=('OK','Apply All'))
                    roiselbox=Pmw.RadioSelect(parent=roidialog.interior(),labelpos='n',label_text="Please select the fluo. channel to load")
                    rbuts=('Lytle','Ge')
                    for rb in rbuts:
                        roiselbox.add(rb)
                    roiselbox.pack(side=TOP,fill='both',expand=1,padx=5,pady=5)
                    select=roidialog.activate()
                    rsel=roiselbox.getvalue()
                    if rsel=='Lytle' or rsel==None:
                        detind.update({'ifst':ffl})
                        detind.update({'ife':ffl})
                #SCA selection for Ge
                if rsel=='Ge' and ffscax[0][0]!=0:
                    scadialog=Pmw.Dialog(parent=master,title="SCA Selection",buttons=('OK',))
                    scaselbox=Pmw.RadioSelect(parent=scadialog.interior(),labelpos='n',label_text="Please select SCA channel to load")
                    sbuts=('SCA1','SCA2')
                    if ffscax[1][0]!=0:sbuts=('SCA1','SCA2','SCA3')
                    for sb in sbuts:
                        scaselbox.add(sb)
                    scaselbox.pack(side=TOP,fill='both',expand=1,padx=5,pady=5)
                    scaselect=scadialog.activate()
                    ssel=scaselbox.getvalue()
                    if ssel=='SCA2':
                        detind.update({'ifst':ffscax[0][0]})
                        detind.update({'ife':ffscax[0][1]})
                    if ssel=='SCA3':
                        detind.update({'ifst':ffscax[1][0]})
                        detind.update({'ife':ffscax[1][1]})  
                if select=='Apply All':
                    try:
                        reloadps=[self.type,rsel,ssel]
                    except:
                        reloadps=[self.type,rsel,'SCA1']
            if detind['ifst']==detind['ife']:
                #lytle/single channel
                self.iF=d[:,detind['ifst']]
                self.ICR=d[:,detind['ifst']]
            else:
                #multi-element
                self.iF=d[:,detind['ifst']:detind['ife']]
                if  detind.has_key('icrst'):
                    self.ICR=d[:,detind['icrst']:detind['icre']]
                else:
                    self.ICR=self.iF
        else:
            self.iF=ones((len(self.i0)),Float32)
            self.ICR=ones((len(self.i0)),Float32)

###################################
##
## DND-CAT Quick Scan Binary
##
###################################

class dndquick:
    def __init__(self,fn,e,edg):
        fid=open(fn,"rb")
        self.type='DQ'
        size=fromstring(fid.read(4),Int)[0]
        memsize=size*2
        headersize=1024*2
        memsize=memsize-headersize
        size=size-1024
        buf=fid.read(headersize)
        rawi0=fromstring(fid.read(memsize),Int16)
        buf=fid.read(headersize)
        rawiF=fromstring(fid.read(memsize),Int16)
        buf=fid.read(headersize)
        rawiT=fromstring(fid.read(memsize),Int16)
        #retype data
        rawi0=rawi0.astype(Float32)
        rawiT=rawiT.astype(Float32)
        rawiF=rawiF.astype(Float32)        
        dspc=3.1353
        chan=((0,size))
        delE=0.5
        delk=0.05
        startk=2
        #calcualte theta slope
        theta=array((arcsin(12398/(2*dspc*e[0])),arcsin(12398/(2*dspc*e[1]))),Float32)
        slope=(theta[0]-theta[1])/(chan[0]-chan[1])
        b=theta[0]-slope*chan[0]
        #calculate starting parameters
        firstindex=0
        firstenergy=12398/(2*dspc*sin(b))
        lastenergy=firstenergy+delE
        lastangle=arcsin(12398/(2*dspc*lastenergy))
        lastindex=int((lastangle-b)/slope)
        finalindex=len(rawi0)
        #calculate parameters from k range
        kst=startk-delk/2
        kregstart=(kst/0.51233)**2+edg
        curk=startk-delk
        #initialize data arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.iF=[]
        #main binning loop
        while lastindex<finalindex:
            channum=float(lastindex-firstindex+1)
            self.rtc.append(channum)
            #sum bin
            ti0=rawi0[firstindex:lastindex]
            i0sum=float(sum(ti0))
            self.i0.append(i0sum)
            tiT=rawiT[firstindex:lastindex]
            iTsum=float(sum(tiT))
            self.i1.append(iTsum)
            tiF=rawiF[firstindex:lastindex]
            iFsum=float(sum(tiF))
            self.iF.append(iFsum)
            #calculate average locations
            avex=(firstindex+lastindex)/2
            avetheta=slope*avex+b
            aveenergy=float(12398/(2*dspc*sin(avetheta)))
            #update lists
            self.eV.append(aveenergy)
            #calculate next interval
            if (lastenergy<kregstart):
                firstenergy=lastenergy
                lastenergy=firstenergy+delE
            else:
                #in kregions
                curk=curk+delk
                firstenergy=((curk-(delk/2))/0.51233)**2+edg
                lastenergy=((curk+(delk/2))/0.51233)**2+edg
            firstangle=arcsin(12398/(2*dspc*firstenergy))
            firstindex=int((firstangle-b)/slope)
            lastangle=arcsin(12398/(2*dspc*lastenergy))
            lastindex=int((lastangle-b)/slope)
        fid.close()
        #array assignments may not be as "assumed"... read an input file
        #look for file in same directory:
        dirchar=rfind(fn,os.sep)
        dirpath=fn[:dirchar]
        dndfn=dirpath+os.sep+'dndread.prm'
        try:
            #open file
            fid=open(dndfn,"r")
        except:
            #file does not exist: create default file
            fid=open(dndfn,"w")
            fid.write("I0:1\n")
            fid.write("I1:2\n")
            fid.write("IF:3\n")
            fid.write("I2:0\n")
            fid.close()
            fid=open(dndfn,"r")            
        #file structure: as above  
        lines=fid.read().split('\n')
        fid.close()
        #make default assignments
        tempi0=ones((len(self.i0)),Float32)
        tempi1=ones((len(self.i0)),Float32)
        tempi2=ones((len(self.i0)),Float32)
        tempiF=ones((len(self.i0)),Float32)
        #parse cards and make assignments
        for l in lines:
            ch=l[:2]
            chn=l[3:]
            if upper(ch)=='I0':
                if chn=='1':
                    tempi0=self.i0
                if chn=='2':
                    tempi0=self.i1
                if chn=='3':
                    tempi0=self.iF
            if upper(ch)=='I1':
                if chn=='1':
                    tempi1=self.i0
                if chn=='2':
                    tempi1=self.i1
                if chn=='3':
                    tempi1=self.iF
            if upper(ch)=='I2':
                if chn=='1':
                    tempi2=self.i0
                if chn=='2':
                    tempi2=self.i1
                if chn=='3':
                    tempi2=self.iF
            if upper(ch)=='IF':
                if chn=='1':
                    tempiF=self.i0
                if chn=='2':
                    tempiF=self.i1
                if chn=='3':
                    tempiF=self.iF
        #make array assignments from these...        
        self.ICR=array(tempiF)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(tempi0)
        self.i1=array(tempi1)
        self.i2=array(tempi2)        
        self.iF=array(tempiF)

###################################
##
## DND-CAT ASCII
##
###################################

class dndascii:
    def __init__(self,fn):
        fid=open(fn,"r")
        self.type='DA'
        lines=fid.read().split('\n')
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        for line in lines:
            if line !='':
                parsed=line.split(' ')
                self.rtc.append(atof(parsed[4]))
                self.eV.append(atof(parsed[3]))
                self.i0.append(atof(parsed[6]))
                self.i1.append(atof(parsed[7]))
                self.i2.append(atof(parsed[8]))
                self.iF.append(atof(parsed[9]))
        fid.close()
        self.ICR=array(self.iF)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)
        self.iF=array(self.iF)

###################################
##
## DND-CAT ASCII for 4-element Canberra
##
###################################

class dndasciime:
    def __init__(self,fn):
        fid=open(fn,"r")
        self.type='DM'
        lines=fid.read().split('\n')
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        iFtemp=[]
        for line in lines:
            if line !='':
                parsed=line.split(' ')
                self.rtc.append(atof(parsed[4]))
                self.eV.append(atof(parsed[3]))
                self.i0.append(atof(parsed[24]))
                self.i1.append(atof(parsed[25]))
                self.i2.append(atof(parsed[25]))
                spare=[]
                spare.append(atof(parsed[26]))
                spare.append(atof(parsed[9]))
                spare.append(atof(parsed[28]))
                spare.append(atof(parsed[29]))
                iFtemp.append(spare)
        fid.close()
        self.iF=iFtemp
        self.ICR=array(self.iF)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)
        self.iF=array(self.iF)


###################################
##
## GSE-CARS Format
##
###################################

class gsecarsascii:
    global reloadps
    
    def __init__(self,fn,master):
        global reloadps
        fid=open(fn,"r")
        self.type='GSE'
        lines=fid.read().split('\n')
        fid.close()
        #init data arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        tempiF1=[]
        tempIFF=[]
        tempICR=[]
        #read lines
        readstate=0
        chandef=[[],[],[],[],[],[],[],[]]
        reg1=''
        reglist=[]
        roibuts=[]
        headerstate=0
        for line in lines:
            if line !='':
                if headerstate==1:
                    lp=split(line)
                    if len(lp)>3:
                        type=lp[3]
                        type2=lp[3][0:4]
                        last=lp[len(lp)-1]
                        if len(last)>2: reg=last[len(last)-2:len(last)]
                        if reg[0]=='R' and len(reg)==2 and reg not in reglist:
                            #rlab=lp[4]+' '+lp[5]
                            rlab=line[rfind(line,'{'):rfind(line,'}')]
                            rlab=rstrip(replace(rlab,'}',' '))
                            rlab=rstrip(replace(rlab,'{',' '))
                            roibuts.append(reg+':'+rlab)
                            reglist.append(reg)
                        if lower(type2)=='{lyt' or lower(type2)=='{mu}' or lower(type2)=='{if}':
                            roibuts.append('Lytle')
                            reglist.append('Lytle')              
                if line[0:3]==';==':  #line denotes data descripton section
                    headerstate=1
                if line[0:3]==';--':  #line denotes data section
                    headerstate=2
        #decide which ROI if needed
        if roibuts!=[]:
            if len(roibuts)>1:
                try:
                    #print reloadps[1],roibuts
                    if reloadps[0]==self.type and reloadps[1] in roibuts:
                        rsel=reloadps[1]
                        reg1=rsel[0:2]
                    else:                    
                        raise ValueError, ""
                except:
                    reloadps=[]
                    roidialog=Pmw.Dialog(parent=master,title="ROI Selection",buttons=('OK','Apply All'))
                    roiselbox=Pmw.RadioSelect(parent=roidialog.interior(),labelpos='n',label_text="Please select an ROI to load")
                    rbuts=roibuts
                    for rb in rbuts:
                        roiselbox.add(rb)
                    roiselbox.pack(side=TOP,fill='both',expand=1,padx=5,pady=5)
                    select=roidialog.activate()
                    rsel=roiselbox.getvalue()
                    reg1=rsel[0:2]
                    if select=='Apply All':
                        reloadps=[self.type,rsel]
            else:
                rsel=roibuts[0]
                reg1=rsel[0:2]
                reloadps=[]
        else:
            reg1='Ly'
            reloadps=[]
        for line in lines:
            if line !='':
                if readstate==1: #in data description zone
                    lp=split(line)
                    if len(lp)>3:
                        type=lp[3]
                        type2=lp[3][0:4]
                        last=lp[len(lp)-1]
                        if len(last)>2: reg=last[len(last)-2:len(last)]
                        if type=='{Energy}':
                            chandef[0].append(lp[1])
                        if type=='{Scaler':
                            chandef[1].append(lp[1])
                        if lower(type)=='{i0}':
                            chandef[2].append(lp[1])
                        if lower(type)=='{i1}':
                            chandef[3].append(lp[1])
                        if lower(type)=='{i2}':
                            chandef[4].append(lp[1])
                        if lower(type2)=='{lyt' or type2=='{mu}' or lower(type2)=='{if}':
                            chandef[7].append(lp[1])
                        if (lower(type2)=='{mca' and reg==reg1): 
                            chandef[5].append(lp[1])
                        elif (len(lp)>6 and rfind(lp[6],'mca') and reg==reg1):
                            chandef[5].append(lp[1])
                if readstate==3: #read data lines
                    parsed=split(line)
                    if colind[1]!=[]:
                        self.rtc.append(atof(parsed[colind[1][0]]))
                    else:
                        self.rtc.append(1.0)
                    if colind[0]!=[]:
                        self.eV.append(atof(parsed[colind[0][0]]))
                    else:
                        self.eV.append(1.0)
                    if colind[2]!=[]:
                        self.i0.append(atof(parsed[colind[2][0]]))
                    else:
                        self.i0.append(1.0)
                    if colind[3]!=[]:
                        self.i1.append(atof(parsed[colind[3][0]]))
                    else:
                        self.i1.append(1.0)
                    if colind[4]!=[]:
                        self.i2.append(atof(parsed[colind[4][0]]))
                    else:
                        self.i2.append(1.0)
                    if colind[7]!=[]:
                        tempIFF.append(atof(parsed[colind[7][0]]))
                    else:
                        tempIFF.append(1.0)
                    if colind[5]!=[]:
                        spare=[]
                        for sc in colind[5]:
                            spare.append(atof(parsed[int(sc)]))
                        tempiF1.append(spare)
                    else:
                        tempiF1.append(1.0)
                if readstate==2: #read data order line
                    readstate=3
                    lp=split(line)
                    lp.remove(';')
                    colind=[]
                    for cat in chandef:
                        temp=cat
                        for tac in cat:
                            ind=0
                            for des in lp:
                                if tac==des:
                                    dex=cat.index(tac)
                                    temp[dex]=ind
                                ind=ind+1                                    
                        colind.append(temp)
                #change state lines at the end of line parser
                if line[0:3]==';==':  #line denotes data descripton section
                    readstate=1
                if line[0:3]==';--':  #line denotes data section
                    readstate=2 
        #decide which ROI
        if reg1=='Ly':
            self.iF=tempIFF
        else:
            self.iF=tempiF1
        #mess with iF and ICR
        self.ICR=self.iF #no ICRs... 
        #convert all to arrays
        self.iF=array(self.iF)       
        self.ICR=array(self.ICR)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)

###################################
##
## ALS Line 10.3.2 Single Element Method        
##
###################################

class als10_3_2ascii:
    def __init__(self,fn):
        fid=open(fn,"r")
        self.type='ALS'
        lines=fid.read().split('\n')
        fid.close()
        #init data arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        self.ICR=[]
        #read lines
        for line in lines:
            if line !='':
                parsed=split(line)
                if parsed[0][0] in digits:
                    #then we have a data line
                    self.rtc.append(atof(parsed[1]))
                    self.eV.append(atof(parsed[0]))
                    self.i0.append(atof(parsed[2]))
                    self.i1.append(atof(parsed[3]))
                    self.i2.append(1.0)
                    self.iF.append(atof(parsed[4]))
                    self.ICR.append(atof(parsed[5]))
        #convert all to arrays
        self.iF=array(self.iF)       
        self.ICR=array(self.ICR)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)

###################################
##
## NSLS Single Fluorescence Channel Method        
##
###################################

class nslsascii:
    def __init__(self,fn):
        fid=open(fn,"r")
        self.type='NSLS'
        lines=fid.read().split('\n')
        fid.close()
        #init data arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        self.ICR=[]
        #read lines
        readstate=0
        for line in lines:
            if line !='':
                parsed=split(line)
                if readstate==1 and parsed[0][0] in digits:
                    #then we have a data line
                    self.rtc.append(1.0)
                    self.eV.append(atof(parsed[0]))
                    self.i0.append(atof(parsed[1]))
                    self.i1.append(atof(parsed[2]))
                    self.i2.append(atof(parsed[3]))
                    self.iF.append(atof(parsed[4]))
                    self.ICR.append(atof(parsed[4]))
                if parsed[0][0]=='-': readstate=1
        #convert all to arrays
        self.iF=array(self.iF)       
        self.ICR=array(self.ICR)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)


###################################
##
## NSLS "Recent" X-11A Formats        
##
###################################

class nslsnewascii:
    def __init__(self,fn):
        fid=open(fn,"r")
        self.type='NSLSv2'
        lines=fid.read().split('\n')
        fid.close()
        #init data arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        self.ICR=[]
        #read lines
        readstate=0
        for line in lines:
            if line !='':
                parsed=split(line)
                if readstate==1 and parsed[0][0] in digits:
                    #then we have a data line
                    self.rtc.append(1.0)
                    self.eV.append(atof(parsed[0]))
                    self.i0.append(atof(parsed[1]))
                    self.i1.append(atof(parsed[2]))
                    self.i2.append(atof(parsed[4]))
                    self.iF.append(atof(parsed[3]))
                    self.ICR.append(atof(parsed[3]))
                if parsed[0][0]=='-': readstate=1
        #convert all to arrays
        self.iF=array(self.iF)       
        self.ICR=array(self.ICR)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)



###################################
##
## SRC Soft X-ray Data Method        
##
###################################

class srcascii:
    def __init__(self,fn):
        fid=open(fn,"r")
        self.type='WiscSRC'
        lines=fid.read().split('\n')
        fid.close()
        #init data arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        self.ICR=[]
        chandef=[[],[],[],[],[],[],[]]        
        #read lines
        readstate=0
        for line in lines:
            if line !='':
                parsed=split(line)
                if readstate==1 and parsed[0][0] in digits:
                    #then we have a data line
                    if chandef[1]!=[]:
                        self.rtc.append(atof(parsed[chandef[1][0]]))
                    else:
                        self.rtc.append(1.0)
                    if chandef[0]!=[]:
                        self.eV.append(atof(parsed[chandef[0][0]]))
                    else:
                        self.eV.append(1.0)
                    if chandef[2]!=[]:
                        self.i0.append(atof(parsed[chandef[2][0]]))
                    else:
                        self.i0.append(1.0)
                    if chandef[3]!=[]:
                        self.i1.append(atof(parsed[chandef[3][0]]))
                    else:
                        self.i1.append(1.0)
                    if chandef[4]!=[]:
                        self.i2.append(atof(parsed[chandef[4][0]]))
                    else:
                        self.i2.append(1.0)
                    if chandef[5]!=[]:
                        temp1=[]
                        temp2=[]
                        i=0
                        for col in chandef[5]:
                            temp1.append(atof(parsed[chandef[5][i]]))
                            temp2.append(1.0)
                            i=i+1
                        self.iF.append(temp1)
                        self.ICR.append(temp2)
                    else:
                        self.iF.append(1.0)
                        self.ICR.append(1.0)
                if parsed[0]=='Energy(eV)':
                    #header line.  Read to determine colums
                    readstate=1
                    for title in parsed:
                        if title=='Energy(eV)': chandef[0].append(parsed.index(title))
                        if title=='Time(s)': chandef[1].append(parsed.index(title))
                        if title=='I0': chandef[2].append(parsed.index(title))
                        if title=='I1': chandef[3].append(parsed.index(title))
                        if title=='I2': chandef[4].append(parsed.index(title))
                        if title[0:2]=='No': chandef[5].append(parsed.index(title))
        #convert all to arrays
        self.iF=array(self.iF)       
        self.ICR=array(self.ICR)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)

###################################
##
## PNC 1-D Scan Data Method        
##
###################################

class pncascii:
    def __init__(self,fn):
        fid=open(fn,"r")
        self.type='PNC'
        lines=fid.read().split('\n')
        fid.close()
        #init data arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        self.ICR=[]
        chandef=[[],[],[],[],[],[],[]]        
        #read lines
        readstate=0
        for line in lines:
            if line !='':
                parsed=split(line)
                if readstate==1 and parsed[0][0] in digits:
                    #then we have a data line
                    if chandef[1]!=[]:
                        self.rtc.append(atof(parsed[chandef[1][0]]))
                    else:
                        self.rtc.append(1.0)
                    if chandef[0]!=[]:
                        self.eV.append(atof(parsed[chandef[0][0]]))
                    else:
                        self.eV.append(1.0)
                    if chandef[2]!=[]:
                        self.i0.append(atof(parsed[chandef[2][0]]))
                    else:
                        self.i0.append(1.0)
                    if chandef[3]!=[]:
                        self.i1.append(atof(parsed[chandef[3][0]]))
                    else:
                        self.i1.append(1.0)
                    if chandef[4]!=[]:
                        self.i2.append(atof(parsed[chandef[4][0]]))
                    else:
                        self.i2.append(1.0)
                    if chandef[5]!=[]:
                        temp1=[]
                        temp2=[]
                        i=0
                        for col in chandef[5]:
                            temp1.append(atof(parsed[chandef[5][i]]))
                            temp2.append(1.0)
                            i=i+1
                        self.iF.append(temp1)
                        self.ICR.append(temp2)
                    else:
                        self.iF.append(1.0)
                        self.ICR.append(1.0)
                if parsed[0]=='#Mono':
                    #header line.  Read to determine colums
                    if parsed[7]=='ID':
                        offset=7
                    elif parsed[8]=='ID':
                        offset=8
                    else:
                        offset=5
                    readstate=1
                    ignore=0
                    for title in parsed:
                        if title=="DT-Corr":
                            offset=offset+1
                            ignore=1
                        if title=='#Mono': chandef[0].append(parsed.index(title))
                        if title=='time': chandef[1].append(1)
                        if title=='I0' and not ignore: chandef[2].append(parsed.index(title)-offset)
                        if title=='I1': chandef[3].append(parsed.index(title)-offset)
                        if title=='IT': chandef[3].append(parsed.index(title)-offset)
                        if title=='I2': chandef[4].append(parsed.index(title)-offset)
                        if title=='IR': chandef[4].append(parsed.index(title)-offset) 
                        if title=='CalDiode': chandef[4].append(parsed.index(title)-offset)
                        if lower(title[0:3])=='dxp': chandef[5].append(parsed.index(title)-offset)
                        if title=='IF': chandef[5].append(parsed.index(title)-offset)
        #convert all to arrays
        self.iF=array(self.iF)       
        self.ICR=array(self.ICR)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)
        
###################################
##
## NSLS BL X26A BSIF binary Loader        
##
###################################

class bsifx26a:
    global reloadps
    
    def __init__(self,fn,master):
        global reloadps
        fid=open(fn,"rb")
        self.type='BSIF'
        #clear all arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        self.ICR=[]
        data=fid.read()
        new=xdrlib.Unpacker(data)
        norows=new.unpack_int()
        nocols=new.unpack_int()
        nodata=new.unpack_int()
        j=0
        while j<3:
            new.unpack_int()
            j=j+1
        (estart,eend)=(new.unpack_float(),new.unpack_float())
        (rstart,rend)=(new.unpack_float(),new.unpack_float())
        colheads=[]
        j=0
        while j<nodata+3:
            length=new.unpack_int()
            if length>0:
                char=new.unpack_string()
                if length>1:colheads.append(char)
            j=j+1
        datatype=new.unpack_int()
        ublen=new.unpack_int() #user buffer -- energy positions!
        #datatype=5 is longarray of 4 bytes per data point
        imagedatabytes=4*norows*nocols*nodata
        image=new.unpack_fopaque(imagedatabytes)
        #parse headers
        roibuts=[]
        for i in range(2,9):
            if colheads[i] not in roibuts:
                roibuts.append(colheads[i])
        #choose ROI
        if len(roibuts)>1:
            try:
                #print reloadps[1],roibuts
                if reloadps[0]==self.type and reloadps[1] in roibuts:
                    rsel=reloadps[1]
                    reg1=rsel[0:2]
                else:                    
                    raise ValueError, ""
            except:
                reloadps=[]
                roidialog=Pmw.Dialog(parent=master,title="ROI Selection",buttons=('OK','Apply All'))
                roiselbox=Pmw.RadioSelect(parent=roidialog.interior(),labelpos='n',label_text="Please select an ROI to load")
                rbuts=roibuts
                for rb in rbuts:
                    roiselbox.add(rb)
                roiselbox.pack(side=TOP,fill='both',expand=1,padx=5,pady=5)
                select=roidialog.activate()
                rsel=roiselbox.getvalue()
                reg1=rsel[0:2]
                if select=='Apply All':
                    reloadps=[self.type,rsel]
        else:
            rsel=roibuts[0]
            reg1=rsel[0:2]
            reloadps=[]
        fmt='>' #change byte order to big-endian
        for i in range(nocols):
            fmt=fmt+'l'
        tempiF=[]
        for i in range(nodata):
            buf=image[i*nocols*4:(i+1)*nocols*4]
            dt=unpack(fmt,buf)
            if upper(colheads[i+2])=='TIME':
                self.rtc=array(dt,typecode=Float32)
            if upper(colheads[i+2])=='ION CHAMBER':
                self.i0=array(dt,typecode=Float32)
            if upper(colheads[i+2])=='PIN DIODE':
                self.i1=array(dt,typecode=Float32)
            if upper(colheads[i+2])=='RING CURRENT':
                self.i2=array(dt,typecode=Float32)
            if colheads[i+2][0:2]==reg1:
                tempiF.append(dt)
        i=0
        energy=[]
        while i<nocols:
            energy.append(new.unpack_float())
            i=i+1
        self.eV=array(energy)
        self.iF=array(tempiF,typecode=Float32)
        self.iF=transpose(self.iF)
        self.ICR=self.iF

        
###################################
##
## Generic File Loader        
##
###################################

class generic:
    global genericps
    
    def __init__(self,fn,master):
        global genericps
        self.type='GC'
        #clear all arrays
        self.rtc=[]
        self.eV=[]
        self.i0=[]
        self.i1=[]
        self.i2=[]
        self.iF=[]
        self.ICR=[]
        #assume path is already good
        #open file and read into buffer
        fid=open(fn,"r")
        self.lines=fid.read().split('\n')
        fid.close()
        #separate file into regions...
        #   comment lines (either #,*,% at char 0)
        #   data line w/ separators (tabs,commas,spaces)
        comchar=['#','*','%','!','A',';']
        comline=[]
        datline=[]
        for newline in self.lines:
            if newline !='':
                if newline[0] in comchar:
                    comline.append(newline)
                else:
                    datline.append(newline)
        numcoms=len(comline)
        #calculate number of columns from end of file...
        i=1
        x=0
        while x==0:
            d=datline[len(datline)-i]
            if len(d)>2:
                x=1
            else:
                i=i+1
        dr=replace(d,',',' ')
        dpp=split(dr)
        colnum=len(dpp)
        self.collist=['None']
        for i in range(1,colnum+1):
            self.collist.append(str(i))
        #open window to display intial lines of file
        ma=self.ma=Toplevel(master)
        ma.focus_set()
        ma.title("Generic File Read")
        f1=Frame(ma)
        f1.pack(side=TOP,padx=5,pady=5)
        #file display
        fwin=Pmw.ScrolledText(f1,label_text=fn,borderframe=1,text_padx=10,text_pady=10,labelpos=NW,usehullsize=1,
                              hull_width=400,hull_height=300,vscrollmode='static',hscrollmode='static',text_wrap=NONE)
        fwin.importfile(fn)
        fwin.pack(side=TOP)
        f2=Frame(ma)
        f2.pack(side=TOP,padx=5,pady=5)
        #allow offset of data
        self.offset=Pmw.Counter(f2,labelpos=W,label_text='Offset lines from start of file:',
                           entryfield_value=numcoms,entryfield_validate='numeric',datatype='numeric',
                           entryfield_entry_width=5)
        self.offset.pack(side=LEFT,padx=2,pady=2)
        f4=Frame(ma)
        f4.pack(side=TOP,padx=5,pady=5)
        #action buttons
        bbox=Pmw.ButtonBox(f4)
        bbox.pack(side=TOP,padx=10,pady=2)
        a=bbox.add('Read',width=10)
        a.bind("<Button-1>",self.pressedabutton)
        a.bind("<ButtonRelease-1>",self.doneclick)
        a=bbox.add('Cancel',width=10)
        a.bind("<Button-1>",self.pressedabutton)
        a.bind("<ButtonRelease-1>",self.doneclick)
        f33=Pmw.ScrolledFrame(ma,labelpos=N,label_text='Column Assignments',vscrollmode='static',usehullsize=1,hull_width=400,hull_height=400)
        f33.pack(side=TOP,pady=2,padx=2)
        f3=f33.interior()
        #column assignments
        #add choices to assign data columns to assigned formats -- each type gets a column choice
        collabels=Pmw.RadioSelect(f3,orient='vertical',padx=1,pady=1,label_text=' ',labelpos=N)
        self.radioRTC=Pmw.RadioSelect(f3,buttontype='radiobutton',orient='vertical',labelpos=N,
                                label_text='RTC',padx=1,pady=1)
        self.radioeV=Pmw.RadioSelect(f3,buttontype='radiobutton',orient='vertical',labelpos=N,
                                label_text='eV',padx=1,pady=1)
        self.radioi0=Pmw.RadioSelect(f3,buttontype='radiobutton',orient='vertical',labelpos=N,
                                label_text='i0',padx=1,pady=1)
        self.radioi1=Pmw.RadioSelect(f3,buttontype='radiobutton',orient='vertical',labelpos=N,
                                label_text='i1',padx=1,pady=1)
        self.radioi2=Pmw.RadioSelect(f3,buttontype='radiobutton',orient='vertical',labelpos=N,
                                label_text='i2',padx=1,pady=1)
        self.radioiF=Pmw.RadioSelect(f3,buttontype='checkbutton',orient='vertical',labelpos=N,
                                label_text='iF',padx=1,pady=1)
        self.radioICR=Pmw.RadioSelect(f3,buttontype='checkbutton',orient='vertical',labelpos=N,
                                label_text='ICR',padx=1,pady=1)
        butlist=[self.radioRTC,self.radioeV,self.radioi0,self.radioi1,self.radioi2]
        butclist=[self.radioiF,self.radioICR]
        for choice in self.collist:
            if choice=='None':
                collabels.add(choice,text='None   ',relief=FLAT,state=DISABLED,disabledforeground='black')
            else:
                collabels.add(choice,text='Column '+choice,relief=FLAT,state=DISABLED,disabledforeground='black')
        collabels.pack(side=LEFT,pady=2)
        for but in butlist:
            i=0  #0?
            for choice in self.collist:
                but.add(choice,text=' ',value=i)
                i=i+1
            but.pack(side=LEFT,pady=2)
            try:
                but.invoke(genericps[butlist.index(but)])
            except:
                but.invoke('None')
        for but in butclist:
            for choice in self.collist:
                but.add(choice,text=' ')
            but.pack(side=LEFT,pady=2)
            try:
                for opt in genericps[len(butlist)+butclist.index(but)]:
                    but.invoke(opt)
            except:
                pass
        #remove None from self.collist...
        self.collist.remove('None')
        ma.wait_window(ma)
        #after window is killed...
            
    def pressedabutton(self,event):
        b=event.widget.cget('text')
        event.widget.config(state=ACTIVE)
        event.widget.config(relief=SUNKEN)
        if b=='Read':
            self.readgenericdata()
            
    def doneclick(self,event):
        event.widget.config(state=NORMAL)
        event.widget.config(relief=RAISED)
        self.ma.destroy()

    def readgenericdata(self):
        global genericps
        #read and format the data
        print "Reading in generic file format"
        tempiF=[]
        tempICR=[]
        i=0
        for l in self.lines:
            if i<int(self.offset.get()):
                if l != '':i=i+1
            else:
                #parse data again...
                if l != '':
                    dr=replace(l,',',' ')
                    dpp=split(dr)
                    #if data type not defined, then assume = 1
                    #rtc
                    if self.radioRTC.getcurselection() in self.collist:
                        self.rtc.append(atof(dpp[int(self.radioRTC.getcurselection())-1]))
                    else:
                        self.rtc.append(1.0)
                    #eV
                    if self.radioeV.getcurselection() in self.collist:
                        self.eV.append(atof(dpp[int(self.radioeV.getcurselection())-1]))
                    else:
                        self.eV.append(1.0)
                    #i0
                    if self.radioi0.getcurselection() in self.collist:
                        self.i0.append(atof(dpp[int(self.radioi0.getcurselection())-1]))
                    else:
                        self.i0.append(1.0)
                    #i1
                    if self.radioi1.getcurselection() in self.collist:
                        self.i1.append(atof(dpp[int(self.radioi1.getcurselection())-1]))
                    else:
                        self.i1.append(1.0)
                    #i2
                    if self.radioi2.getcurselection() in self.collist:
                        self.i2.append(atof(dpp[int(self.radioi2.getcurselection())-1]))
                    else:
                        self.i2.append(1.0)
                    #iF -- deal with multiple selections
                    if self.radioiF.getcurselection() != ():
                        spare=[]
                        for sc in self.radioiF.getcurselection():
                            spare.append(atof(dpp[int(sc)-1]))
                        tempiF.append(spare)
                        fl=len(spare)
                    else:
                        tempiF.append(1.0)
                        fl=0
                    #ICRs -- deal with multiple selections
                    if self.radioICR.getcurselection() != ():
                        spare=[]
                        for sc in self.radioICR.getcurselection():
                            spare.append(atof(dpp[int(sc)-1]))
                        tempICR.append(spare)
                        icrl=len(spare)
                    else:
                        tempICR.append(1.0)
                        icrl=0
        #mess parameter history
        genericps=[self.radioRTC.getcurselection(),self.radioeV.getcurselection(),self.radioi0.getcurselection(),
                   self.radioi1.getcurselection(),self.radioi2.getcurselection(),self.radioiF.getcurselection(),
                   self.radioICR.getcurselection()]
        #mess with iF and ICR
        self.iF=tempiF
        self.ICR=tempICR
        if type(self.ICR[0]) != type(self.iF[0]):
            self.ICR=self.iF
        #convert all to arrays
        self.iF=array(self.iF)       
        self.ICR=array(self.ICR)
        self.rtc=array(self.rtc)
        self.eV=array(self.eV)
        self.i0=array(self.i0)
        self.i1=array(self.i1)
        self.i2=array(self.i2)
                        
###################################
##
## Main Loading Function
##
###################################

def PsCLEAR():
    global reloadps
    reloadps=[]

def XASget(fn,master):
    global starte,ende
    global edge#,dndcolassgn
    global genericps
    global reloadps
    fid=open(fn,"r")
    buf=fid.readline()
    buf2=fid.readline()
    fid.close()
    exten=fn[len(fn)-3:]
    if exten=='cdf':
        print "No support for .cdf files yet"
    if exten=='dme':
        return dndasciime(fn)
    if exten=='dnd':
        if len(buf)<500:  #must modify this condition!
            return dndascii(fn)
        else:
            ##start dialog box for extra input!
            #dndcolassgn=[6,7,8,0]
            #columndialog=Pmw.Dialog(master,buttons=('OK','Cancel'),defaultbutton='OK',
            #                        title='DND Column Assignments')
            #i0col=Pmw.Counter(columndialog.interior(),labelpos='w',label_text='I0:',
            #                  entry_width=5,entryfield_value=dndcolassgn[0],
            #                  entryfield_validate={'validator':'integer','min':0})
            #i1col=Pmw.Counter(columndialog.interior(),labelpos='w',label_text='IT1:',
            #                  entry_width=5,entryfield_value=dndcolassgn[1],
            #                  entryfield_validate={'validator':'integer','min':0})
            #ifcol=Pmw.Counter(columndialog.interior(),labelpos='w',label_text='IF:',
            #                  entry_width=5,entryfield_value=dndcolassgn[2],
            #                  entryfield_validate={'validator':'integer','min':0})
            #i2col=Pmw.Counter(columndialog.interior(),labelpos='w',label_text='IT2:',
            #                  entry_width=5,entryfield_value=dndcolassgn[3],
            #                  entryfield_validate={'validator':'integer','min':0})
            #wids=(i0col,i1col,ifcol,i2col)
            #Pmw.alignlabels(wids)
            #for w in wids:
            #    w.pack(side=TOP,padx=5,pady=2)
            #columndialog.withdraw()
            #columndialog.activate(globalMode = 'nograb')
            #dndcolassgn=[i0col.getvalue(),i1col.getvalue(),ifcol.getvalue(),i2col.getvalue()]
            #print dndcolassgn
            try:
                starte=tkSimpleDialog.askfloat('Input for DND-Quick','Enter starting energy',initialvalue=starte,parent=master)
            except:
                starte=tkSimpleDialog.askfloat('Input for DND-Quick','Enter starting energy',parent=master)                
            try:
                ende=tkSimpleDialog.askfloat('Input for DND-Quick','Enter final energy',initialvalue=ende,parent=master)
            except:
                ende=tkSimpleDialog.askfloat('Input for DND-Quick','Enter final energy',parent=master)                
            try:
                edge=tkSimpleDialog.askfloat('Input for DND-Quick','Enter edge energy',initialvalue=edge,parent=master)
            except:
                edge=tkSimpleDialog.askfloat('Input for DND-Quick','Enter edge energy',parent=master)                
            e=array((starte,ende))
            return dndquick(fn,e,edge)
    elif count(buf,'SRL -')==1:
        if float(buf.split()[-1])<2:
            return ssrlbin(fn,master)
        else:
            return ssrlbin(fn,master,ieee=1)
    elif count(buf,'SRL  ')==1:
        return ssrlascii(fn,master)
    elif binascii.b2a_hex(buf[0:6])=='000000010000':
        return bsifx26a(fn,master)
    elif buf[0]==';':
        return gsecarsascii(fn,master)
    elif count(lower(buf),'title')>0 and count(lower(buf2),'region')>0:
        return als10_3_2ascii(fn)
    elif count(upper(buf),'NSLS')>0: 
        return nslsascii(fn)
    elif count(upper(buf),'XDAC V1')>0:
        return nslsnewascii(fn)
    elif count(upper(buf),'DCM')>0 and count(upper(buf),'ACQUSITION')>0:
        return srcascii(fn)
    elif count(upper(buf),'SCAN')>0 and count(upper(buf),'LABVIEW')>0:
        return pncascii(fn)
    else:
        #try generic routine
        #try:
        return generic(fn,master)
        #except:
        #    return generic(fn,master)
        

        