/*  smplayer, GUI front-end for mplayer.
    Copyright (C) 2007 Ricardo Villalba <rvm@escomposlinux.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "inforeader.h"
#include <qprocess.h>
#include <qstringlist.h>
#include <qapplication.h>
#include <qregexp.h>
#include "helper.h"

#define NOME 0
#define VO 1
#define AO 2
#define DEMUXER 3
#define VC 4
#define AC 5

InfoReader::InfoReader( QString mplayer_bin, QObject * parent, const char * name )
	: QObject(parent, name)
{
	mplayerbin = mplayer_bin;

	proc = new QProcess(this, "inforeader_process");
    proc->setCommunication( QProcess::Stdin | QProcess::Stdout | QProcess::Stderr );

	connect( proc, SIGNAL(readyReadStdout()), this, SLOT(readLines()) );
	//connect( proc, SIGNAL(processExited()), this, SLOT(procFinished()) );
}

InfoReader::~InfoReader() {
}

void InfoReader::getInfo() {
	waiting_for_key = TRUE;
	vo_list.clear();
	ao_list.clear();
	demuxer_list.clear();

	run("-vo help -ao help -demuxer help -vc help -ac help");
	//run("-demuxer help");

	list();
}

void InfoReader::list() {
	qDebug("InfoReader::list");

	InfoList::iterator it;

	qDebug(" vo_list:");
	for ( it = vo_list.begin(); it != vo_list.end(); ++it ) {
		qDebug( "driver: '%s', desc: '%s'", (*it).name().utf8().data(), (*it).desc().utf8().data());
	}

	qDebug(" ao_list:");
	for ( it = ao_list.begin(); it != ao_list.end(); ++it ) {
		qDebug( "driver: '%s', desc: '%s'", (*it).name().utf8().data(), (*it).desc().utf8().data());
	}

	qDebug(" demuxer_list:");
	for ( it = demuxer_list.begin(); it != demuxer_list.end(); ++it ) {
		qDebug( "demuxer: '%s', desc: '%s'", (*it).name().utf8().data(), (*it).desc().utf8().data());
	}

	qDebug(" vc_list:");
	for ( it = vc_list.begin(); it != vc_list.end(); ++it ) {
		qDebug( "codec: '%s', desc: '%s'", (*it).name().utf8().data(), (*it).desc().utf8().data());
	}

	qDebug(" ac_list:");
	for ( it = ac_list.begin(); it != ac_list.end(); ++it ) {
		qDebug( "codec: '%s', desc: '%s'", (*it).name().utf8().data(), (*it).desc().utf8().data());
	}

}

static QRegExp rx_vo_key("^Available video output drivers");
static QRegExp rx_ao_key("^Available audio output drivers");
static QRegExp rx_demuxer_key("^Available demuxers");
static QRegExp rx_ac_key("^Available audio codecs");
static QRegExp rx_vc_key("^Available video codecs");

static QRegExp rx_driver("\\t(.*)\\t(.*)");
static QRegExp rx_demuxer("^\\s+([A-Z,a-z,0-9]+)\\s+(\\d+)\\s+(\\S.*)");
static QRegExp rx_codec("^([A-Z,a-z,0-9]+)\\s+([A-Z,a-z,0-9]+)\\s+([A-Z,a-z,0-9]+)\\s+(\\S.*)");

void InfoReader::readLines() {
	qDebug("InfoReader::readLines");

	QString line;

	while (proc->canReadLineStdout ()) {
		line = proc->readLineStdout();
		qDebug("InfoReader::readLines: line: '%s'", line.utf8().data());
		//qDebug("waiting_for_key: %d", waiting_for_key);

		if (!waiting_for_key) {
			if ( rx_driver.search(line) > -1 ) {
				QString name = rx_driver.cap(1);
				QString desc = rx_driver.cap(2);
				qDebug("InfoReader::readLines: found driver: '%s' '%s'", name.utf8().data(), desc.utf8().data());
				if (reading_type==VO) {
					vo_list.append( InfoData(name, desc) );
				} 
				else
				if (reading_type==AO) {
					ao_list.append( InfoData(name, desc) );
				}
				else
				qWarning("InfoReader::readLines: Unknown type! Ignoring");
			}
			else
			if ( rx_demuxer.search(line) > -1 ) {
				QString name = rx_demuxer.cap(1);
				QString desc = rx_demuxer.cap(3);
				qDebug("InfoReader::readLines: found demuxer: '%s' '%s'", name.utf8().data(), desc.utf8().data());
				demuxer_list.append( InfoData(name, desc) );
			}
			else
			if ( rx_codec.search(line) > -1 ) {
				QString name = rx_codec.cap(1);
				QString desc = rx_codec.cap(4);
				qDebug("InfoReader::readLines: found codec: '%s' '%s'", name.utf8().data(), desc.utf8().data());
				if (reading_type==VC) {
					vc_list.append( InfoData(name, desc) );
				} 
				else
				if (reading_type==AC) {
					ac_list.append( InfoData(name, desc) );
				}
				else
				qWarning("InfoReader::readLines: Unknown type! Ignoring");
			}
		}

		if ( rx_vo_key.search(line) > -1 ) {
			reading_type = VO;
			waiting_for_key = FALSE;
			qDebug("InfoReader::readLines: found key: vo");
		}

		if ( rx_ao_key.search(line) > -1 ) {
			reading_type = AO;
			waiting_for_key = FALSE;
			qDebug("InfoReader::readLines: found key: ao");
		}

		if ( rx_demuxer_key.search(line) > -1 ) {
			reading_type = DEMUXER;
			waiting_for_key = FALSE;
			qDebug("InfoReader::readLines: found key: demuxer");
		}

		if ( rx_ac_key.search(line) > -1 ) {
			reading_type = AC;
			waiting_for_key = FALSE;
			qDebug("InfoReader::readLines: found key: ac");
		}

		if ( rx_vc_key.search(line) > -1 ) {
			reading_type = VC;
			waiting_for_key = FALSE;
			qDebug("InfoReader::readLines: found key: vc");
		}

	}

}


bool InfoReader::run(QString options) {
	qDebug("InfoReader::run: '%s'", options.utf8().data());

	if (proc->isRunning()) {
		qWarning("InfoReader::run: process already running");
		return false;
	}

	proc->clearArguments();

	proc->addArgument(mplayerbin);

	QStringList args = QStringList::split(" ", options);
	QStringList::Iterator it = args.begin();
	while( it != args.end() ) {
		proc->addArgument( (*it) );
		++it;
	}

	if ( !proc->start() ) {
		qWarning("InfoReader::run: process can't start!");
		return false;
	}

	//Wait until finish
	/*
	while (proc->isRunning()) {
		qApp->processEvents();
		Helper::msleep(200);
	}
	*/
	Helper::finishProcess( proc );

	qDebug("InfoReader::run : terminating");

	return TRUE;
}

