/*  smplayer, GUI front-end for mplayer.
    Copyright (C) 2007 Ricardo Villalba <rvm@escomposlinux.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "preferencesdialog.h"
#include "helper.h"
#include "images.h"
#include "global.h"
#include "encodings.h"
#include "seekwidget.h"

#include <qlistbox.h>
#include <qwidgetstack.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qcolor.h>
#include <qslider.h>
#include <qfontdialog.h>
#include <qfiledialog.h>
#include <qcolordialog.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qstylefactory.h>
//#include <qlayout.h> // QSpacerItem
#include <qfont.h>
#include <qwhatsthis.h>
#include <qregexp.h>
#include "filedialog.h"

#define PAGE_GENERAL 0
#define PAGE_SUBTITLES 1
#define PAGE_ADVANCED 2
#define PAGE_PERFORMANCE 3
#define PAGE_GUI 4
#define PAGE_DRIVES 5
#define PAGE_MOUSE 6

PreferencesDialog::PreferencesDialog( QWidget* parent, const char* name, bool modal, WFlags fl )
	: PreferencesDialogBase(parent, name, modal, fl)
{
    sections->insertItem( "General", General );
    sections->insertItem( "Drives", Drives );
    sections->insertItem( "Performance", Performance );
    sections->insertItem( "Subtitles", Subtitles );
    sections->insertItem( "Advanced", Advanced );
    sections->insertItem( "Interface", Gui );
    sections->insertItem( "Keyboard and mouse", Mouse );

	connect( sections, SIGNAL(highlighted(int)), 
             this, SLOT(changeSection(int)) );

	connect( search_mplayerbin_button, SIGNAL(clicked()),
             this, SLOT(selectMplayerPath()) );

	connect( screenshot_button, SIGNAL(clicked()),
             this, SLOT(selectScreenshotDirectory()) );

	connect( remember_all_check, SIGNAL(toggled(bool)),
             this, SLOT(rememberAllButtonToggled(bool)) );

	connect( system_font_button, SIGNAL(clicked()),
             this, SLOT(systemFontRadioButtonSelected()) );

	connect( ttf_font_button, SIGNAL(clicked()),
             this, SLOT(ttfFontRadioButtonSelected()) );

	connect( select_system_font_button, SIGNAL(clicked()),
             this, SLOT(selectSystemFont()) );

	connect( select_ttf_font_button, SIGNAL(clicked()),
             this, SLOT(selectTtfFont()) );

	connect( font_ass_check, SIGNAL(toggled(bool)),
             this, SLOT(assButtonToggled(bool)) );

	connect( asscolor_button, SIGNAL(clicked()),
             this, SLOT(selectAssColor()) );

	connect( assbordercolor_button, SIGNAL(clicked()),
             this, SLOT(selectAssBorderColor()) );

	connect( autosync_check, SIGNAL(toggled(bool)),
             this, SLOT(autoSyncButtonToggled(bool)) );

	connect( single_instance_check, SIGNAL(toggled(bool)),
             this, SLOT(singleInstanceButtonToggled(bool)) );

	sections->setSelected( General, TRUE );

	// Style combo
#if STYLE_SWITCHING
	style_combo->insertItem( "<default>" );
	QStringList k = QStyleFactory::keys();
	for ( QStringList::Iterator it = k.begin(); it != k.end(); ++it ) {
		style_combo->insertItem( *it );
    }
#endif

	// DVD device combo
	// In windows, insert the drives letters
#ifdef Q_OS_WIN
	#if QT_VERSION < 0x040000
	const QFileInfoList* roots = QDir::drives();
    QPtrListIterator<QFileInfo> i(*roots);
    QFileInfo* fi;
    while ( (fi = *i) ) {
        ++i;
		QString s = fi->filePath();
		if (s.endsWith("/")) s = s.remove( s.length()-1,1);
		dvd_device_combo->insertItem( s );
        cdrom_device_combo->insertItem( s );
    }
	#else
	QFileInfoList list = QDir::drives();
	for (int n = 0; n < list.size(); n++) {
		QString s = list[n].filePath();
		if (s.endsWith("/")) s = s.remove( s.length()-1,1);
		dvd_device_combo->insertItem( s );
		cdrom_device_combo->insertItem( s );
	}
	#endif
#else
#define ADD_IF_EXISTS( string ) \
	if (QFileInfo( string ).exists()) { \
		dvd_device_combo->insertItem( string ); \
        cdrom_device_combo->insertItem( string ); \
    }

	ADD_IF_EXISTS("/dev/dvd");
	ADD_IF_EXISTS("/dev/dvdrecorder");
	ADD_IF_EXISTS("/dev/cdrom");
	ADD_IF_EXISTS("/dev/cdrecorder");
#endif

	// Priority is only for windows, so we disable for other systems
#ifndef Q_OS_WIN
	priority_group->hide();
#endif

	drivers_set = FALSE;

	// Monitor aspect
	monitoraspect_combo->insertItem("Auto");
	monitoraspect_combo->insertItem("4:3");
	monitoraspect_combo->insertItem("16:9");
	monitoraspect_combo->insertItem("5:4");
	monitoraspect_combo->insertItem("16:10");

#if STYLE_SWITCHING == 0
	style_label->hide();
	style_combo->hide();
#endif

	/*
	autoq_slider->setTickmarks( QSlider::Below );
	autoq_slider->setTickInterval( 1 );
	autoq_slider->setLineStep( 1 );
	autoq_slider->setPageStep( 1 );
	*/

	encodings = new Encodings(this, "encodings" );
	font_encoding_combo->insertStringList( encodings->list() );

	// Mouse function combos
	left_click_combo->insertItem( "None" );
	/*left_click_combo->insertStringList( key_list->functionList() );*/

	double_click_combo->insertItem( "None" );
	/*double_click_combo->insertStringList( key_list->functionList() );*/

	// Language combo
	//createLanguageCombo();

	// Icon set combo
	QDir icon_dir = Helper::themesPath();
	qDebug("icon_dir: %s", icon_dir.absPath().utf8().data());
	QStringList iconsets = icon_dir.entryList(QDir::Dirs);
	iconset_combo->insertItem( "Default" );
	for (int n=0; n < iconsets.count(); n++) {
		if ((iconsets[n]!=".") && (iconsets[n]!=".."))
			iconset_combo->insertItem( iconsets[n] );
	}

	// MPlayer language combos.
    endoffile_combo->insertItem( "Exiting... \\(End of file\\)" );
    endoffile_combo->insertItem( "Saliendo... \\(Fin de archivo\\.\\)" );
    endoffile_combo->insertItem( "Beenden... \\(Dateiende erreicht\\)" );
    endoffile_combo->insertItem( "Sortie... \\(Fin du fichier\\)" );
    endoffile_combo->insertItem( "In uscita... \\(Fine del file\\)" );

    novideo_combo->insertItem( "Video: no video" );
	novideo_combo->insertItem( QString::fromLatin1("Vdeo: no hay video") );
    novideo_combo->insertItem( "Video: kein Video" );
    novideo_combo->insertItem( QString::fromLatin1("Vido : pas de vido") );
    novideo_combo->insertItem( "Video: nessun video" );

	languageChange();
}

PreferencesDialog::~PreferencesDialog() {
}

void PreferencesDialog::setActionsList(QStringList l) {
	left_click_combo->insertStringList( l );
	double_click_combo->insertStringList( l );
}

void PreferencesDialog::createLanguageCombo() {
	QMap <QString,QString> m;
	m["bg"] = tr("Bulgarian");
	m["cs"] = tr("Czech");
	m["de"] = tr("German");
	m["el"] = tr("Greek");
	m["en_US"] = tr("English");
	m["es"] = tr("Spanish");
	m["fi"] = tr("Finnish");
	m["fr"] = tr("French");
	m["hu"] = tr("Hungarian");
	m["it"] = tr("Italian");
	m["ja"] = tr("Japanese");
	m["ka"] = tr("Georgian");
	m["nl"] = tr("Dutch");
	m["pl"] = tr("Polish");
	m["pt_BR"] = tr("Portuguese - Brazil");
	m["pt_PT"] = tr("Portuguese - Portugal");
	m["ro_RO"] = tr("Romanian");
	m["ru_RU"] = tr("Russian");
	m["sk"] = tr("Slovak");
	m["sr"] = tr("Serbian");
	m["sv"] = tr("Swedish");
	m["tr"] = tr("Turkish");
	m["uk_UA"] = tr("Ukrainian");
	m["zh_CN"] = tr("Simplified-Chinese");
	m["zh_TW"] = tr("Traditional Chinese");

	// Language combo
	QDir translation_dir = Helper::translationPath();
	QStringList languages = translation_dir.entryList("*.qm");
	QRegExp rx_lang("smplayer_(.*)\\.qm");
	language_combo->clear();
	language_combo->insertItem( tr("<Autodetect>") );
	for (int n=0; n < languages.count(); n++) {
		if (rx_lang.search(languages[n]) > -1) {
			QString l = rx_lang.cap(1);
			languages_list.append(l);
			if (m.contains(l)) l = m[l] + " ("+l+")";
			language_combo->insertItem( l );
		}
	}
}

void PreferencesDialog::languageChange() {
	qDebug("PreferencesDialog::languageChange");

	//QString system_font = system_font_label->text();
	int font_autoscale_item = font_autoscale_combo->currentItem();
	int font_autoload_item = font_autoload_combo->currentItem();
	int priority_item = priority_combo->currentItem();
	int mainwindow_resize = mainwindow_resize_combo->currentItem();
	int wheel_function = wheel_function_combo->currentItem();

	PreferencesDialogBase::languageChange();

	//system_font_label->setText( system_font );
	font_autoscale_combo->setCurrentItem(font_autoscale_item);
	font_autoload_combo->setCurrentItem(font_autoload_item);
	priority_combo->setCurrentItem(priority_item);
	mainwindow_resize_combo->setCurrentItem(mainwindow_resize);
	wheel_function_combo->setCurrentItem(wheel_function);
	initial_volume_label->setNum( initial_volume_slider->value() );
	sub_pos_label->setNum( sub_pos_slider->value() );

	int current_item = sections->currentItem();

	sections->changeItem( Images::icon("pref_general"),
                          tr( "General" ), General );
    sections->changeItem( Images::icon("pref_devices"),
                          tr( "Drives" ), Drives );
    sections->changeItem( Images::icon("pref_performance"),
                          tr( "Performance" ), Performance );
    sections->changeItem( Images::icon("pref_subtitles"), 
                          tr( "Subtitles" ), Subtitles );
    sections->changeItem( Images::icon("pref_advanced"),
                          tr( "Advanced" ), Advanced );
    sections->changeItem( Images::icon("pref_gui"),
                          tr( "Interface" ), Gui );
    sections->changeItem( Images::icon("input_devices"),
                          tr( "Keyboard and mouse" ), Mouse );

	sections->setCurrentItem(current_item);

	// Icons
	cdrom_drive_icon->setText("");
	dvd_drive_icon->setText("");
	mouse_icon->setText("");
	keyboard_icon->setText("");
	resize_window_icon->setText("");
	monitor_aspect_icon->setText("");
	volume_icon->setText("");

	cdrom_drive_icon->setPixmap( Images::icon("cdrom_drive") );
	dvd_drive_icon->setPixmap( Images::icon("dvd_drive") );
	mouse_icon->setPixmap( Images::icon("mouse") );
	keyboard_icon->setPixmap( Images::icon("keyboard") );
	resize_window_icon->setPixmap( Images::icon("resize_window") );
	monitor_aspect_icon->setPixmap( Images::icon("monitor") );
	volume_icon->setPixmap( Images::icon("speaker") );

	// Seek widgets
	seek1->setLabel( tr("Short jump") );
	seek2->setLabel( tr("Medium jump") );
	seek3->setLabel( tr("Long jump") );
	seek4->setLabel( tr("Mouse wheel jump") );

	seek1->setIcon( Images::icon("forward10s") );
	seek2->setIcon( Images::icon("forward1m") );
	seek3->setIcon( Images::icon("forward10m") );
	seek4->setIcon( Images::icon("mouse_small") );

	// Mouse function combos
	left_click_combo->changeItem( tr("None"), 0);
	double_click_combo->changeItem( tr("None"), 0);

	// Language combo
	//language_combo->changeItem( tr("<Auto>"), 0 );
	int language_item = language_combo->currentItem();
	createLanguageCombo();
	language_combo->setCurrentItem( language_item );

	// Iconset combo
	iconset_combo->changeItem( tr("Default"), 0 );

	// Encodings combo
	int font_encoding_item = font_encoding_combo->currentItem();
	font_encoding_combo->clear();
	encodings->retranslate();
	font_encoding_combo->insertStringList( encodings->list() );
	font_encoding_combo->setCurrentItem(font_encoding_item);

	createHelpMessages();

	setIcon( Images::icon("logo") );

	/*
	setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
	adjustSize();
	setFixedSize( sizeHint() );
	*/
	adjustSize();
}

void PreferencesDialog::setDrivers(InfoList vo_list, 
                                   InfoList ao_list)
{
	InfoList::iterator it;

	for ( it = vo_list.begin(); it != vo_list.end(); ++it ) {
		vo_combo->insertItem( (*it).name() );
		// Add directx:noaccel
		if ( (*it).name() == "directx" ) {
			vo_combo->insertItem( "directx:noaccel" );
		}
	}

	for ( it = ao_list.begin(); it != ao_list.end(); ++it ) {
		ao_combo->insertItem( (*it).name() );
	}

	drivers_set = TRUE;
}

void PreferencesDialog::accept() {
	hide();
	setResult( QDialog::Accepted );
	emit applyButtonPressed();
}

void PreferencesDialog::apply() {
	setResult( QDialog::Accepted );
	emit applyButtonPressed();
}


void PreferencesDialog::showSection(Section n) {
	changeSection(n);
	sections->setSelected( n, TRUE );
}

void PreferencesDialog::changeSection(int n) {
	qDebug("PreferencesDialog::changeSection: %d", n);

	switch (n) {
		case Subtitles:
			pages->raiseWidget( PAGE_SUBTITLES );
			break;
		case Advanced:
			pages->raiseWidget( PAGE_ADVANCED );
			break;
		case Performance:
			pages->raiseWidget( PAGE_PERFORMANCE );
			break;
		case Gui:
			pages->raiseWidget( PAGE_GUI );
			break;
		case Drives:
			pages->raiseWidget( PAGE_DRIVES );
			break;
		case Mouse:
			pages->raiseWidget( PAGE_MOUSE );
			break;
		default:
			pages->raiseWidget( PAGE_GENERAL );
	}
}



// General Page
void PreferencesDialog::setMplayerPath( QString path ) {
	mplayerbin_edit->setText( path );	
}

QString PreferencesDialog::mplayerPath() {
	return mplayerbin_edit->text();
}

void PreferencesDialog::setScreenshotDir( QString path ) {
	screenshot_edit->setText( path );
}

QString PreferencesDialog::screenshotDir() {
	return screenshot_edit->text();
}

void PreferencesDialog::setVO( QString vo_driver ) {
	vo_combo->setCurrentText( vo_driver );
}

void PreferencesDialog::setAO( QString ao_driver ) {
	ao_combo->setCurrentText( ao_driver );
}

QString PreferencesDialog::VO() {
	return vo_combo->currentText();
}

QString PreferencesDialog::AO() {
	return ao_combo->currentText();
}

void PreferencesDialog::setEq2(bool b) {
	eq2_check->setChecked(b);
}

bool PreferencesDialog::eq2() {
	return eq2_check->isChecked();
}

void PreferencesDialog::setSoftVol(bool b) {
	softvol_check->setChecked(b);
	amplification_label->setEnabled(b);
	softvol_max_spin->setEnabled(b);
}

bool PreferencesDialog::softVol() {
	return softvol_check->isChecked();
}

void PreferencesDialog::setAc3DTSPassthrough(bool b) {
	hwac3_check->setChecked(b);
}

bool PreferencesDialog::Ac3DTSPassthrough() {
	return hwac3_check->isChecked();
}

void PreferencesDialog::setVolNorm(bool b) {
	volnorm_check->setChecked(b);
}

bool PreferencesDialog::volNorm() {
	return volnorm_check->isChecked();
}

void PreferencesDialog::setPostprocessing(bool b) {
	postprocessing_check->setChecked(b);
}

bool PreferencesDialog::postprocessing() {
	return postprocessing_check->isChecked();
}


void PreferencesDialog::setAmplification(int n) {
	softvol_max_spin->setValue(n);
}

int PreferencesDialog::amplification() {
	return softvol_max_spin->value();
}


void PreferencesDialog::setRememberSettings(bool b) {
	remember_all_check->setChecked(b);
	rememberAllButtonToggled(b);
}

bool PreferencesDialog::rememberSettings() {
	return remember_all_check->isChecked();
}

void PreferencesDialog::setDontRememberTimePos(bool b) {
	dont_remember_time_check->setChecked(b);
}

bool PreferencesDialog::dontRememberTimePos() {
	return dont_remember_time_check->isChecked();
}

void PreferencesDialog::setStartInFullscreen(bool b) {
	start_fullscreen_check->setChecked(b);
}

bool PreferencesDialog::startInFullscreen() {
	return start_fullscreen_check->isChecked();
}

void PreferencesDialog::setAudioLang(QString lang) {
	audio_lang_edit->setText(lang);
}

QString PreferencesDialog::audioLang() {
	return audio_lang_edit->text();
}

void PreferencesDialog::setSubtitleLang(QString lang) {
	subtitle_lang_edit->setText(lang);
}

QString PreferencesDialog::subtitleLang() {
	return subtitle_lang_edit->text();
}


void PreferencesDialog::setSubtitlesOnScreenshots(bool b) {
	subtitles_on_screeshots_check->setChecked(b);
}

bool PreferencesDialog::subtitlesOnScreenshots() {
	return subtitles_on_screeshots_check->isChecked();
}


// Drive page
void PreferencesDialog::setDVDDevice( QString dir ) {
	dvd_device_combo->setCurrentText( dir );
}

QString PreferencesDialog::dvdDevice() {
	return dvd_device_combo->currentText();
}

void PreferencesDialog::setCDRomDevice( QString dir ) {
	cdrom_device_combo->setCurrentText( dir );
}

QString PreferencesDialog::cdromDevice() {
	return cdrom_device_combo->currentText();
}

// GUI Page
void PreferencesDialog::setDisableScreensaver(bool b) {
	screensaver_check->setChecked(b);
}

bool PreferencesDialog::disableScreensaver() {
	return screensaver_check->isChecked();
}

void PreferencesDialog::setResizeMethod(int v) {
	mainwindow_resize_combo->setCurrentItem(v);
}

int PreferencesDialog::resizeMethod() {
	return mainwindow_resize_combo->currentItem();
}

void PreferencesDialog::setMonitorAspect(QString asp) {
	if (asp.isEmpty())
		monitoraspect_combo->setCurrentText("Auto");
	else
		monitoraspect_combo->setCurrentText(asp);
}

QString PreferencesDialog::monitorAspect() {
	if (monitoraspect_combo->currentText() == "Auto") 
		return "";
	else
		return monitoraspect_combo->currentText();
}

void PreferencesDialog::setStyle(QString style) {
	if (style.isEmpty()) style="<default>";
	style_combo->setCurrentText(style);
}

QString PreferencesDialog::style() {
	QString s = style_combo->currentText();
	if (s=="<default>") s="";
	return s;
}

void PreferencesDialog::setUseSingleInstance(bool b) {
	single_instance_check->setChecked(b);
	singleInstanceButtonToggled(b);
}

bool PreferencesDialog::useSingleInstance() {
	return single_instance_check->isChecked();
}

void PreferencesDialog::clearRecents() {
	emit clearRecentsButtonPressed();
}

void PreferencesDialog::setServerPort(int port) {
	server_port_spin->setValue(port);
}

int PreferencesDialog::serverPort() {
	return server_port_spin->value();
}

void PreferencesDialog::setRecentsMaxItems(int n) {
	recents_max_items_spin->setValue(n);
}

int PreferencesDialog::recentsMaxItems() {
	return recents_max_items_spin->value();
}

void PreferencesDialog::setSeeking1(int n) {
	seek1->setTime(n);
}

int PreferencesDialog::seeking1() {
	return seek1->time();
}

void PreferencesDialog::setSeeking2(int n) {
	seek2->setTime(n);
}

int PreferencesDialog::seeking2() {
	return seek2->time();
}

void PreferencesDialog::setSeeking3(int n) {
	seek3->setTime(n);
}

int PreferencesDialog::seeking3() {
	return seek3->time();
}

void PreferencesDialog::setSeeking4(int n) {
	seek4->setTime(n);
}

int PreferencesDialog::seeking4() {
	return seek4->time();
}

void PreferencesDialog::setLeftClickFunction(QString f) {
	if (f.isEmpty()) {
		left_click_combo->setCurrentItem(0);
	} else {
		left_click_combo->setCurrentText(f);
	}
}

QString PreferencesDialog::leftClickFunction() {
	if (left_click_combo->currentItem()==0) {
		return "";
	} else {
		return left_click_combo->currentText();
	}
}

void PreferencesDialog::setDoubleClickFunction(QString f) {
	if (f.isEmpty()) {
		double_click_combo->setCurrentItem(0);
	} else {
		double_click_combo->setCurrentText(f);
	}
}

QString PreferencesDialog::doubleClickFunction() {
	if (double_click_combo->currentItem()==0) {
		return "";
	} else {
		return double_click_combo->currentText();
	}
}

void PreferencesDialog::setWheelFunction(int function) {
	wheel_function_combo->setCurrentItem(function);
}

int PreferencesDialog::wheelFunction() {
	return wheel_function_combo->currentItem();
}


void PreferencesDialog::setInitialVolume(int v) {
	initial_volume_slider->setValue(v);
	initial_volume_label->setNum(v);
}

int PreferencesDialog::initialVolume() {
	return initial_volume_slider->value();
}

void PreferencesDialog::setLanguage(QString lang) {
	if (lang.isEmpty()) {
		language_combo->setCurrentItem(0);
	}
	else {
		int pos = languages_list.findIndex( lang );
		qDebug("lang: '%s' pos: %d", lang.utf8().data(), pos);
		if (pos != -1) 
			language_combo->setCurrentItem( pos +1 ); // 0 = Auto
		else
			language_combo->setCurrentText(lang);
	}
}

QString PreferencesDialog::language() {
	if (language_combo->currentItem()==0) 
		return "";
	else 
		//return language_combo->currentText();
		return languages_list[ language_combo->currentItem() -1 ]; // 0 = Auto
}

void PreferencesDialog::setIconSet(QString set) {
	if (set.isEmpty())
		iconset_combo->setCurrentItem(0);
	else
		iconset_combo->setCurrentText(set);
}

QString PreferencesDialog::iconSet() {
	if (iconset_combo->currentItem()==0) 
		return "";
	else
		return iconset_combo->currentText();
}


void PreferencesDialog::singleInstanceButtonToggled(bool b) {
	server_port_spin->setEnabled(b);
	port_label->setEnabled(b);
}


// General Page stuff
void PreferencesDialog::selectMplayerPath() {
	QString	s = MyFileDialog::getOpenFileName(
	                    mplayerbin_edit->text(),
#ifdef Q_OS_WIN
    	                tr("Executables") +" (*.exe)",
#else
    	                tr("All files") +" (*)",
#endif
        	            this, "select_mplayer_dialog",
            	        tr("Select the mplayer executable") );

	if (!s.isEmpty()) {
		mplayerbin_edit->setText(s);
	}
}


void PreferencesDialog::selectScreenshotDirectory() {
	QString s = MyFileDialog::getExistingDirectory(
                    screenshot_edit->text(),
                    this,
                    "select_screenshot_directory",
                    tr("Select a directory"),
                    TRUE );
	if (!s.isEmpty()) {
		screenshot_edit->setText(s);
	}
}


void PreferencesDialog::rememberAllButtonToggled(bool b) {
	dont_remember_time_check->setEnabled(b);
}


// Subtitles Page
void PreferencesDialog::setFontName(QString font_name) {
	system_font_edit->setText( font_name );
}

void PreferencesDialog::setFontFile(QString font_file) {
	ttf_font_edit->setText( font_file );
}

void PreferencesDialog::setUseFontconfig(bool b) {
	if (b) {
		system_font_button->setChecked(TRUE);
		ttf_font_button->setChecked(FALSE);
		systemFontRadioButtonSelected();
	} else {
		ttf_font_button->setChecked(TRUE);
		system_font_button->setChecked(FALSE);
		ttfFontRadioButtonSelected();
	}
}

QString PreferencesDialog::fontName() {
	return system_font_edit->text();
}

QString PreferencesDialog::fontFile() {
	return ttf_font_edit->text();
}

bool PreferencesDialog::useFontconfig() {
	return system_font_button->isChecked();
}

void PreferencesDialog::setFontAutoscale(int n) {
	font_autoscale_combo->setCurrentItem(n);
}

int PreferencesDialog::fontAutoscale() {
	return font_autoscale_combo->currentItem();
}

void PreferencesDialog::setFontTextscale(int n) {
	font_text_scale->setValue(n);
}

int PreferencesDialog::fontTextscale() {
	return font_text_scale->value();
}

void PreferencesDialog::setAutoloadSub(bool v) {
	font_autoload_check->setChecked(v);
}

bool PreferencesDialog::autoloadSub() {
	return font_autoload_check->isChecked();
}

void PreferencesDialog::setFontEncoding(QString s) {
	int n = encodings->findEncoding( s );
	if (n != -1) 
		font_encoding_combo->setCurrentItem(n);
	else
		font_encoding_combo->setCurrentText(s);
}

QString PreferencesDialog::fontEncoding() {
	qDebug("PreferencesDialog::fontEncoding");
	QString res = encodings->parseEncoding( font_encoding_combo->currentText() );
	qDebug(" * res: '%s'", res.utf8().data() );
	return res;
}

void PreferencesDialog::setSubPos(int pos) {
	sub_pos_slider->setValue(pos);
}

int PreferencesDialog::subPos() {
	return sub_pos_slider->value();
}

void PreferencesDialog::setUseFontASS(bool v) {
	font_ass_check->setChecked(v);
	assButtonToggled(v);
}

bool PreferencesDialog::useFontASS() {
	return font_ass_check->isChecked();
}

void PreferencesDialog::setAssColor( unsigned int color ) {
	ass_color = color;
	#if QT_VERSION >= 0x040200
	asscolor_button->setStyleSheet( "border-width: 1px; border-style: solid; border-color: #000000; background: #" + Helper::colorToRGB(ass_color) + ";");
	#else
	asscolor_button->setPaletteBackgroundColor( ass_color );
	#endif
}

unsigned int PreferencesDialog::assColor() {
	return ass_color;
}

void PreferencesDialog::setAssBorderColor( unsigned int color ) {
	ass_border_color = color;

	#if QT_VERSION >= 0x040200
	assbordercolor_button->setStyleSheet( "border-width: 1px; border-style: solid; border-color: #000000; background: #" + Helper::colorToRGB(ass_border_color) + ";");
	#else
	assbordercolor_button->setPaletteBackgroundColor( ass_border_color );
	#endif
}

unsigned int PreferencesDialog::assBorderColor() {
	return ass_border_color;
}

void PreferencesDialog::setAssStyles(QString styles) {
	ass_styles_edit->setText(styles);
}

QString PreferencesDialog::assStyles() {
	return ass_styles_edit->text();
}

void PreferencesDialog::setFontFuzziness(int n) {
	font_autoload_combo->setCurrentItem(n);
}

int PreferencesDialog::fontFuzziness() {
	return font_autoload_combo->currentItem();
}

void PreferencesDialog::setUseSubfont(bool b) {
	subfont_check->setChecked(b);
}

bool PreferencesDialog::useSubfont() {
	return subfont_check->isChecked();
}


// Subtitles Page stuff
void PreferencesDialog::systemFontRadioButtonSelected() {
	qDebug("PreferencesDialog::systemFontRadioButtonSelected");

	system_font_edit->setEnabled(TRUE);
	select_system_font_button->setEnabled(TRUE);

	ttf_font_edit->setEnabled(FALSE);
	select_ttf_font_button->setEnabled(FALSE);

}

void PreferencesDialog::ttfFontRadioButtonSelected() {
	qDebug("PreferencesDialog::ttfFontRadioButtonSelected");

	ttf_font_edit->setEnabled(TRUE);
	select_ttf_font_button->setEnabled(TRUE);

	system_font_edit->setEnabled(FALSE);
	select_system_font_button->setEnabled(FALSE);
}

void PreferencesDialog::assButtonToggled(bool b) {
	asscolor_label->setEnabled(b);
	asscolor_button->setEnabled(b);
	assbordercolor_label->setEnabled(b);
	assbordercolor_button->setEnabled(b);
	ass_styles_label->setEnabled(b);
	ass_styles_edit->setEnabled(b);
	styles_desc_label->setEnabled(b);
	ass_styles_line->setEnabled(b);
}

void PreferencesDialog::selectSystemFont() {
	bool ok;
    QFont font = QFontDialog::getFont(
                    &ok, QFont( system_font_edit->text(), 10 ), this );
    if ( ok ) {
        // font is set to the font the user selected
		system_font_edit->setText( font.family() );
    } else {
        // the user canceled the dialog; font is set to the initial
        // value, in this case Helvetica [Cronyx], 10
    }

}

void PreferencesDialog::selectTtfFont() {
	QString s = "";

#ifdef Q_OS_WIN
	// The windows native file dialog doesn't allow to select a ttf file! :-O
	// So we use a Qt one.
	QFileDialog fd( QFileInfo(ttf_font_edit->text()).dirPath(TRUE),
                    tr("Truetype Fonts") +" (*.ttf)",
                    this, "open file dialog", TRUE );

	if ( fd.exec() == QDialog::Accepted ) {
		s = fd.selectedFile();
	}
#else
	s = MyFileDialog::getOpenFileName(
	                    ttf_font_edit->text(),
    	                tr("Truetype Fonts") + " (*.ttf)",
        	            this, "open file dialog",
            	        tr("Choose a ttf file") );
#endif
	if (!s.isEmpty()) {
		ttf_font_edit->setText(s);
	}
}

void PreferencesDialog::selectAssColor() {
	QColor c = QColorDialog::getColor ( ass_color, this );
	if (c.isValid()) {
		setAssColor( c.rgb() );
	}
}

void PreferencesDialog::selectAssBorderColor() {
	QColor c = QColorDialog::getColor ( ass_border_color, this );
	if (c.isValid()) {
		setAssBorderColor( c.rgb() );
	}
}

// Advanced Page
void PreferencesDialog::setClearBackground(bool b) {
	not_clear_background_check->setChecked(!b);
}

bool PreferencesDialog::clearBackground() {
	return !not_clear_background_check->isChecked();
}

void PreferencesDialog::setUseMplayerWindow(bool v) {
	mplayer_use_window_check->setChecked(v);
}

bool PreferencesDialog::useMplayerWindow() {
	return mplayer_use_window_check->isChecked();
}


void PreferencesDialog::setCacheEnabled(bool b) {
	use_cache_check->setChecked(b);

	cache_spin->setEnabled(b);
	cache_size_label->setEnabled(b);
	cache_kb_label->setEnabled(b);
}

bool PreferencesDialog::cacheEnabled() {
	return use_cache_check->isChecked();
}

void PreferencesDialog::setCache(int n) {
	cache_spin->setValue(n);
}

int PreferencesDialog::cache() {
	return cache_spin->value();
}

void PreferencesDialog::setMplayerAdditionalArguments(QString args) {
	mplayer_args_edit->setText(args);
}

QString PreferencesDialog::mplayerAdditionalArguments() {
	return mplayer_args_edit->text();
}

void PreferencesDialog::setMplayerAdditionalVideoFilters(QString s) {
	mplayer_vfilters_edit->setText(s);
}

QString PreferencesDialog::mplayerAdditionalVideoFilters() {
	return mplayer_vfilters_edit->text();
}

void PreferencesDialog::setMplayerAdditionalAudioFilters(QString s) {
	mplayer_afilters_edit->setText(s);
}

QString PreferencesDialog::mplayerAdditionalAudioFilters() {
	return mplayer_afilters_edit->text();
}

void PreferencesDialog::setColorKey(unsigned int c) {
	QString color = QString::number(c, 16);
	while (color.length() < 6) color = "0"+color;
	colorkey_view->setText( "#" + color );
}

unsigned int PreferencesDialog::colorKey() {
	QString c = colorkey_view->text();
	if (c.startsWith("#")) c = c.mid(1);

	bool ok;
	unsigned int color = c.toUInt(&ok, 16);

	if (!ok) 
		qWarning("PreferencesDialog::colorKey: cannot convert color to uint");

	qDebug("PreferencesDialog::colorKey: color: %s", QString::number(color,16).utf8().data() );

	return color;
}

void PreferencesDialog::selectColorkey() {
	//bool ok;
	//int color = colorkey_view->text().toUInt(&ok, 16);
	QColor color( colorkey_view->text() );
	QColor c = QColorDialog::getColor ( color, this );
	if (c.isValid()) {
		//colorkey_view->setText( QString::number( c.rgb(), 16 ) );
		colorkey_view->setText( c.name() );
	}
}

// Performance Page
void PreferencesDialog::setPriority(int n) {
	priority_combo->setCurrentItem(n);
}

int PreferencesDialog::priority() {
	return priority_combo->currentItem();
}

void PreferencesDialog::setFrameDrop(bool b) {
	framedrop_check->setChecked(b);
}

bool PreferencesDialog::frameDrop() {
	return framedrop_check->isChecked();
}

void PreferencesDialog::setHardFrameDrop(bool b) {
	hardframedrop_check->setChecked(b);
}

bool PreferencesDialog::hardFrameDrop() {
	return hardframedrop_check->isChecked();
}


void PreferencesDialog::setAutoq(int n) {
	autoq_spin->setValue(n);
}

int PreferencesDialog::autoq() {
	return autoq_spin->value();
}

void PreferencesDialog::setAutoSyncFactor(int factor) {
	autosync_spin->setValue(factor);
}

int PreferencesDialog::autoSyncFactor() {
	return autosync_spin->value();
}

void PreferencesDialog::setAutoSyncActivated(bool b) {
	autosync_check->setChecked(b);
	autoSyncButtonToggled(b);
}

bool PreferencesDialog::autoSyncActivated() {
	return autosync_check->isChecked();
}

void PreferencesDialog::autoSyncButtonToggled(bool b) {
	factor_label->setEnabled(b);
	autosync_spin->setEnabled(b);
}

void PreferencesDialog::setFastChapterSeeking(bool b) {
	fast_chapter_check->setChecked(b);
}

bool PreferencesDialog::fastChapterSeeking() {
	return fast_chapter_check->isChecked();
}

void PreferencesDialog::setFastAudioSwitching(bool b) {
	fastaudioswitching_check->setChecked(b);
}

bool PreferencesDialog::fastAudioSwitching() {
	return fastaudioswitching_check->isChecked();
}


// Log options
void PreferencesDialog::setLogMplayer(bool b) {
	log_mplayer_check->setChecked(b);
}

bool PreferencesDialog::logMplayer() {
	return log_mplayer_check->isChecked();
}

void PreferencesDialog::setLogSmplayer(bool b) {
	log_smplayer_check->setChecked(b);
}

bool PreferencesDialog::logSmplayer() {
	return log_smplayer_check->isChecked();
}

void PreferencesDialog::setLogFilter(QString filter) {
	log_filter_edit->setText(filter);
}

QString PreferencesDialog::logFilter() {
	return log_filter_edit->text();
}


// MPlayer language page
void PreferencesDialog::setEndOfFileText(QString t) {
	endoffile_combo->setCurrentText(t);
}

QString PreferencesDialog::endOfFileText() {
	return endoffile_combo->currentText();
}

void PreferencesDialog::setNoVideoText(QString t) {
	novideo_combo->setCurrentText(t);
}

QString PreferencesDialog::noVideoText() {
	return novideo_combo->currentText();
}


void PreferencesDialog::createHelpMessages() {
	// General tab
	QWhatsThis::add(mplayerbin_edit,
		tr("Here you must specify the mplayer "
           "executable that smplayer will use.<br>"
           "smplayer requires at least mplayer 1.0rc1 (svn recommended).<br>"
           "<b>If this setting is wrong, smplayer won't "
           "be able to play anything!</b>") );

	QWhatsThis::add(screenshot_edit,
		tr("Here you can specify a folder where the screenshots taken by "
           "smplayer will be stored. If this field is empty the "
           "screenshot feature will be disabled.") );

	QWhatsThis::add(vo_combo,
		tr("Select the video output driver. Usually xv (linux) "
           "and directx (windows) provide the best performance.") );

	QWhatsThis::add(ao_combo,
		tr("Select the audio output driver.") );

	QWhatsThis::add(eq2_check,
		tr("You can check this option if video equalizer is not supported by "
           "your graphic card or the selected video output driver.<br>"
           "<b>Note:</b> this option can be incompatible "
           "with some video output drivers.") );

	QWhatsThis::add(softvol_check,
		tr("Check this option to use the software mixer, instead of "
           "using the sound card mixer.") );

	QWhatsThis::add(remember_all_check,
		tr("Usually smplayer will remember the settings for each file you "
           "play (audio track selected, volume, filters...). Uncheck this "
           "option if you don't like this feature.") );

	QWhatsThis::add(dont_remember_time_check,
		tr("If you check this option, smplayer will play all files from "
           "the beginning.") );

	QWhatsThis::add(start_fullscreen_check,
		tr("If this option is checked, all videos will start to play in "
           "fullscreen mode.") );

	QWhatsThis::add(screensaver_check,
		tr("Check this option to disable the screensaver while playing.<br>"
           "The screensaver will enabled again when play finishes.<br>"
           "<b>Note:</b> This option works only in X11 and Windows.") );

	QWhatsThis::add(audio_lang_edit,
		tr("Here you can type your preferred language for the audio streams. "
           "When a media with multiple audio streams is found, smplayer will "
           "try to use your preferred language.<br>"
           "This only will work with media that offer info about the language "
           "of the audio streams, like DVDs or mkv files.<br>"
           "This field accepts regular expressions. Example: <b>es|esp|spa</b> "
           "will select the audio track if it matches with <i>es</i>, "
           "<i>esp</i> or <i>spa</i>.") );

	QWhatsThis::add(subtitle_lang_edit,
		tr("Here you can type your preferred language for the subtitle stream. "
           "When a media with multiple subtitle streams is found, smplayer will "
           "try to use your preferred language.<br>"
           "This only will work with media that offer info about the language "
           "of the subtitle streams, like DVDs or mkv files.<br>"
           "This field accepts regular expressions. Example: <b>es|esp|spa</b> "
           "will select the subtitle stream if it matches with <i>es</i>, "
           "<i>esp</i> or <i>spa</i>.") );

	// Subtitles tab
	QWhatsThis::add(sub_pos_slider,
		tr("This option specifies the position of the subtitles over the "
           "video window. <i>100</i> means the bottom, while <i>0</i> means "
           "the top." ) );

	QWhatsThis::add(ass_styles_edit,
		tr("Here you can override styles for SSA/ASS subtitles. "
           "It can be also used for fine-tuning the rendering of srt and sub "
           "subtitles by the SSA/ASS library.<br>"
           "Example: <b>Bold=1,Outline=2,Shadow=2</b>"));

	// Performance tab
	QWhatsThis::add(priority_combo,
		tr("Set process priority for mplayer according to the predefined "
           "priorities available under Windows.<br>"
           "<b>WARNING:</b> Using realtime priority can cause system lockup.")
#ifndef Q_OS_WIN
        + tr("<br><b>Note:</b> This option is for Windows only.") 
#endif
		);

	QWhatsThis::add(cache_spin,
		tr("This option specifies how much memory (in kBytes) to use when "
           "precaching a file or URL. Especially useful on slow media.") );

	QWhatsThis::add(framedrop_check,
		tr("Skip displaying some frames to maintain A/V sync on slow systems." ) );

	QWhatsThis::add(hardframedrop_check,
		tr("More intense frame dropping (breaks decoding). "
           "Leads to image distortion!") );

	QWhatsThis::add(autosync_check,
		tr("Gradually adjusts the A/V sync based on audio delay "
           "measurements.") );

	QWhatsThis::add(autoq_spin,
		tr("Dynamically changes the level of postprocessing depending on the "
           "available spare CPU time. The number you specify will be the "
           "maximum level used. Usually you can use some big number.") );

	// Log tab
	QWhatsThis::add(log_mplayer_check,
		tr("If checked, smplayer will store the output of mplayer "
           "(you can see it in <b>Options->View logs->mplayer</b>). "
           "In case of problems this log can contain important information, "
           "so it's recommended to keep this option checked.") );

	QWhatsThis::add(log_smplayer_check,
		tr("If this option is checked, smplayer will store the debugging "
           "messages that smplayer outputs "
           "(you can see the log in <b>Options->View logs->smplayer</b>). "
           "This information can be very useful for the developer in case "
           "you find a bug." ) );

	QWhatsThis::add(log_filter_edit,
		tr("This option allows to filter the smplayer messages that will "
           "be stored in the log. Here you can write any regular expression.<br>"
           "For instance: <i>^Core::.*</i> will display only the lines "
           "starting with <i>Core::</i>") );

	// Advanced tab
	QWhatsThis::add(not_clear_background_check,
		tr("Checking this option may reduce flickering, but it also might "
           "produce that the video won't be displayed properly.") );
}
