#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - "neutral" messages for US English
 *   
 *   This module provides standard library messages with a parser/narrator
 *   that's as invisible (neutral) as possible.  These messages are
 *   designed to reduce the presence of the computer as mediator in the
 *   story, to give the player the most direct contact that we can with
 *   the scenario.
 *   
 *   The parser almost always refers to itself in the third person (by
 *   calling itself something like "this story") rather than in the first
 *   person, and, whenever possible, avoids referring to itself in the
 *   first place.  Our ideal phrasing is either second-person, describing
 *   things directly in terms of the player character's experience, or
 *   "no-person," simply describing things without mentioning the speaker
 *   or listener at all.  For example, rather than saying "I don't see
 *   that here," we say "you don't see that here," or "that's not here."
 *   We occasionally stray from this ideal where achieving it would be too
 *   awkward.
 *   
 *   In the earliest days of adventure games, the parser was usually a
 *   visible presence: the early parsers frequently reported things in the
 *   first person, and some even had specific personalities.  This
 *   conspicuous parser style has become less prevalent in modern games,
 *   though, and authors now usually prefer to treat the parser as just
 *   another part of the user interface, which like all good UI's is best
 *   when the user doesn't notice it.  
 */

#include "adv3.h"
#include "en_us.h"


/* ------------------------------------------------------------------------ */
/*
 *   Library Messages 
 */
libMessages: MessageHelper
    /*
     *   The pronoun to use for the object form of the personal
     *   interrogative pronoun.  The "right" word for this usage is
     *   "whom," but "whom" sounds too formal to some people.  To change
     *   this to "who," change the property via a 'modify' statement.  
     */
    whomPronoun = 'whom'

    /*
     *   Flag: offer an explanation of the "OOPS" command when it first
     *   comes up.  We'll only show this the first time the player enters
     *   an unknown word.  If you never want to offer this message at all,
     *   simply set this flag to nil initially.
     *   
     *   See also oopsNote() below.  
     */
    offerOopsNote = true

    /* 
     *   some standard commands for insertion into <a> tags - these are in
     *   the messages so they can translated along with the command set
     */
    commandLookAround = 'look around'
    commandFullScore = 'full score'
    
    /* announce a complete remapped action */
    announceRemappedAction(action)
    {
        return '\n<.assume>' + action.getParticiplePhrase() + '<./assume>\n';
    }

    /*
     *   Get a string to announce an implicit action.  This announces the
     *   current global action.  
     */
    announceImplicitAction(action)
    {
        /* 
         *   show the present participle form of the current action,
         *   within parens, on a line by itself 
         */
        return '\n<.assume>first ' + action.getParticiplePhrase()
            + '<./assume>\n';
    }

    /*
     *   Get a string to announce that we're implicitly moving an object
     *   to a bag of holding to make room for taking something new.  
     */
    announceMoveToBag(action)
    {
        /* 
         *   show the present participle form of the action, and add our
         *   explanation 
         */
        return '\n<.assume>first ' + action.getParticiplePhrase()
            + ' to make room<./assume>\n';
    }

    /* show a library credit (for a CREDITS listing) */
    showCredit(name, byline) { "<<name>> <<byline>>"; }

    /* show a library version number (for a VERSION listing) */
    showVersion(name, version) { "<<name>> version <<version>>"; }

    /* there's no "about" information in this game */
    noAboutInfo = "<.parser>This story has no ABOUT
                   information.<./parser> "

    /* 
     *   Show a list state name - this is extra state information that we
     *   show for an object in a listing involving the object.  For
     *   example, a light source might add a state like "(providing
     *   light)".  We simply show the list state name in parentheses.  
     */
    showListState(state) { " (<<state>>)"; }

    /* a set of equivalents are all in a given state */
    allInSameListState(num, stateName)
        { " (<<num == 2 ? 'both' : 'all'>> <<stateName>>)"; }

    /* show a status line addendum for actor postures */
    statusStanding = ""
    statusSitting = " (sitting)"
    statusLying = " (lying down)"

    /* show a status line addendum: standing in/on something */
    statusStandingOn(obj) { " (standing <<obj.actorInName>>)"; }

    /* show a status line addendum: seated in/on something */
    statusSittingOn(obj) { " (sitting <<obj.actorInName>>)"; }

    /* show a status line addendum: lying in/on something */
    statusLyingOn(obj) { " (lying <<obj.actorInName>>)"; }

    /* generic short description of a dark room */
    roomDarkName = 'In the dark'

    /* generic long description of a dark room */
    roomDarkDesc = "It's pitch black. "

    /* generic long description of a Thing from a distance */
    distantThingDesc(obj)
    {
        gMessageParams(obj);
        "{It's obj} too far away to make out any detail. ";
    }

    /* generic long description of a Thing under obscured conditions */
    obscuredThingDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "{You/he} can't make out any detail through {the obs/him}. ";
    }

    /* generic "listen" desription of a Thing */
    thingSoundDesc(obj)
        { "{You/he} hear{s} nothing out of the ordinary. "; }

    /* generic "listen" description of a Thing at a distance */
    distantThingSoundDesc(obj)
        { "{You/he} can't hear any detail from this distance. "; }

    /* generic obscured "listen" description */
    obscuredThingSoundDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "{You/he} can't hear any detail through {the obs/him}. ";
    }

    /* generic "smell" description of a Thing */
    thingSmellDesc(obj)
        { "{You/he} smell{s} nothing out of the ordinary. "; }

    /* generic "smell" description of a Thing at a distance */
    distantThingSmellDesc(obj)
        { "{You/he} can't smell much at this distance. "; }

    /* generic obscured "smell" description */
    obscuredThingSmellDesc(obj, obs)
    {
        gMessageParams(obj, obs);
        "{You/he} can't smell much through {the obs/him}. ";
    }

    /* generic "taste" description of a Thing */
    thingTasteDesc(obj)
        { "{You/he} taste{s} nothing out of the ordinary. "; }

    /* generic "feel" description of a Thing */
    thingFeelDesc(obj)
        { "{You/he} feel{s} nothing out of the ordinary. "; }

    /* obscured "read" description */
    obscuredReadDesc(obj)
    {
        gMessageParams(obj);
        return '{You/he} can\'t see {that obj/him} well enough to
            read {it/him}. ';
    }

    /* dim light "read" description */
    dimReadDesc(obj)
    {
        gMessageParams(obj);
        return 'There\'s not enough light to read {that obj/him}. ';
    }

    /* lit/unlit match description */
    litMatchDesc(obj) { "\^<<obj.nameIs>> lit. "; }
    unlitMatchDesc(obj) { "\^<<obj.nameIs>> an ordinary match. "; }

    /* lit candle description */
    litCandleDesc(obj) { "\^<<obj.nameIs>> lit. "; }

    /* 
     *   Prepositional phrases for putting objects into different types of
     *   objects. 
     */
    putDestContainer(obj) { return 'into ' + obj.theNameObj; }
    putDestSurface(obj) { return 'onto ' + obj.theNameObj; }
    putDestFloor(obj) { return 'to ' + obj.theNameObj; }

    /* the list separator character in the middle of a list */
    listSepMiddle = ", "

    /* the list separator character for a two-element list */
    listSepTwo = " and "

    /* the list separator for the end of a list of at least three elements */
    listSepEnd = ", and "

    /* 
     *   the list separator for the middle of a long list (a list with
     *   embedded lists not otherwise set off, such as by parentheses) 
     */
    longListSepMiddle = "; "

    /* the list separator for a two-element list of sublists */
    longListSepTwo = ", and "

    /* the list separator for the end of a long list */
    longListSepEnd = "; and "

    /* status suffixes for listing nested rooms an actor is occupying */
    withActorStanding(room, actor)
        { " (<<room.actorInPrep>> which <<actor.nameIs>> standing)"; }
    withActorSitting(room, actor)
        { " (<<room.actorInPrep>> which <<actor.nameIs>> sitting)"; }
    withActorLying(room, actor)
        { " (<<room.actorInPrep>> which <<actor.nameIs>> lying)"; }

    /*
     *   Score ranking list.  Games will almost always want to override
     *   this with a customized list.  For each entry in the list, the
     *   first element is the minimum score for the rank, and the second
     *   is a string describing the rank.  The ranks should be given in
     *   ascending order, since we simply search the list for the first
     *   item whose minimum score is greater than our score, and use the
     *   preceding item.
     *   
     *   If this is set to nil, which it is by default, we'll simply skip
     *   score ranks entirely.  
     */
    scoreRankTable = nil

    /* show the full message for a given score rank string */
    showScoreRankMessage(msg)
    {
        "This makes {you/him} <<msg>>. ";
    }

    /* show the basic score message */
    showScoreMessage(points, maxPoints, turns)
    {
        "{Your} score is <<points>> of a possible <<maxPoints>>,
        in <<turns>> move<<turns == 1 ? '' : 's'>>. ";
    }

    /* 
     *   show the list prefix for the full score listing; this is shown on
     *   a line by itself before the list of full score items, shown
     *   indented and one item per line 
     */
    showFullScorePrefix = "{Your/his} score consists of:"

    /*
     *   show the item prefix, with the number of points, for a full score
     *   item - immediately after this is displayed, we'll display the
     *   description message for the achievement 
     */
    fullScoreItemPoints(points)
    {
        "<<points>> point<<points == 1 ? '' : 's'>> for ";
    }

    /* score change - first notification */
    firstScoreChange(delta)
    {
        "<.commandsep><.notification><<
            basicScoreChange(delta)>><./notification>
        \n<.notification>If you would prefer not to see notifications about
        score changes in the future, type <<
        aHref('notify off', 'NOTIFY OFF', 'Turn off score notifications')
        >>.<./notification> ";
    }

    /* score change - notification other than the first time */
    scoreChange(delta)
    {
        "<.commandsep><.notification><<
        basicScoreChange(delta)>><./notification> ";
    }

    /* 
     *   basic score change notification message - this is an internal
     *   service routine for scoreChange and firstScoreChange 
     */
    basicScoreChange(delta)
    {
        "{Your} <<aHref(commandFullScore, 'score', 'Show full score')>>
        has just gone <<delta > 0 ? 'up' : 'down'>>
        by <<spellInt(delta > 0 ? delta : -delta)>>
        point<<delta is in (1, -1) ? '' : 's'>>.";
    }

    /* get the string to display for a footnote reference */
    footnoteRef(num)
    {
        local str;

        /* set up a hyperlink for the note that enters the "note n" command */
        str = '<sup>[<a href="footnote ' + num + '"><.a>';

        /* show the footnote number in square brackets */
        str += num;
        
        /* end the hyperlink */
        str += '<./a></a>]</sup>';

        /* return the text */
        return str;
    }

    /* first footnote notification */
    firstFootnote()
    {
        "<.commandsep><.notification>When you see a number in
        [square brackets], it indicates a note that you can read with
        the FOOTNOTE command - for example, <<
        aHref('footnote 1', 'FOOTNOTE 1', 'Show footnote [1]')>>.
        Type <<aHref('footnotes', 'FOOTNOTES',
                     'Control footnote appearance')>> for information on
        controlling whether or not footnote references
        appear.<./notification> ";
    }

    /* there is no such footnote as the given number */
    noSuchFootnote(num)
    {
        "The story has never referred to any such footnote. ";
    }

    /* show the current footnote status */
    showFootnoteStatus(stat)
    {
        "The current setting is FOOTNOTES ";
        switch(stat)
        {
        case FootnotesOff:
            "OFF, which hides all footnote references.
            Type <<aHref('footnotes medium', 'FOOTNOTES MEDIUM',
                         'Set footnotes to Medium')>> to
            show references to footnotes except those you've
            already seen, or <<aHref('footnotes full', 'FOOTNOTES FULL',
                                     'Set footnotes to Full')>>
            to show all footnote references. ";
            break;

        case FootnotesMedium:
            "MEDIUM, which shows references to footnotes except
            for those you've already read.  Type
            <<aHref('footnotes off', 'FOOTNOTES OFF',
                    'Turn off footnotes')>> to hide
            footnote references entirely, or <<aHref(
                'footnotes full', 'FOOTNOTES FULL',
                'Set footnotes to Full')>> to show every reference, even to
            notes you've already read. ";
            break;

        case FootnotesFull:
            "FULL, which shows every footnote reference, even to
            notes you've already read.  Type <<aHref('footnotes medium',
            'FOOTNOTES MEDIUM', 'Set footnotes to Medium')>> to show
            only references to notes you
            haven't yet read, or <<aHref('footnotes off', 'FOOTNOTES OFF',
                'Turn off footnotes')>>
            to hide footnote references entirely. ";
            break;
        }
    }

    /* acknowledge a change in the footnote status */
    acknowledgeFootnoteStatus(stat)
    {
        "The setting is now FOOTNOTES <<
        stat == FootnotesOff ? 'OFF'
        : stat == FootnotesMedium ? 'MEDIUM'
        : 'FULL' >>. ";
    }

    /* 
     *   Show the main command prompt.
     *   
     *   'which' is one of the rmcXxx phase codes indicating what kind of
     *   command we're reading.  This default implementation shows the
     *   same prompt for every type of input, but games can use the
     *   'which' value to show different prompts for different types of
     *   queries, if desired.  
     */
    mainCommandPrompt(which) { "<.p>&gt;"; }

    /* show the response to an empty command line */
    emptyCommandResponse = "<.parser>I beg your pardon?<./parser> "

    /* invalid token (i.e., punctuation) in command line */
    invalidCommandToken(ch)
    {
        "<.parser>The story doesn't know how to use the character
        &lsquo;<<ch>>&rsquo; in a command.<./parser> ";
    }

    /* 
     *   Command group prefix - this is displayed after a command line and
     *   before the first command results shown after the command line.
     *   
     *   By default, we'll show the "zero-space paragraph" marker, which
     *   acts like a paragraph break in that it swallows up immediately
     *   following paragraph breaks, but doesn't actually add any space.
     *   This will ensure that we don't add any space between the command
     *   input line and the next text.  
     */
    commandResultsPrefix = '<.p0>'

    /*
     *   Command "interruption" group prefix.  This is displayed after an
     *   interrupted command line - a command line editing session that
     *   was interrupted by a timeout event - just before the text that
     *   interrupted the command line.
     *   
     *   By default, we'll show a paragraph break here, to set off the
     *   interrupting text from the command line under construction.  
     */
    commandInterruptionPrefix = '<.p>'

    /* 
     *   Command separator - this is displayed after the results from a
     *   command when another command is about to be executed without any
     *   more user input.  That is, when a command line contains more than
     *   one command, this message is displayed between each successive
     *   command, to separate the results visually.
     *   
     *   This is not shown before the first command results after a
     *   command input line, and is not shown after the last results
     *   before a new input line.  Furthermore, this is shown only between
     *   adjacent commands for which output actually occurs; if a series
     *   of commands executes without any output, we won't show any
     *   separators between the silent commands.
     *   
     *   By default, we'll just start a new paragraph.  
     */
    commandResultsSeparator = '<.p>'

    /*
     *   "Complex" result separator - this is displayed between a group of
     *   messages for a "complex" result set and adjoining messages.  A
     *   command result list is "complex" when it's built up out of
     *   several generated items, such as object identification prefixes
     *   or implied command prefixes.  We use additional visual separation
     *   to set off these groups of messages from adjoining messages,
     *   which is especially important for commands on multiple objects,
     *   where we would otherwise have several results shown together.  By
     *   default, we use a paragraph break.  
     */
    complexResultsSeparator = '<.p>'

    /*
     *   Internal results separator - this is displayed to visually
     *   separate the results of an implied command from the results for
     *   the initiating command, which are shown after the results from
     *   the implied command.  By default, we show a line break. 
     */
    internalResultsSeparator = '\n'

    /*
     *   Command results suffix - this is displayed just before a new
     *   command line is about to be read if any command results have been
     *   shown since the last command line.
     *   
     *   By default, we'll show nothing extra.  
     */
    commandResultsSuffix = ''

    /*
     *   Empty command results - this is shown when we read a command line
     *   and then go back and read another without having displaying
     *   anything.
     *   
     *   By default, we'll return a message indicating that nothing
     *   happened.  
     */
    commandResultsEmpty = 'Nothing obvious happens.<.p>'

    /*
     *   Intra-command report separator.  This is used to separate report
     *   messages within a single command's results.  By default, we show
     *   a paragraph break.  
     */
    intraCommandSeparator = '<.p>'

    /* 
     *   separator for "smell" results - we ordinarily show each item's
     *   odor description as a separate paragraph 
     */
    smellDescSeparator()
    {
        "<.p>";
    }

    /*
     *   separator for "listen" results 
     */
    soundDescSeparator()
    {
        "<.p>";
    }

    /* a command was issued to a non-actor */
    cannotTalkTo(targetActor, issuingActor)
    {
        "\^<<targetActor.nameIs>> not something <<issuingActor.itNom>>
        can talk to. ";
    }

    /*
     *   Show a note about the OOPS command.  This is, by default, added
     *   to the "I don't know that word" error the first time that error
     *   occurs.  
     */
    oopsNote()
    {
        /* only offer this note if the note flag is set */
        if (offerOopsNote)
        {
            /* show the note */
            "<.commandsep><.notification>If this was an accidental
            misspelling, you can correct it by typing OOPS followed by the
            corrected word now.  Any time the story points out an unknown
            word, you can correct a misspelling using OOPS as your next
            command.<./notification> ";
            
            /* 
             *   Don't offer the note again.  Note that we explicitly set
             *   this flag in the libMessages object, so that any
             *   subclassed message objects (playerMessages, npcMessages)
             *   all share the same class-level copy of this flag, so that
             *   we only offer the message once in the entire game.  
             */
            libMessages.offerOopsNote = nil;
        }
    }

    /* acknowledge setting VERBOSE mode (true) or TERSE mode (nil) */
    acknowledgeVerboseMode(verbose)
    {
        if (verbose)
            "VERBOSE mode is now selected. ";
        else
            "TERSE mode is now selected. ";
    }

    /* show the current score notify status */
    showNotifyStatus(stat)
    {
        "Score notifications are currently <<stat ? 'on' : 'off'>>. ";
    }

    /* acknowledge a change in the score notification status */
    acknowledgeNotifyStatus(stat)
    {
        "Score notifications are now <<stat ? 'on' : 'off'>>. ";
    }

    /*
     *   Announce the current object of a set of multiple objects on which
     *   we're performing an action.  This is used to tell the player
     *   which object we're acting upon when we're iterating through a set
     *   of objects specified in a command targeting multiple objects.  
     */
    announceMultiActionObject(obj, whichObj, action)
    {
        return '\n' + obj.name + ': ';
    }

    /*
     *   Announce a singleton object that we selected from a set of
     *   ambiguous objects.  This is used when we disambiguate a command
     *   and choose an object over other objects that are also logical but
     *   are less likely.  In such cases, it's courteous to tell the
     *   player what we chose, because it's possible that the user meant
     *   one of the other logical objects - announcing this type of choice
     *   helps reduce confusion by making it immediately plain to the
     *   player when we make a choice other than what they were thinking.  
     */
    announceAmbigActionObject(obj, whichObj, action)
    {
        /* announce the object in "assume" style, ending with a newline */
        return '<.assume>' + obj.theName + '<./assume>\n';
    }

    /*
     *   Announce a singleton object we selected as a default for a
     *   missing noun phrase.
     *   
     *   'resolvedAllObjects' indicates where we are in the command
     *   processing: this is true if we've already resolved all of the
     *   other objects in the command, nil if not.  We use this
     *   information to get the phrasing right according to the situation.
     */
    announceDefaultObject(obj, whichObj, action, resolvedAllObjects)
    {
        /* 
         *   put the action's default-object message in "assume" style,
         *   and start a new line after it 
         */
        return '<.assume>'
            + action.announceDefaultObject(obj, whichObj, resolvedAllObjects)
            + '<./assume>\n';
    }

    /* 'again' used with no prior command */
    noCommandForAgain()
    {
        "<.parser>There's nothing to repeat.<./parser> ";
    }

    /* 'again' cannot be directed to a different actor */
    againCannotChangeActor()
    {
        "<.parser>To repeat a command like <q>turtle, go north,</q>
        just say <q>again,</q> not <q>turtle, again.</q><./parser> ";
    }

    /* 'again': can no longer talk to target actor */
    againCannotTalkToTarget(issuer, target)
    {
        "\^<<issuer.theName>> cannot repeat that command. ";
    }

    /* system actions cannot be directed to non-player characters */
    systemActionToNPC()
    {
        "<.parser>This command cannot be directed to another
        character in the story.<./parser> ";
    }

    /* confirm that we really want to quit */
    confirmQuit()
    {
        "Do you really want to quit? (<<aHref('y', 'Y', 'Confirm quitting')
        >> is affirmative) >\ ";
    }

    /* 
     *   Termination message - this is displayed after we terminate the
     *   story with a 'quit' command or some other route.  This is the
     *   last message the game displays on the way out; there is no need
     *   to offer any options at this point, because the player has
     *   decided to exit the game.
     *   
     *   By default, we show nothing; games can override this to display a
     *   farewell message ("thanks for playing"), if they like.  
     */
    terminating() { }

    /* 
     *   "not terminating" confirmation - this is displayed when the
     *   player doesn't acknowledge a 'quit' command with an affirmative
     *   response to our confirmation question 
     */
    notTerminating()
    {
        "<.parser>Okay, continuing the story.<./parser> ";
    }

    /* confirm that they really want to restart */
    confirmRestart()
    {
        "Do you really want to start over? (<<aHref('Y', 'Y',
        'Confirm restart')>> is affirmative) >\ ";
    }

    /* "not restarting" confirmation */
    notRestarting() { "Okay. "; }

    /* 
     *   Get the save-game file prompt.  Note that this must return a
     *   single-quoted string value, not display a value itself, because
     *   this prompt is passed to inputFile(). 
     */
    getSavePrompt =
        'Please select a file in which to save the current position'

    /* get the restore-game prompt */
    getRestorePrompt = 'Please select the saved position file to restore'

    /* successfully saved */
    saveOkay() { "<.parser>Saved.<./parser> "; }

    /* save canceled */
    saveCanceled() { "<.parser>Canceled.<./parser> "; }

    /* saved failed due to a file write or similar error */
    saveFailed(exc)
    {
        "<.parser>Failed - your computer might be running low
        on disk space, or you might not have the necessary permissions
        to write this file.<./parser> ";
    }

    /* successfully restored */
    restoreOkay() { "<.parser>Restored.<./parser> "; }

    /* restore canceled */
    restoreCanceled() { "<.parser>Canceled.<./parser> "; }

    /* restore failed because the file was not a valid saved game file */
    restoreInvalidFile()
    {
        "<.parser>Failed - this is not a valid saved
        position file.<./parser> ";
    }

    /* restore failed because the file was corrupted */
    restoreCorruptedFile()
    {
        "<.parser>Failed - this saved state file appears to be
        corrupted.  This can occur if the file was modified by another
        program, or the file was copied between computers in a non-binary
        transfer mode, or the physical media storing the file were
        damaged.<./parser> ";
    }

    /* restore failed because the file was for the wrong game or version */
    restoreInvalidMatch()
    {
        "<.parser>Failed - the file was not saved by this
        story (or was saved by an incompatible version of
        the story).<./parser> ";
    }

    /* restore failed for some reason other than those distinguished above */
    restoreFailed(exc)
    {
        "<.parser>Failed - the position could not be
        restored.<./parser> ";
    }

    /* error showing the input file dialog (or whatever) */
    filePromptFailed()
    {
        "<.parser>A system error occurred asking for a filename.
        Your computer might be running low on memory, or might have a
        configuration problem.<./parser> ";
    }

    /* PAUSE prompt */
    pausePrompt()
    {
        "<.parser>The story is now paused.  Please press
        the space bar when you are ready to resume the story, or
        press the 'S' key to save the current position.<./parser><.p>";
    }

    /* saving from within a pause */
    pauseSaving()
    {
        "<.parser>Saving the story...<./parser><.p>";
    }

    /* PAUSE ended */
    pauseEnded()
    {
        "<.parser>Resuming the story.<./parser> ";
    }

    /* get the scripting inputFile prompt message */
    getScriptingPrompt = 'Please select a name for the new script file'

    /* acknowledge scripting on */
    scriptingOkay()
    {
        "<.parser>Text will now be saved to the script file.
        Type <<aHref('unscript', 'UNSCRIPT', 'Turn off scripting')>> to
        discontinue scripting.<./parser> ";
    }

    /* acknowledge cancellation of script file dialog */
    scriptingCanceled() { "<.parser>Canceled.<./parser> "; }

    /* acknowledge scripting off */
    unscriptingOkay()
    {
        "<.parser>Scripting ended.<./parser> ";
    }

    /* undo command succeeded */
    undoOkay(actor, cmd)
    {
        "<.parser>Taking back one turn: <q>";

        /* show the target actor prefix, if an actor was specified */
        if (actor != nil)
            "<<actor>>, ";

        /* show the command */
        "<<cmd>></q>.<./parser><.p>";
    }

    /* undo command failed */
    undoFailed()
    {
        "<.parser>No more undo information is
        available.<./parser> ";
    }

    /* invalid finishGame response */
    invalidFinishOption(resp)
    {
        "\bThat isn't one of the options. ";
    }

    /* acknowledge new "exits on/off" status */
    exitsOnOffOkay(stat)
    {
        "Okay, the list of exits will <<stat ? 'now' : 'no longer'>> be
        displayed in each room description. ";
    }

    /* explain how to turn exit display on and off */
    explainExitsOnOff =
        "<.p><.parser>You can control whether or not the exit list
        is automatically displayed in each room description by typing
        <<aHref('exits on', 'EXITS ON', 'Activate exit display')>> or
        <<aHref('exits off', 'EXITS OFF', 'Turn off exit display'
               )>>.<./parser> "

    /* optional command is not supported in this game */
    commandNotPresent = "<.parser>That command is not used
                         in this story.<./parser> "

    /* this game doesn't use scoring */
    scoreNotPresent = "<.parser>This story doesn't use
                       scoring.<./parser> "

    /* 
     *   cannot reach (i.e., touch) an object that is to be manipulated in
     *   a command - this is a generic message used when we cannot
     *   identify the specific reason that the object is in scope but
     *   cannot be touched 
     */
    cannotReachObject(obj)
    {
        "{You/he} cannot reach <<obj.theNameObj>>. ";
    }

    /* sound is coming from inside/outside a container */
    soundIsFromWithin(obj, loc)
    {
        "\^<<obj.theName>> appear<<obj.verbEndingS>> to be
        coming from inside <<loc.theNameObj>>. ";
    }
    soundIsFromWithout(obj, loc)
    {
        "\^<<obj.theName>> appear<<obj.verbEndingS>> to be
        coming from outside <<loc.theNameObj>>. ";
    }

    /* odor is coming from inside/outside a container */
    smellIsFromWithin(obj, loc)
    {
        "\^<<obj.theName>> appear<<obj.verbEndingS>> to be
        coming from inside <<loc.theNameObj>>. ";
    }
    smellIsFromWithout(obj, loc)
    {
        "\^<<obj.theName>> appear<<obj.verbEndingS>> to be
        coming from outside <<loc.theNameObj>>. ";
    }

    /* default description of the player character */
    pcDesc(actor)
    {
        "\^<<actor.theName>> look<<actor.verbEndingS>> the same
        as usual. ";
    }

    /* describe an actor as part of a room description */
    actorHereDesc(actor) { "\^<<actor.nameIs>> here."; }

    /* 
     *   describe an actor as part of a room description, when the actor
     *   is sitting or lying, without specifying what the actor is sitting
     *   or lying on (this is useful in the dark, for example) 
     */
    actorSittingHere(actor) { "\^<<actor.nameIs>> sitting here."; }
    actorLyingHere(actor) { "\^<<actor.nameIs>> lying here."; }

    /* 
     *   describe an actor sitting/lying on/in something as part of a room
     *   description 
     */
    actorHereStandingOn(actor, cont)
        { "\^<<actor.nameIs>> here, standing <<cont.actorInName>>. "; }
    actorHereSittingOn(actor, cont)
        { "\^<<actor.nameIs>> here, sitting <<cont.actorInName>>. "; }
    actorHereLyingOn(actor, cont)
        { "\^<<actor.nameIs>> here, lying <<cont.actorInName>>. "; }

    /*
     *   describe an actor standing/sitting/lying on something, as part of
     *   a list of actors (such as part of an "examine" description of a
     *   nested room) 
     */
    listActorStandingOn(actor, cont)
        { "\^<<actor.nameIs>> standing <<cont.actorInName>>. "; }
    listActorSittingOn(actor, cont)
        { "\^<<actor.nameIs>> sitting <<cont.actorInName>>. "; }
    listActorLyingOn(actor, cont)
        { "\^<<actor.nameIs>> lying <<cont.actorInName>>. "; }

    /* suffix for group list of actors in common posture and container */
    actorStandingGroupPrefix(loc, cnt) { }
    actorStandingGroupSuffix(loc, cnt)
        { " <<cnt == 1 ? 'is' : 'are'>> standing <<loc.actorInName>>. "; }
    actorSittingGroupPrefix(loc, cnt) { }
    actorSittingGroupSuffix(loc, cnt)
        { " <<cnt == 1 ? 'is' : 'are'>> sitting <<loc.actorInName>>. "; }
    actorLyingGroupPrefix(loc, cnt) { }
    actorLyingGroupSuffix(loc, cnt)
        { " <<cnt == 1 ? 'is' : 'are'>> lying <<loc.actorInName>>. "; }
    
        
    /*
     *   describe an actor as standing/sitting/lying on something, as part
     *   of the actor's "examine" description 
     */
    actorStandingOnDesc(actor, cont)
        { "\^<<actor.itIs>> standing <<cont.actorInName>>. "; }
    actorSittingOnDesc(actor, cont)
        { "\^<<actor.itIs>> sitting <<cont.actorInName>>. "; }
    actorLyingOnDesc(actor, cont)
        { "\^<<actor.itIs>> lying <<cont.actorInName>>. "; }

    /* 
     *   describe an actor's posture as part of an actor's "examine"
     *   description, showing only the generic posture without mentioning
     *   the container 
     */
    actorStandingDesc(actor) { ""; }
    actorSittingDesc(actor) { "\^<<actor.itIs>> sitting. "; }
    actorLyingDesc(actor) { "\^<<actor.itIs>> lying down. "; }

    /* a traveler is arriving, but not from a compass direction */
    sayArriving(traveler)
    {
        "\^<<traveler.travelerName(true)>> enter<<traveler.verbEndingS>>
        the area. ";
    }

    /* a traveler is departing, but not in a compass direction */
    sayDeparting(traveler)
    {
        "\^<<traveler.travelerName(nil)>> leave<<traveler.verbEndingS>>
        the area. ";
    }

    /* a traveler is arriving from a compass direction */
    sayArrivingDir(traveler, dirName)
    {
        "\^<<traveler.travelerName(true)>> enter<<traveler.verbEndingS>>
        from the <<dirName>>. ";
    }

    /* a traveler is leaving in a given compass direction */
    sayDepartingDir(traveler, dirName)
    {
        "\^<<traveler.travelerName(nil)>> leave<<traveler.verbEndingS>>
        to the <<dirName>>. ";
    }
    
    /* a traveler is arriving from a shipboard direction */
    sayArrivingShipDir(traveler, dirName)
    {
        "\^<<traveler.travelerName(true)>> enter<<traveler.verbEndingS>>
        from <<dirName>>. ";
    }

    /* a traveler is leaving in a given shipboard direction */
    sayDepartingShipDir(traveler, dirName)
    {
        "\^<<traveler.travelerName(nil)>> leave<<traveler.verbEndingS>>
        to <<dirName>>. ";
    }

    /* a traveler is going aft */
    sayDepartingAft(traveler)
    {
        "\^<<traveler.travelerName(nil)>> go<<traveler.verbEndingEs>> aft. ";
    }

    /* a traveler is going fore */
    sayDepartingFore(traveler)
    {
        "\^<<traveler.travelerName(nil)>>
        go<<traveler.verbEndingEs>> forward. ";
    }

    /* a shipboard direction was attempted while not onboard a ship */
    notOnboardShip = "That direction isn't meaningful here. "

    /* a traveler is leaving via a passage */
    sayDepartingByPassage(traveler, passage)
    {
        "\^<<traveler.travelerName(nil)>> leave<<traveler.verbEndingS>>
        through <<passage.theNameObj>>. ";
    }

    /* a traveler is arriving via a passage */
    sayArrivingByPassage(traveler, passage)
    {
        "\^<<traveler.travelerName(true)>> enter<<traveler.verbEndingS>>
        through <<passage.theNameObj>>. ";
    }

    /* a traveler is leaving up a stairway */
    sayDepartingUpStairs(traveler, stairs)
    {
        "\^<<traveler.travelerName(nil)>> go<<traveler.verbEndingEs>>
        up <<stairs.theNameObj>>. ";
    }

    /* a traveler is leaving down a stairway */
    sayDepartingDownStairs(traveler, stairs)
    {
        "\^<<traveler.travelerName(nil)>> go<<traveler.verbEndingEs>>
        down <<stairs.theNameObj>>. ";
    }

    /* a traveler is arriving by coming up a stairway */
    sayArrivingUpStairs(traveler, stairs)
    {
        "\^<<traveler.travelerName(true)>> come<<traveler.verbEndingS>>
        up <<stairs.theNameObj>>. ";
    }

    /* a traveler is arriving by coming down a stairway */
    sayArrivingDownStairs(traveler, stairs)
    {
        "\^<<traveler.travelerName(true)>> come<<traveler.verbEndingS>>
        down <<stairs.theNameObj>>. ";
    }

    /* note that a door is being opened/closed remotely */
    sayOpenDoorRemotely(door, stat)
    {
        "Someone <<stat ? 'open' : 'close'>>s <<door.theNameObj>> from
        the other side. ";
    }

    /* 
     *   open/closed status - these are simply adjectives that can be used
     *   to describe the status of an openable object 
     */
    openMsg = 'open'
    closedMsg = 'closed'

    /* locked/unlocked status - adjectives describing lock states */
    lockedMsg = 'locked'
    unlockedMsg = 'unlocked'

    /*
     *   on/off status - these are simply adjectives that can be used to
     *   describe the status of a switchable object 
     */
    onMsg = 'on'
    offMsg = 'off'

    /* daemon report for burning out a match */
    matchBurnedOut(obj)
    {
        "\^<<obj.theName>> finish<<obj.verbEndingEs>> burning,
        and disappears into a cloud of ash. ";
    }

    /* daemon report for burning out a candle */
    candleBurnedOut(obj)
    {
        "\^<<obj.nameVerb('burn')>> down too far to stay lit, and
        go<<obj.verbEndingEs>> out. ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Player Character messages.  These messages are generated when the
 *   player issues a regular command to the player character (i.e.,
 *   without specifying a target actor).  
 */
playerMessages: libMessages
    /* invalid command syntax */
    commandNotUnderstood(actor)
    {
        "<.parser>The story doesn't understand that command.<./parser> ";
    }

    /* no match for a noun phrase */
    noMatch(actor, txt)
    {
        "{You/he} see{s} no <<txt>> here. ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "<.parser>{You/he} see{s} nothing to use
        for <q>all</q> here.<./parser> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "<.parser>{You/he} see{s} nothing else here.<./parser> ";
    }

    /* nothing left in a plural phrase after removing 'except' items */
    noMatchForListBut(actor) { noMatchForAllBut(actor); }

    /* no match for a pronoun */
    noMatchForPronoun(actor, typ, pronounWord)
    {
        /* show the message */
        "<.parser>The word <q><<pronounWord>></q> doesn't refer to
        anything right now.<./parser> ";
    }

    /* no match for a possessive phrase */
    noMatchForPossessive(actor, owner, txt)
    {
        "<.parser>\^<<owner.theName>> do<<owner.verbEndingEs>>
        not appear to have any such thing.<./parser> ";
    }

    /* no match for a containment phrase */
    noMatchForLocation(actor, loc, txt)
    {
        "<.parser>\^<<actor.nameVerb('see')>> no 
        <<loc.childInName(txt)>>.<./parser> ";
    }

    /* nothing in a container whose contents are specifically requested */
    nothingInLocation(actor, loc)
    {
        "<.parser>\^<<actor.nameVerb('see')>> 
        <<loc.childInName('nothing')>>.<./parser> ";
    }

    /* no match for the response to a disambiguation question */
    noMatchDisambig(actor, origPhrase, disambigResponse)
    {
        /* 
         *   show the message, leaving the <.parser> tag mode open - we
         *   always show another disambiguation prompt after this message,
         *   so we'll let the prompt close the <.parser> mode 
         */
        "<.parser>That was not one of the choices. ";
    }

    /* empty noun phrase ('take the') */
    emptyNounPhrase(actor)
    {
        "<.parser>You seem to have left out some words.<./parser> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "<.parser>\^<<actor.theName>> can't do that to zero of
        something.<./parser> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "<.parser>\^<<actor.nameVerb('do')>>n't see that many <<txt>>
        here.<./parser> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "<.parser>Multiple objects aren't allowed with that
        command.<./parser> ";
    }

    /* a single noun phrase is required, but a noun list was used */
    singleObjectRequired(actor, txt)
    {
        "<.parser>Multiple objects aren't allowed with that
        command.<./parser> ";
    }

    /* 
     *   The answer to a disambiguation question specifies an invalid
     *   ordinal ("the fourth one" when only three choices were offered).
     *   
     *   'ordinalWord' is the ordinal word entered ('fourth' or the like).
     *   'originalText' is the text of the noun phrase that caused the
     *   disambiguation question to be asked in the first place.  
     */
    disambigOrdinalOutOfRange(actor, ordinalWord, originalText)
    {
        /* leave the <.parser> tag open, for the re-prompt that will follow */
        "<.parser>There were not that many choices. ";
    }

    /* 
     *   Ask the canonical disambiguation question: "Which x do you
     *   mean...?".  'matchList' is the list of ambiguous objects with any
     *   redundant equivalents removed; and 'fullMatchList' is the full
     *   list, including redundant equivalents that were removed from
     *   'matchList'.
     *   
     *   If askingAgain is true, it means that we're asking the question
     *   again because we got an invalid response to the previous attempt
     *   at the same prompt.  We will have explained the problem, and now
     *   we're going to give the user another crack at the same response.
     *   
     *   To prevent interactive disambiguation, do this:
     *   
     *   throw new ParseFailureException(&ambiguousNounPhrase,
     *.  originalText, matchList, fullMatchList); 
     */
    askDisambig(actor, originalText, matchList, fullMatchList,
                requiredNum, askingAgain, dist)
    {
        /* 
         *   Open the "<.parser>" tag, if we're not "asking again."  If we
         *   are asking again, we will already have shown a message
         *   explaining why we're asking again, and that message will have
         *   left us in <.parser> tag mode, so we don't need to open the
         *   tag again. 
         */
        if (!askingAgain)
            "<.parser>";
        
        /* 
         *   the question varies depending on whether we want just one
         *   object or several objects in the final result 
         */
        if (requiredNum == 1)
        {
            /* 
             *   One object needed - use the original text in the query.
             *   
             *   Note that if we're "asking again," we will have shown an
             *   additional message first explaining *why* we're asking
             *   again, and that message will have left us in <.parser>
             *   tag mode; so we need to close the <.parser> tag in this
             *   case, but we don't need to show a new one. 
             */
            if (askingAgain)
                "Which did you mean,
                <<askDisambigList(matchList, fullMatchList, nil, dist)>>?";
            else
                "Which <<originalText>> do you mean,
                <<askDisambigList(matchList, fullMatchList, nil, dist)>>?";
        }
        else
        {
            /* 
             *   Multiple objects required - ask by number, since we can't
             *   easily guess what the plural might be given the original
             *   text.
             *   
             *   As above, we only need to *close* the <.parser> tag if
             *   we're asking again, because we will already have shown a
             *   prompt that opened the tag in this case.  
             */
            if (askingAgain)
                "Which <<spellInt(requiredNum)>> (of
                <<askDisambigList(matchList, fullMatchList, true, dist)>>)
                did you mean?";
            else
                "Which <<spellInt(requiredNum)>>
                (of <<askDisambigList(matchList, fullMatchList,
                                      true, dist)>>) do you mean?";
        }

        /* close the <.parser> tag */
        "<./parser> ";
    }

    /* 
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "<.parser>The story doesn't know which
        <<originalText>> you mean.<./parser> ";
    }

    /* the actor is missing in a command */
    missingActor(actor)
    {
        "<.parser>You must be more specific about <<
        whomPronoun>> you want to address.<./parser> ";
    }

    /* only a single actor can be addressed at a time */
    singleActorRequired(actor)
    {
        "<.parser>You can only address one person
        at a time.<./parser> ";
    }

    /* cannot change actor mid-command */
    cannotChangeActor()
    {
        "<.parser>You cannot address more than one character on
        a single command line in this story.<./parser> ";
    }

    /* 
     *   tell the user they entered a word we don't know, offering the
     *   chance to correct it with "oops" 
     */
    askUnknownWord(actor, txt)
    {
        /* start the message */
        "<.parser>The word <q><<txt>></q> is not necessary in this
        story.<./parser> ";

        /* mention the OOPS command, if appropriate */
        oopsNote();
    }

    /* 
     *   tell the user they entered a word we don't know, but don't offer
     *   an interactive way to fix it (i.e., we can't use OOPS at this
     *   point) 
     */
    wordIsUnknown(actor, txt)
    {
        "<.parser>The story doesn't understand that
        command.<./parser> ";
    }

    /* cannot speak to multiple actors */
    cannotAddressMultiple(actor)
    {
        "<.parser>\^<<actor.theName>> cannot address multiple
        people at once.<./parser> ";
    }
;

/* ------------------------------------------------------------------------ */
/* 
 *   Non-Player Character (NPC) messages - parser-mediated format.  These
 *   versions of the NPC messages report errors through the
 *   parser/narrator.
 *   
 *   Note that a separate set of messages can be selected to report
 *   messages in the voice of the NPC - see npcMessagesDirect below.  
 */

/*
 *   Standard Non-Player Character (NPC) messages.  These messages are
 *   generated when the player issues a command to a specific non-player
 *   character. 
 */
npcMessages: playerMessages
    /* the target cannot hear a command we gave */
    commandNotHeard(actor)
    {
        "\^<<actor.nameVerb('do')>> not respond. ";
    }

    /* no match for a noun phrase */
    noMatch(actor, txt)
    {
        "\^<<actor.nameVerb('see')>> no <<txt>>. ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "<.parser>\^<<actor.nameVerb('do')>>n't see what you
        mean by <q>all.</q><./parser> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "<.parser>\^<<actor.nameVerb('see')>> nothing
        else.<./parser> ";
    }

    /* no match for the response to a disambiguation question */
    noMatchDisambig(actor, origPhrase, disambigResponse)
    {
        /* leave the <.parser> tag open, for the re-prompt that will follow */
        "<.parser>That was not one of the choices. ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "<.parser>\^<<actor.nameVerb('do')>>n't see that many
         <<txt>>.<./parser> ";
    }

    /* an actor doesn't accept a command from another actor */
    refuseCommand(targetActor, issuingActor)
    {
        "\^<<targetActor.nameVerb('refuse')>> <<issuingActor.itPossAdj>>
        request. ";
    }

    /* the actor refuses the command because it's busy with something else */
    refuseCommandBusy(targetActor, issuingActor)
    {
        "\^<<targetActor.nameIs>> busy. ";
    }

    /* 
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "<.parser>\^<<actor.nameVerb('do')>>n't know which
        <<originalText>> you mean.<./parser> ";
    }

    /*
     *   Missing object query and error message templates 
     */
    askMissingObject(actor, action, which)
    {
        "<.parser>\^<<action.whatObj(which)>> do you want
        <<actor.theNameObj>> to <<action.verbInf(which)>>?<./parser> ";
    }
    missingObject(actor, action, which)
    {
        "<.parser>You must be more specific
        about <<action.whatObj(which)>> you want <<actor.theNameObj>>
        to <<action.verbInf(which)>>.<./parser> ";
    }

    /* missing literal phrase query and error message templates */
    missingLiteral(actor, action, which)
    {
        "<.parser>You must be more specific
        about <<action.whatObj(which)>> you want <<actor.theNameObj>> to
        <<action.verbInf(which, nil)>>.  For example:
        <<actor.theName>>, <<action.verbInf(which, nil)>>
        <q>something</q>.<./parser> ";
    }
;

/*
 *   Deferred NPC messages.  We use this to report deferred messages from
 *   an NPC to the player.  A message is deferred when a parsing error
 *   occurs, but the NPC can't talk to the player because there's no sense
 *   path to the player.  When this happens, the NPC queues the message
 *   for eventual delivery; when a sense path appears later that lets the
 *   NPC talk to the player, we deliver the message through this object.
 *   Since these messages describe conditions that occurred in the past,
 *   we use the past tense to phrase the messages.
 *   
 *   This default implementation simply doesn't report deferred errors at
 *   all.  The default message voice is the parser/narrator character, and
 *   there is simply no good way for the parser/narrator to say that a
 *   command failed in the past for a given character: "Bob looks like he
 *   didn't know which box you meant" just doesn't work.  So, we'll simply
 *   not report these errors at all.
 *   
 *   To report messages in the NPC's voice directly, modify the NPC's
 *   Actor object, or the Actor base class, to return
 *   npcDeferredMessagesDirect rather than this object from
 *   getParserMessageObj().  
 */
npcDeferredMessages: object
;

/* ------------------------------------------------------------------------ */
/*
 *   NPC messages, reported directly in the voice of the NPC.  These
 *   messages are not selected by default, but a game can use them instead
 *   of the parser-mediated versions by modifying the actor object's
 *   getParserMessageObj() to return these objects.  
 */

/*
 *   Standard Non-Player Character (NPC) messages.  These messages are
 *   generated when the player issues a command to a specific non-player
 *   character. 
 */
npcMessagesDirect: npcMessages
    /* no match for a noun phrase */
    noMatch(actor, txt)
    {
        "\^<<actor.nameVerb('look')>> around. <q>I don't
        see any <<txt>>.</q> ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "\^<<actor.nameVerb('say')>>, <q>I don't see what
        you mean by <q>all</q>.</q> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "\^<<actor.nameVerb('say')>>, <q>I see nothing else here.</q> ";
    }

    /* no match for the response to a disambiguation question */
    noMatchDisambig(actor, origPhrase, disambigResponse)
    {
        "\^<<actor.nameVerb('say')>>, <q>That was not one of
        the choices.</q> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "\^<<actor.nameVerb('say')>>, <q>I can't do that to
            zero of something.</q> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "\^<<actor.nameVerb('say')>>, <q>I don't see that many <<txt>>
            here.</q> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "\^<<actor.nameVerb('say')>>, <q>I can't use multiple objects
            like that.</q> ";
    }

    /* a single noun phrase is required, but a noun list was used */
    singleObjectRequired(actor, txt)
    {
        "\^<<actor.nameVerb('say')>>, <q>I can't use multiple objects
        like that.</q> ";
    }

    /* 
     *   The answer to a disambiguation question specifies an invalid
     *   ordinal ("the fourth one" when only three choices were offered).
     *   
     *   'ordinalWord' is the ordinal word entered ('fourth' or the like).
     *   'originalText' is the text of the noun phrase that caused the
     *   disambiguation question to be asked in the first place.  
     */
    disambigOrdinalOutOfRange(actor, ordinalWord, originalText)
    {
        /* leave the <.parser> tag open, for the re-prompt that will follow */
        "<.parser>There were not that many choices. ";
    }

    /* an actor doesn't accept a command from another actor */
    refuseCommand(targetActor, issuingActor)
    {
        "\^<<targetActor.nameVerb('refuse')>> <<issuingActor.itPossAdj>>
        request. ";
    }

    /* the actor refuses the command because it's busy with something else */
    refuseCommandBusy(targetActor, issuingActor)
    {
        "\^<<targetActor.nameIs>> busy. ";
    }

    /* 
     *   Ask the canonical disambiguation question: "Which x do you
     *   mean...?".  'matchList' is the list of ambiguous objects with any
     *   redundant equivalents removed, and 'fullMatchList' is the full
     *   list, including redundant equivalents that were removed from
     *   'matchList'.  
     *   
     *   To prevent interactive disambiguation, do this:
     *   
     *   throw new ParseFailureException(&ambiguousNounPhrase,
     *.  originalText, matchList, fullMatchList); 
     */
    askDisambig(actor, originalText, matchList, fullMatchList,
                requiredNum, askingAgain, dist)
    {
        /* the question depends on the number needed */
        if (requiredNum == 1)
        {
            /* one required - ask with the original text */
            if (askingAgain)
                "\n";
            
            "\^<<actor.nameVerb('ask')>>, <q>Which <<originalText>> do
            you mean, <<askDisambigList(matchList, fullMatchList,
                                        nil, dist)>>?</q> ";
        }
        else
        {
            /* 
             *   more than one required - we can't guess at the plural
             *   given the original text, so just use the number 
             */
            if (askingAgain)
                "\n";
            
            "\^<<actor.nameVerb('ask')>>,
            <q>Which <<spellInt(requiredNum)>> (of
            <<askDisambigList(matchList, fullMatchList, true, dist)>>)
            do you mean?</q> ";
        }
    }

    /* 
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "\^<<actor.nameVerb('say')>>, <q>I don't know which
            <<originalText>> you mean.</q> ";
    }

    /*
     *   Missing object query and error message templates 
     */
    askMissingObject(actor, action, which)
    {
        "\^<<actor.nameVerb('say')>>, <q>\^<<action.whatObj(which)>> do
        you want me to <<action.verbInf(which)>>?</q> ";
    }
    missingObject(actor, action, which)
    {
        "\^<<actor.nameVerb('say')>>,
        <q>I don't know <<action.whatObj(which)>>
        you want me to <<action.verbInf(which)>>.</q> ";
    }
    missingLiteral(actor, action, which)
    {
        /* use the same message we use for a missing ordinary object */
        missingObject(actor, action, which);
    }

    /* tell the user they entered a word we don't know */
    askUnknownWord(actor, txt)
    {
        "\^<<actor.nameVerb('say')>>, <q>I don't know the word
         '<<txt>>'.</q> ";
    }

    /* tell the user they entered a word we don't know */
    wordIsUnknown(actor, txt)
    {
        "\^<<actor.nameVerb('say')>>, <q>You used a word I don't know.</q> ";
    }
;

/*
 *   Deferred NPC messages.  We use this to report deferred messages from
 *   an NPC to the player.  A message is deferred when a parsing error
 *   occurs, but the NPC can't talk to the player because there's no sense
 *   path to the player.  When this happens, the NPC queues the message
 *   for eventual delivery; when a sense path appears later that lets the
 *   NPC talk to the player, we deliver the message through this object.
 *   Since these messages describe conditions that occurred in the past,
 *   we use the past tense to phrase the messages.
 *   
 *   Some messages will never be deferred:
 *   
 *   commandNotHeard - if a command is not heard, it will never enter an
 *   actor's command queue; the error is given immediately in response to
 *   the command entry.
 *   
 *   refuseCommand - like commandNotHeard, this is generated in immediate
 *   response to a command entry.
 *   
 *   refuseCommandBusy - same as commandNotHeard
 *   
 *   noMatchDisambig - interactive disambiguation will not happen in a
 *   deferred response situation, so it is impossible to have an
 *   interactive disambiguation failure.  
 *   
 *   disambigOrdinalOutOfRange - for the same reason noMatchDisambig can't
 *   be deferred.
 *   
 *   askDisambig - if we couldn't display a message, we definitely
 *   couldn't perform interactive disambiguation.
 *   
 *   askMissingObject - for the same reason that askDisambig can't be
 *   deferred
 *   
 *   askUnknownWord - for the same reason that askDisambig can't be
 *   deferred.  
 */
npcDeferredMessagesDirect: npcDeferredMessages
    commandNotUnderstood(actor)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't understand
            what you meant.</q> ";
    }

    /* no match for a noun phrase */
    noMatch(actor, txt)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't see any <<txt>>.</q> ";
    }

    /* no match for 'all' */
    noMatchForAll(actor)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't see what
            you meant by <q>all</q>.</q> ";
    }

    /* nothing left for 'all' after removing 'except' items */
    noMatchForAllBut(actor)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't see what you meant.</q> ";
    }

    /* empty noun phrase ('take the') */
    emptyNounPhrase(actor)
    {
        "\^<<actor.nameVerb('say')>>, <q>You left some words out.</q> ";
    }

    /* 'take zero books' */
    zeroQuantity(actor, txt)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't
             understand what you meant.</q> ";
    }

    /* insufficient quantity to meet a command request ('take five books') */
    insufficientQuantity(actor, txt, matchList, requiredNum)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't see enough <<txt>>.</q> ";
    }

    /* a unique object is required, but multiple objects were specified */
    uniqueObjectRequired(actor, txt, matchList)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't
            understand what you meant.</q> ";
    }

    /* a unique object is required, but multiple objects were specified */
    singleObjectRequired(actor, txt)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't
            understand what you meant.</q> ";
    }

    /* 
     *   we found an ambiguous noun phrase, but we were unable to perform
     *   interactive disambiguation 
     */
    ambiguousNounPhrase(actor, originalText, matchList, fullMatchList)
    {
        "\^<<actor.nameVerb('say')>>, <q>I couldn't tell which
        <<originalText>> you meant.</q> ";
    }

    /* an object phrase was missing */
    askMissingObject(actor, action, which)
    {
        "\^<<actor.nameVerb('say')>>, <q>I didn't know
        <<action.whatObj(which)>> you wanted me to
        <<action.verbInf(which)>>.</q> ";
    }

    /* tell the user they entered a word we don't know */
    wordIsUnknown(actor, txt)
    {
        "\^<<actor.nameVerb('say')>>, <q>You used a word I don't know.</q> ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Verb messages for standard library verb implementations for actions
 *   performed by the player character.  These return strings suitable for
 *   use in VerifyResult objects as well as for action reports
 *   (defaultReport, mainReport, and so on).
 *   
 *   Most of these messages are generic enough to be used for player and
 *   non-player character alike.  However, some of the messages either are
 *   too terse (such as the default reports) or are phrased awkwardly for
 *   NPC use, so the NPC verb messages override those.  
 */
playerActionMessages: MessageHelper
    /* 
     *   generic "can't do that" message - this is used when verification
     *   fails because an object doesn't define the action ("doXxx")
     *   method for the verb 
     */
    cannotDoThat = '{You/he} can\'t do that. '

    /* must be holding something before a command */
    mustBeHolding(obj)
    {
        gMessageParams(obj);
        return '{You/he} must be holding {the obj/him} to do that. ';
    }

    /* it's too dark to do that */
    tooDark = 'It\'s too dark to do that. '

    /* object must be visible */
    mustBeVisible(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot see {that obj/him}. ';
    }

    /* object can be heard but not seen */
    heardButNotSeen(obj)
    {
        gMessageParams(obj);
        return '{You/he} can hear {an obj/him}, but {you/he}
                 can\'t see{s} {it obj/him}. ';
    }

    /* object can be smelled but not seen */
    smelledButNotSeen(obj)
    {
        gMessageParams(obj);
        return '{You/he} can smell {an obj/him}, but {you/he}
                can\'t see{s} {it obj/him}. ';
    }

    /* cannot hear object */
    cannotHear(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot hear {that obj/him}. ';
    }

    /* cannot smell object */
    cannotSmell(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot smell {that obj/him}. ';
    }

    /* cannot taste object */
    cannotTaste(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot taste {that obj/him}. ';
    }

    /* must remove an article of clothing before a command */
    cannotBeWearing(obj)
    {
        gMessageParams(obj);
        return '{You/he} must take off {the obj/him}
                before {it actor/he} can do that. ';
    }

    /* object must be opened before doing that */
    mustBeOpen(obj)
    {
        gMessageParams(obj);
        return '{You/he} must open {the obj/him}
                before {it actor/he} can do that. ';
    }

    /* object must be closed before doing that */
    mustBeClosed(obj)
    {
        gMessageParams(obj);
        return '{You/he} must close {the obj/him}
               before {it actor/he} can do that. ';
    }

    /* object must be unlocked before doing that */
    mustBeUnlocked(obj)
    {
        gMessageParams(obj);
        return '{You/he} must unlock {the obj/him}
                before {it actor/he} can do that. ';
    }

    /* no key is needed to lock or unlock this object */
    noKeyNeeded = '{The dobj/he} do{es} not appear to take a key. '

    /* actor must be standing before doing that */
    mustBeStanding = '{You/he} must stand up before {it actor/he}
                      can do that. '

    /* must be sitting on/in chair */
    mustSitOn(obj)
    {
        gMessageParams(obj);
        return '{You/he} must sit {in obj} first. ';
    }

    /* must get on/in object */
    mustGetOn(obj)
    {
        gMessageParams(obj);
        return '{You/he} must get {in obj} first. ';
    }

    /* generic "that's not important" message for decorations */
    notImportant(obj)
    {
        gMessageParams(obj);
        return '{The obj/he} {is}n\'t important. ';
    }

    /* generic "that's too far away" message for Distant items */
    tooDistant(obj)
    {
        gMessageParams(obj);
        return '{The obj/he} {is} too far away. ';
    }

    /* generic "no can do" message for intangibles */
    notWithIntangible(obj)
    {
        gMessageParams(obj);
        return '{You/he} can\'t do that to {an obj/him}. ';
    }

    /* 
     *   cannot reach (i.e., touch) an object that is to be manipulated in
     *   a command - this is a generic message used when we cannot
     *   identify the specific reason that the object is in scope but
     *   cannot be touched 
     */
    cannotReachObject(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot reach {the obj/him}. ';
    }

    /*
     *   cannot reach an object, because the object is inside the given
     *   container 
     */
    cannotReachContents(obj, loc)
    {
        gMessageParams(obj, loc);
        return '{You/he} cannot reach {the obj/him} through {the loc/him}. ';
    }

    /* cannot reach an object because it's outisde the given container */
    cannotReachOutside(obj, loc)
    {
        gMessageParams(obj, loc);
        return '{You/he} cannot reach {the obj/him} through {the loc/him}. ';
    }

    /* cannot reach an object through an obstructor */
    cannotReachThrough(obj, loc)
    {
        gMessageParams(obj, loc);
        return '{You/he} cannot reach {the obj/him} through {the loc/him}. ';
    }

    /* generic long description of a Thing */
    thingDesc(obj)
    {
        gMessageParams(obj);
        return '{You/he} see{s} nothing unusual about {it obj/him}. ';
    }

    /* default description of a non-player character */
    npcDesc(npc)
    {
        gMessageParams(npc);
        return '{You/he} see{s} nothing unusual about {the npc/him}. ';
    }

    /* generic messages for looking prepositionally */
    nothingInside = 'There\'s nothing in {the dobj/him}. '
    nothingUnder = '{You/he} see{s} nothing unusual under {the dobj/him}. '
    nothingBehind = '{You/he} see{s} nothing unusual behind {the dobj/him}. '
    nothingThrough = '{You/he} can see nothing through {the dobj/him}. '

    /* there's nothing here with a specific odor */
    nothingToSmell = '{You/he} smell{s} nothing out of the ordinary. '

    /* there's nothing here with a specific noise */
    nothingToHear = '{You/he} hear{s} nothing out of the ordinary. '

    /* a sound appears to be coming from a source */
    noiseSource(src)
    {
        return '{The dobj/he} seem{s} to be coming from '
            + src.theNameObj + '. ';
    }

    /* an odor appears to be coming from a source */
    odorSource(src)
    {
        return '{The dobj/he} seem{s} to be coming from '
            + src.theNameObj + '. ';
    }

    /* an item is not wearable */
    notWearable = '{That dobj/he} {is}n\'t something {you/he} can wear. '

    /* doffing something that isn't wearable */
    notDoffable = '{That dobj/he} {is}n\'t something {you/he} can remove. '

    /* already wearing item */
    alreadyWearing = '{You\'re} already wearing {it dobj/him}. '

    /* not wearing (item being doffed) */
    notWearing = '{You\'re} not wearing {that dobj/him}. '

    /* default response to 'wear obj' */
    okayWear = 'Okay, {you\'re} now wearing {the dobj/him}. '

    /* default response to 'doff obj' */
    okayDoff = 'Okay, {you\'re} no longer wearing {the dobj/him}. '

    /* default response to open/close */
    okayOpen = 'Opened. '
    okayClose = 'Closed. '

    /* default response to lock/unlock */
    okayLock = 'Locked. '
    okayUnlock = 'Unlocked. '

    /* cannot dig here */
    cannotDig = '{You/he} {have} no reason to dig in {that dobj/him}. '

    /* not a digging implement */
    cannotDigWith =
        '{You/he} see{s} no way to use {that dobj/him} as a shovel. '

    /* taking something already being held */
    alreadyHolding = '{You/he} {are} already carrying {the dobj/him}. '

    /* actor taking self ("take me") */
    takingSelf = '{You/he} can\'t take {yourself}. '

    /* dropping an object not being carried */
    notCarrying = '{You\'re} not carrying {that dobj/him}. '

    /* actor dropping self */
    droppingSelf = '{You/he} can\'t drop {yourself}. '

    /* we can't put the dobj in the iobj because it's already there */
    alreadyPutIn = '{The dobj/he} {is} already in {the iobj/him}. '

    /* we can't put the dobj on the iobj because it's already there */
    alreadyPutOn = '{The dobj/he} {is} already on {the iobj/him}. '

    /* 
     *   trying to move a fixed item to a new container by some means
     *   (take, drop, put in, put on, etc) 
     */
    cannotMoveFixed = '{The dobj/he} cannot be moved. '

    /* trying to take a fixed item */
    cannotTakeFixed = '{You/he} can\'t take {that dobj/him}. '

    /* trying to move a component object */
    cannotMoveComponent(loc)
    {
        return '{The dobj/he} {is} part of ' + loc.theNameObj + '. ';
    }

    /* trying to take a component object */
    cannotTakeComponent(loc)
    {
        return '{You/he} can\'t have {that/him dobj}; '
            + '{it\'s dobj} part of ' + loc.theNameObj + '. ';
    }

    /* trying to put a component in something */
    cannotPutComponent(loc)
    {
        return '{You/he} can\'t put {that/him dobj} anywhere; '
            + '{it\'s dobj} part of ' + loc.theNameObj + '. ';
    }

    /* speicalized Fixed messages for TravelPushables */
    cannotTakePushable = '{You/he} can\'t take {that/him dobj}, but
                          {it actor/he} might be able to push it somewhere. '
    cannotMovePushable = 'It wouldn\'t accomplish anything to move
                          {the dobj/him} around aimlessly, but {it actor/he}
                          might be able to push {it/him} in a specific
                          direction. '
    cannotPutPushable = '{You/he} can\'t put {that/him dobj} anywhere,
                         but {it actor/he} might be able to push it
                         somewhere. '

    /* can't take something while occupying it */
    cannotTakeLocation = '{You/he} can\'t take {that/him dobj}
                          while {you\'re} occupying {it/him dobj}. '

    /* default 'take' response */
    okayTake = 'Taken. '

    /* default 'drop' response */
    okayDrop = 'Dropped. '

    /* default 'put in' response */
    okayPutIn = 'Done. '

    /* default 'put on' response */
    okayPutOn = 'Done. '

    /* try to take/move/put an untakeable actor */
    cannotTakeActor = '{The dobj/he} won\'t let {you/him} have {it/him}. '
    cannotMoveActor = '{The dobj/he} won\'t let {you/him} do that. '
    cannotPutActor = '{The dobj/he} won\'t let {you/him} do that. '

    /* trying to take/move/put a person */
    cannotTakePerson = '{The dobj/he} probably wouldn\'t like that. '
    cannotMovePerson = '{The dobj/he} probably wouldn\'t like that. '
    cannotPutPerson = '{The dobj/he} probably wouldn\'t like that. '

    /* cannot move obj because cont is closed */
    cannotMoveThroughClosed(obj, cont)
    {
        gMessageParams(cont);
        return '{You/he} can\'t do that because {the cont/he} {is} closed. ';
    }

    /* cannot move obj through obstructor */
    cannotMoveThrough(obj, obs)
    {
        gMessageParams(obj, obs);
        return '{You/he} can\'t move {that obj/him} through {the obs/him}. ';
    }

    /* cannot fit obj into cont through cont's opening */
    cannotFitIntoOpening(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{You/he} can\'t do that because {the obj/he} {is}
                too big to put into {the cont/him}. ';
    }

    /* cannot fit obj out of cont through cont's opening */
    cannotFitOutOfOpening(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{You/he} can\'t do that because {the obj/he} {is}
                too big to take out of {the cont/him}. ';
    }

    /* actor 'obj' cannot reach through cont because cont is closed */
    cannotTouchThroughClosed(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{The obj/he} can\'t do that because
               {the cont/he} {is} closed. ';
    }

    /* actor cannot fit hand into cont through cont's opening */
    cannotReachIntoOpening(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{The obj/he} can\'t fit {its/her} hand into {the cont/him}. ';
    }

    /* actor cannot fit hand into cont through cont's opening */
    cannotReachOutOfOpening(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{The obj/he} can\'t fit {its/her} hand through
               {the cont/him}. ';
    }

    /* the object is too large for the actor to hold */
    tooLargeForActor(obj)
    {
        gMessageParams(obj);
        return '{The obj/he} {is} too large for {you/him} to hold. ';
    }

    /* the actor doesn't have room to hold the object */
    handsTooFullFor(obj)
    {
        return '{Your} hands are too full to hold ' + obj.theNameObj + '. ';
    }

    /* the object is becoming too large for the actor to hold */
    becomingTooLargeForActor(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot do that because {the obj/he}
                would become too large for {you/him} to hold. ';
    }

    /* the object is becoming large enough that the actor's hands are full */
    handsBecomingTooFullFor(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot do that because {your} hands
            would become too full to hold {the obj/him}. ';
    }

    /* the object is too heavy (all by itself) for the actor to hold */
    tooHeavyForActor(obj)
    {
        gMessageParams(obj);
        return '{The obj/he} {is} too heavy for {you/him} to pick up. ';
    }

    /* 
     *   the object is too heavy (in combination with everything else
     *   being carried) for the actor to pick up 
     */
    totalTooHeavyFor(obj)
    {
        gMessageParams(obj);
        return '{The obj/he} {is} too heavy; {you/he} will have to put
            something else down first. ';
    }

    /* object is too large for container */
    tooLargeForContainer(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{The obj/he} {is} too large for {the cont/him}. ';
    }

    /* container doesn't have room for object */
    containerTooFull(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{The cont/he} {is} already too full to hold {the obj/him}. ';
    }

    /* surface doesn't have room for object */
    surfaceTooFull(obj, cont)
    {
        gMessageParams(obj, cont);
        return 'There\'s no room for {the obj/him} on {the cont/him}. ';
    }

    /* the current action would make obj too large for its container */
    becomingTooLargeForContainer(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{You/he} cannot do that because it would make
            {the obj/him} too large for {the cont/him}. ';
    }

    /* 
     *   the current action would increase obj's bulk so that container is
     *   too full 
     */
    containerBecomingTooFull(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{You/he} cannot do that because {the obj/he}
            would no longer fit in {the cont/him}. ';
    }

    /* trying to put an object in a non-container */
    notAContainer = '{You/he} can\'t put anything in {the iobj/him}. '

    /* trying to put an object on a non-surface */
    notASurface = 'There\'s no good surface on {the iobj/him}. '

    /* can't put anything under iobj */
    cannotPutUnder = '{You/he} can\'t put anything under {that iobj/him}. '

    /* trying to put a fixed item in something */
    cannotPutFixed = '{You/he} can\'t put {the dobj/him} anywhere. '

    /* trying to put something in itself */
    cannotPutInSelf = '{You/he} can\'t put {the dobj/him} in {itself}. '

    /* trying to put something on itself */
    cannotPutOnSelf = '{You/he} can\'t put {the dobj/him} on {itself}. '

    /* trying to return something to a remove-only dispenser */
    cannotReturnToDispenser =
        '{You/he} can\'t put {a dobj/him} back in {the iobj/him}. '

    /* wrong item type for dispenser */
    cannotPutInDispenser =
        '{You/he} can\'t put {a dobj/him} in {the iobj/him}. '

    /* the dobj doesn't fit on this keyring */
    objNotForKeyring = '{The dobj/he} do{es}n\'t fit on {the iobj/him}. '

    /* the dobj isn't on the keyring */
    keyNotOnKeyring = '{The dobj/he} {is} not attached to {the iobj/him}. '

    /* can't detach key (with no iobj specified) because it's not on a ring */
    keyNotDetachable = '{The dobj/he} {is}n\'t attached to anything. '

    /* we took a key and attached it to a keyring */
    takenAndMovedToKeyring(keyring)
    {
        gMessageParams(keyring);
        return '{You/he} pick{s} up {the dobj/him} and attach{es actor}'
            + ' {it dobj/him} to {the keyring/him}. ';
    }

    /* we attached a key to a keyring automatically */
    movedKeyToKeyring(keyring)
    {
        gMessageParams(keyring);
        return '{You/he} attach{es} {the dobj/him} to {the keyring/him}. ';
    }

    /* putting y in x when x is already in y */
    circularlyIn(x, y)
    {
        gMessageParams(x, y);
        return '{You/he} can\'t do that because {the x/he} {is}
            already in {the y/him}. ';
    }

    /* putting y in x when x is already on y */
    circularlyOn(x, y)
    {
        gMessageParams(x, y);
        return '{You/he} can\'t do that because {the x/he} {is}
            already on {the y/him}. ';
    }

    /* taking dobj from iobj, but dobj isn't in iobj */
    takeFromNotIn = '{The dobj/he} {is}n\'t in {that iobj/him}. '

    /* taking dobj from surface, but dobj isn't on iobj */
    takeFromNotOn = '{The dobj/he} {is}n\'t on {that iobj/him}. '

    /* taking dobj from an actor, but actor doesn't have iobj */
    takeFromNotInActor = '{The dobj/he} do{es}n\'t have {that iobj/him}. '

    /* actor won't let go of a possession */
    willNotLetGo(holder, obj)
    {
        gAction.setMessageParams('holder', holder, 'obj', obj);
        return '{The holder/he} won\'t let {you/him} have {that obj/him}. ';
    }

    /* travel attempted in a direction with no exit */
    cannotGoThatWay = '{You/he} can\'t go that way. '

    /* travel attempted in the dark in a direction with no exit */
    cannotGoThatWayInDark = 'It\'s too dark; {you/he} can\'t see
                             where {you\'re} going. '

    /* we don't know the way back for a GO BACK */
    cannotGoBack = '{You/he} do{es}n\'t know how to return from here. '

    /* cannot carry out a command from this location */
    cannotDoFromHere = '{You/he} can\'t do that from here. '

    /* cannot carry out travel while 'dest' is within 'cont' */
    invalidStagingContainer(cont, dest)
    {
        gMessageParams(cont, dest);
        return '{You/he} can\'t do that while {the dest/he} {is}
                {in cont}. ';
    }

    /* can't carry out travel because 'dest' isn't a valid staging location */
    invalidStagingLocation(dest)
    {
        gMessageParams(dest);
        return '{You/he} can\'t get {in dest}. ';
    }

    /* destination is too high to enter from here */
    nestedRoomTooHigh(obj)
    {
        return '\^' + obj.nameIs + ' too high to reach from here. ';
    }

    /* cannot carry out a command from a nested room */
    cannotDoFrom(obj)
        { return '{You/he} can\'t do that from ' + obj.theNameObj + '. '; }

    /* cannot carry out a command from within a vehicle in a nested room */
    vehicleCannotDoFrom(obj)
    {
        return '{You/he} can\'t do that while ' + obj.nameIs
            + ' ' + obj.location.actorInName + '. ';
    }

    /* cannot go that way in a vehicle */
    cannotGoThatWayInVehicle(traveler)
    {
        return '{You/he} can\'t do that ' + traveler.actorInName + '. ';
    }

    /* cannot push an object that way */
    cannotPushObjectThatWay(obj)
    {
        return '{You/he} can\'t go that way pushing '
            + obj.theNameObj + '. ';
    }

    /* cannot enter an exit-only passage */
    cannotEnterExitOnly(obj)
    {
        return '{You/he} can\'t enter ' + obj.theNameObj + ' from here. ';
    }

    /* must open door before going that way */
    mustOpenDoor(obj)
        { return '{You/he} must open ' + obj.theNameObj + ' first. '; }

    /* door closes behind actor during travel through door */
    doorClosesBehind(obj)
    {
        return 'After {you/he} go{es} through ' + obj.theName
            + ', ' + obj.itNom + ' close' + obj.verbEndingS
            + ' behind {you/him}. ';
    }

    /* the stairway does not go up/down */
    stairwayNotUp = '{The dobj/he} only go{es} down from here. '
    stairwayNotDown = '{The dobj/he} only go{es} up from here. '

    /* "wait" */
    timePasses = 'Time passes... '

    /* "hello" with no target actor */
    sayHello
    {
        return '{You/he} must be more specific about '
            + libMessages.whomPronoun + ' to talk to. ';
    }

    /* "yell" */
    okayYell = '{You/he} scream{s} as loud as {it actor/he} can. '

    /* "jump" */
    okayJump = '{You/he} jump{s} a little, and land{s} back where
                {it actor/he} started. '

    /* cannot jump over object */
    cannotJumpOver = '{You/he} can\'t jump over {that dobj/him}. '

    /* cannot jump off object */
    cannotJumpOff = '{You/he} can\'t jump off {that dobj/him}. '

    /* cannot jump off (with no direct object) from here */
    cannotJumpOffHere = 'There\'s nowhere to jump from here. '

    /* cannot talk to an object (because it makes no sense to do so) */
    notAddressable(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot talk to {that obj/him}. ';
    }

    /* actor won't respond to a request or other communicative gesture */
    noResponseFrom(actor)
    {
        return '\^' + actor.nameVerb('do') + ' not respond. ';
    }

    /* can't ask yourself about anything */
    cannotAskSelf =
        'Talking to {yourself/himself} won\'t accomplish anything. '

    /* can't tell yourself about anything */
    cannotTellSelf =
        'Talking to {yourself/himself} won\'t accomplish anything. '
    
    /* actor isn't interested in something being given/shown */
    notInterested(actor)
    {
        return '\^' + actor.nameVerb('do') + ' not appear interested. ';
    }

    /* object cannot hear actor */
    objCannotHearActor(obj)
    {
        return '\^' + obj.nameVerb('do')
            + ' not appear to hear {you/him}. ';
    }

    /* actor cannot see object being shown to actor */
    actorCannotSee(actor, obj)
    {
        return '\^' + actor.theName + ' does not appear to be able to see '
            + obj.theNameObj + '. ';
    }

    /* not a followable object */
    notFollowable = '{You/he} cannot follow {that dobj/him}. '

    /* cannot follow yourself */
    cannotFollowSelf = '{You/he} cannot follow {yourself}. '

    /* following an object that's in the same location as the actor */
    followAlreadyHere = '{The dobj/he} {is} right here. '

    /* 
     *   following an object that we *think* is in our same location (in
     *   other words, we're already in the location where we thought we
     *   last saw the object go), but it's too dark to see if that's
     *   really true 
     */
    followAlreadyHereInDark = '{The dobj/he} should be right here,
                               but {you/he} can\'t see {it dobj/him}. '

    /* trying to follow an object, but don't know where it went */
    followUnknown = '{You/he} do{es}n\'t know where {the dobj/he} went. '

    /* trying to follow from the wrong location */
    cannotFollowFromHere =
        '{You/he} cannot follow {that dobj/him} from here. '

    /* acknowledge a 'follow' for a target that was in sight */
    okayFollowInSight(loc)
    {
        return '{You/he} follow{s} {the dobj/him} '
            + loc.actorIntoName + '. ';
    }

    /* obj is not a weapon */
    notAWeapon = '{You/he} can\'t attack anything with {the iobj/him}. '

    /* no effect attacking obj */
    uselessToAttack = '{You/he} cannot attack {that iobj/him}. '

    /* pushing object has no effect */
    pushNoEffect = 'Pushing {the dobj/him} has no effect. '

    /* default 'push button' acknowledgment */
    okayPushButton { return withQuotes('Click.'); }

    /* lever is already in pushed state */
    alreadyPushed = '{It\'s dobj} already pushed as far as {it dobj/he}
                     will go. '

    /* default acknowledgment to pushing a lever */
    okayPushLever = '{You/he} push{es} {the dobj/him} to
                     {its/her dobj} stop. '

    /* pulling object has no effect */
    pullNoEffect = 'Pulling {the dobj/him} has no effect. '

    /* lever is already in pulled state */
    alreadyPulled = '{It\'s dobj} already pulled as far as {it dobj/he}
                     will go. '

    /* default acknowledgment to pulling a lever */
    okayPullLever = '{You/he} pull{s} {the dobj/him} to
                     {its/her dobj} stop. '

    /* default acknowledgment to pulling a spring-loaded lever */
    okayPullSpringLever = '{You/he} pull{s} {the dobj/him}, which
                           spring{s} back to {its/her} starting
                           position as soon as {you/he} let{s}
                           go of {it dobj/him}. '

    /* moving object has no effect */
    moveNoEffect = 'Moving {the dobj/him} has no effect. '

    /* cannot move object to other object */
    moveToNoEffect = 'This would accomplish nothing. '

    /* cannot push an object through travel */
    cannotPushTravel = 'This would accomplish nothing. '

    /* acknowledge pushing an object through travel */
    okayPushTravel(obj)
    {
        return '<.p>{You/he} push{es} ' + obj.theNameObj
            + ' into the area. ';
    }

    /* cannot use object as an implement to move something */
    cannotMoveWith = '{You/he} cannot move anything with {the iobj/him}. '

    /* cannot turn object */
    cannotTurn = '{You/he} cannot turn {that dobj/him}. '

    /* must specify setting to turn object to */
    mustSpecifyTurnTo = '{You/he} must specify the setting to
                         turn {that dobj/him} to. '

    /* cannot turn anything with object */
    cannotTurnWith = '{You/he} cannot turn anything with {that iobj/him}. '

    /* invalid setting for dial */
    turnToInvalid = '{The dobj/he} {has} no such setting. '

    /* default 'turn to' acknowledgment */
    okayTurnTo(val)
        { return 'Okay, {the dobj/he} {is} now set to ' + val + '. '; }

    /* switch is already on/off */
    alreadySwitchedOn = '{The dobj/he} {is} already on. '
    alreadySwitchedOff = '{The dobj/he} {is} already off. '

    /* default acknowledgment for switching on/off */
    okayTurnOn = 'Okay, {the dobj/he} {is} now on. '
    okayTurnOff = 'Okay, {the dobj/he} {is} now off. '

    /* default acknowledgment for eating something */
    okayEat = '{You/he} eat{s} {the dobj/him}. '

    /* object must be burning before doing that */
    mustBeBurning(obj)
    {
        return '{You/he} must light ' + obj.theNameObj
            + ' before {it actor/he} can do that. ';
    }

    /* match not lit */
    matchNotLit = '{The dobj/he} {is}n\'t lit. '

    /* lighting a match */
    okayBurnMatch = '{You/he} strike{s} {the dobj/him}, igniting a
                     small flame. '

    /* extinguishing a match */
    okayExtinguishMatch = '{You/he} put{s} out {the dobj/him}, which
                           disappear{s} into a cloud of ash. '

    /* trying to light a candle with no fuel */
    candleOutOfFuel = '{The dobj/he} {is} too burned down; {it/he} cannot
                       be lit. '

    /* lighting a candle */
    okayBurnCandle = '{You/he} light{s} {the dobj/him}. '

    /* extinguishing a candle that isn't lit */
    candleNotLit = '{The dobj/he} {is} not lit. '

    /* extinguishing a candle */
    okayExtinguishCandle = 'Done. '

    /* cannot consult object */
    cannotConsult = '{That dobj/he} {is} not something {you/he}
                     can consult. '

    /* cannot type anything on object */
    cannotTypeOn = '{You/he} cannot type anything on {that dobj/him}. '

    /* cannot enter anything on object */
    cannotEnterOn = '{You/he} cannot enter anything on {that dobj/him}. '

    /* cannot switch object */
    cannotSwitch = '{You/he} cannot switch {that dobj/him}. '

    /* cannot flip object */
    cannotFlip = '{You/he} cannot flip {that dobj/him}. '

    /* cannot turn object on/off */
    cannotTurnOn = '{That dobj/he} {is}n\'t something {you/he} can turn on. '
    cannotTurnOff =
        '{That dobj/he} {is}n\'t something {you/he} can turn off. '

    /* cannot light */
    cannotLight = '{You/he} cannot light {that dobj/him}. '

    /* cannot burn */
    cannotBurn = '{That dobj/he} {is} not something {you/he} can burn. '
    cannotBurnWith = '{You/he} cannot burn anything with {that iobj/him}. '

    /* cannot burn this specific direct object with this specific iobj */
    cannotBurnDobjWith = '{You/he} cannot light {the dobj/him}
                          with {the iobj/him}. '

    /* object is already burning */
    alreadyBurning = '{The dobj/he} {is} already burning. '

    /* cannot extinguish */
    cannotExtinguish = '{You/he} cannot extinguish {that dobj/him}. '

    /* cannot pour/pour in/pour on */
    cannotPour = '{That dobj/he} {is} not something {you/he} can pour. '
    cannotPourInto = '{You/he} cannot pour anything into {that dobj/him}. '
    cannotPourOnto = '{You/he} cannot pour anything onto {that dobj/him}. '

    /* cannot attach object to object */
    cannotAttach = '{You/he} cannot attach {that dobj/him} to anything. '
    cannotAttachTo = '{You/he} cannot attach anything to {that iobj/him}. '

    /* cannot detach object from object */
    cannotDetach = '{You/he} cannot detach {that dobj/him}. '
    cannotDetachFrom =
        '{You/he} cannot detach anything from {that iobj/him}. '

    /* breaking object would serve no purpose */
    shouldNotBreak = 'Breaking {that dobj/him} would serve no purpose. '

    /* cannot cut that */
    cutNoEffect = '{The iobj/he} can\'t seem to cut {the dobj/him}. '

    /* can't use iobj to cut anything */
    cannotCutWith = '{You/he} can\'t cut anything with {the iobj/him}. '

    /* cannot climb object */
    cannotClimb = '{That dobj/he} {is} not something {you/he} can climb. '

    /* object is not openable/closable */
    cannotOpen = '{That dobj/he} {is} not something {you/he} can open. '
    cannotClose = '{That dobj/he} {is} not something {you/he} can close. '

    /* already open/closed */
    alreadyOpen = '{The dobj/he} {is} already open. '
    alreadyClosed = '{The dobj/he} {is} already closed. '

    /* already locked/unlocked */
    alreadyLocked = '{The dobj/he} {is} already locked. '
    alreadyUnlocked = '{The dobj/he} {is} already unlocked. '

    /* object is currently open/closed */
    currentlyOpen = '{It\'s dobj} currently open. '
    currentlyClosed = '{It\'s dobj} currently closed. '

    /* cannot look in container because it's closed */
    cannotLookInClosed = '{The dobj/he} {is} closed. '

    /* object is not lockable/unlockable */
    cannotLock = '{That dobj/he} {is} not something {you/he} can lock. '
    cannotUnlock = '{That dobj/he} {is} not something {you/he} can unlock. '

    /* object is not a key */
    cannotLockWith =
        '{The iobj/he} do{es}n\'t look suitable for locking that. '
    cannotUnlockWith =
        '{The iobj/he} do{es}n\'t look suitable for unlocking that. '

    /* the key (iobj) does not fit the lock (dobj) */
    keyDoesNotFitLock = '{The iobj/he} does not fit the lock. '

    /* found key on keyring */
    foundKeyOnKeyring(ring, key)
    {
        gMessageParams(ring, key);
        return '{You/he} tr{ies} each key on {the ring/him}, and
            find{s actor} that {the key/he} fit{s} the lock. ';
    }

    /* failed to find a key on keyring */
    foundNoKeyOnKeyring(ring)
    {
        gMessageParams(ring);
        return '{You/he} tr{ies} each key on {the ring/him},
            but cannot find anything that fits the lock. ';
    }

    /* not edible/drinkable */
    cannotEat = '{The dobj/he} do{es} not appear to be edible. '
    cannotDrink = '{That dobj/he} do{es} not appear to be something
                   {you/he} can drink. '

    /* cannot clean object */
    cannotClean = '{You/he} wouldn\'t know how to clean {that dobj/him}. '
    cannotCleanWith = '{You/he} can\'t clean anything with {that iobj/him}. '

    /* cannot attach key (dobj) to (iobj) */
    cannotAttachKeyTo = '{You/he} cannot attach {the dobj/him}
                         to {that iobj/him}. '

    /* actor cannot sleep */
    cannotSleep = '{You/he} do{es}n\'t need to sleep right now. '

    /* cannot sit/lie/stand/get on/get out of */
    cannotSitOn = '{That dobj/he} {is}n\'t something {you/he} can sit on. '
    cannotLieOn = '{That dobj/he} {is}n\'t something {you/he} can lie on. '
    cannotStandOn = '{You/he} can\'t stand on {that dobj/him}. '
    cannotBoard = '{You/he} can\'t board {that dobj/him}. '
    cannotUnboard = '{You/he} can\'t get out of {that dobj/him}. '
    cannotGetOffOf = '{You/he} can\'t get off of {that dobj/him}. '

    /* cannot sit/lie/stand on something being held */
    cannotEnterHeld = '{You/he} can\'t do that while {you\'re}
                       holding {the dobj/him}. '

    /* cannot get out (of current location) */
    cannotGetOut = '{You\'re} not in anything {you/he} can disembark. '

    /* actor is already standing/sitting on/lying on */
    alreadyStanding = '{You\'re} already standing. '
    alreadyStandingOn = '{You\'re} already standing {on dobj}. '
    alreadySittingOn = '{You\'re} already sitting {on dobj}. '
    alreadyLyingOn = '{You\'re} already lying {on dobj}. '

    /* getting off something you're not on */
    notOnPlatform = '{You\'re} not {on dobj}. '

    /* no room to stand/sit/lie on dobj */
    noRoomToStand = 'There\'s no room for {you/him} to stand {on dobj}. '
    noRoomToSit = 'There\'s no room for {you/him} to sit {on dobj}. '
    noRoomToLie = 'There\'s no room for {you/him} to lie {on dobj}. '

    /* default report for standing up/sitting down/lying down */
    okayStand = 'Okay, {you\'re} now standing. '
    okayStandOn = 'Okay, {you\'re} now standing {on dobj}. '
    okaySit = 'Okay, {you\'re} now seated. '
    okaySitOn = 'Okay, {you\'re} now sitting {on dobj}. '
    okayLieOn = 'Okay, {you\'re} now lying {on dobj}. '
    okayLie = 'Okay, {you\'re} now lying down. '

    /* default report for standing up on a given object */
    okayStandOnObj(obj)
        { return 'Okay, {you\'re} now standing ' + obj.actorInName + '. '; }
    okaySitOnObj(obj)
        { return 'Okay, {you\'re} now sitting ' + obj.actorInName + '. '; }
    okayLieOnObj(obj)
        { return 'Okay, {you\'re} now lying ' + obj.actorInName + '. '; }

    /* default report for getting off of a platform */
    okayNotStandingOn = 'Okay, {you\'re} no longer {on dobj}. '

    /* cannot fasten/unfasten */
    cannotFasten = '{You/he} cannot fasten {the dobj/him}. '
    cannotFastenTo = '{You/he} cannot fasten anything to {the iobj/him}. '
    cannotUnfasten = '{You/he} cannot unfasten {the dobj/him}. '
    cannotUnfastenFrom = '{You/he} cannot unfasten anything
                          from {the iobj/him}. '

    /* cannot plug/unplug */
    cannotPlugIn = '{You/he} see{s} no way to plug in {the dobj/him}. '
    cannotPlugInTo = '{You/he} see{s} no way to plug anything
                      into {the iobj/him}. '
    cannotUnplug = '{You/he} see{s} no way to unplug {the dobj/him}. '
    cannotUnplugFrom = '{You/he} see{s} no way to unplug anything
                        from {the iobj/him}. '

    /* cannot screw/unscrew */
    cannotScrew = '{You/he} see{s} no way to screw {the dobj/him}. '
    cannotScrewWith = '{You/he} cannot screw anything with {the iobj/him}. '
    cannotUnscrew = '{You/he} see{s} no way to unscrew {the dobj/him}. '
    cannotUnscrewWith = '{You/he} cannot unscrew anything
                         with {the iobj/him}. '

    /* cannot enter/go through */
    cannotEnter = '{That/he dobj} {is} not something {you/he} can enter. '
    cannotGoThrough =
        '{That/he dobj} {is} not something {you/he} can go through. '

    /* can't throw something at itself */
    cannotThrowAtSelf = '{You/he} can\'t throw {that dobj/him} at {itself}. '

    /* can't throw something at an object inside itself */
    cannotThrowAtContents = '{You/he} must remove {the iobj/him}
                             from {the dobj/him} before {it actor/he} can
                             do that. '

    /* shouldn't throw something at the floor */
    shouldNotThrowAtFloor =
        '{You/he} should just put {it dobj/him} down instead. '

    /* object bounces off target and falls to destination */
    throwHit(obj, target, dest)
    {
        return '\^' + obj.nameVerb('hit') + ' ' + target.theNameObj
            + ' without any obvious effect, and falls '
            + libMessages.(dest.putDestMessage)(dest) + '. ';
    }

    /* target catches object */
    throwCatch(obj, target)
    {
        return '\^' + target.nameVerb('catch')
            + ' ' + obj.theNameObj + '. ';
    }

    /* target does not want to catch anything */
    willNotCatch(catcher)
    {
        return '\^' + catcher.nameVerb('do')
            + 'n\'t look like ' + catcher.itNom + ' want'
            + catcher.verbEndingS + ' to catch anything. ';
    }

    /* cannot kiss something */
    cannotKiss = 'Kissing {the dobj/him} has no obvious effect. '

    /* person uninterested in being kissed */
    cannotKissActor = '{The dobj/he} probably wouldn\'t like that. '

    /* cannot kiss yourself */
    cannotKissSelf = '{You/he} cannot kiss {yourself}. '

    /* it is now dark at actor's location */
    newlyDark = 'It is now pitch black. '
;

/*
 *   Non-player character verb messages.  By default, we inherit all of
 *   the messages defined for the player character, but we override some
 *   that must be rephrased slightly to make sense for NPC's.
 */
npcActionMessages: playerActionMessages
    /* "wait" */
    timePasses = '{You/he} waits... '

    /* trying to move a fixed item */
    cannotMoveFixed = '{You/he} cannot move {that dobj/him}. '

    /* trying to move a component object */
    cannotMoveComponent(loc)
    {
        return '{You/he} cannot do that because {the dobj/he} {is} part
            of ' + loc.theNameObj + '. ';
    }

    /* default 'take' response */
    okayTake = '{You/he} take{s} {the dobj/him}. '

    /* default 'drop' response */
    okayDrop = '{You/he} put{s} down {the dobj/him}. '

    /* default 'put in' response */
    okayPutIn = '{You/he} put{s} {the dobj/him} in {the iobj/him}. '

    /* default 'put on' response */
    okayPutOn = '{You/he} put{s} {the dobj/him} on {the iobj/him}. '

    /* default response to 'wear obj' */
    okayWear = '{You/he} put{s} on {the dobj/him}. '

    /* default response to 'doff obj' */
    okayDoff = '{You/he} take{s} off {the dobj/him}. '

    /* default response to open/close */
    okayOpen = '{You/he} open{s} {the dobj/him}. '
    okayClose = '{You/he} close{s} {the dobj/him}. '

    /* default response to lock/unlock */
    okayLock = '{You/he} locks{s} {the dobj/him}. '
    okayUnlock = '{You/he} unlock{s} {the dobj/him}. '

    /* push/pull/move with no effect */
    pushNoEffect =
        '{You/he} tr{ies} to push {the dobj/him}, with no obvious effect. '
    pullNoEffect =
        '{You/he} tr{ies} to pull {the dobj/him}, with no obvious effect. '
    moveNoEffect =
        '{You/he} tr{ies} to move {the dobj/him}, with no obvious effect. '
    moveToNoEffect =
        '{You/he} leave{s} {the dobj/he} where {it/he} {is}. '

    /* object is too large for container */
    tooLargeForContainer(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{You/he} cannot do that because {the obj/he} {is}
            too large for {the cont/him}. ';
    }

    /* container doesn't have room for object */
    containerTooFull(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{You/he} cannot do that because {the cont/he} {is}
            already too full to hold {the obj/him}. ';
    }

    /* surface doesn't have room for object */
    surfaceTooFull(obj, cont)
    {
        gMessageParams(obj, cont);
        return '{You/he} cannot do that because there\'s
            no room for {the obj/him} on {the cont/him}. ';
    }

    /* the dobj doesn't fit on this keyring */
    objNotForKeyring = '{You/he} cannot do that because
                        {that dobj/he} do{es}n\'t fit on {the iobj/him}. '

    /* taking dobj from iobj, but dobj isn't in iobj */
    takeFromNotIn = '{You/he} cannot do that because
                     {the dobj/he} {is}n\'t in {that iobj/him}. '

    /* taking dobj from surface, but dobj isn't on iobj */
    takeFromNotOn = '{You/he} cannot do that because
                     {the dobj/he} {is}n\'t on {that iobj/him}. '

    /* cannot jump off (with no direct object) from here */
    cannotJumpOffHere = 'There\'s nowhere for {you/him} to jump. '

    /* should not break object */
    shouldNotBreak = '{You/he} do{es}n\'t want to break {that dobj/him}. '

    /* default report for standing up/sitting down/lying down */
    okayStand = '{You/he} stand{s} up. '
    okaySitOn = '{You/he} sit{s} {on dobj}. '
    okayLieOn = '{You/he} lie{s} down {on dobj}. '

    /* default 'turn to' acknowledgment */
    okayTurnTo = '{You/he} turn{s} {the dobj/him} to {literal}. '

    /* default 'push button' acknowledgment */
    okayPushButton = '{You/he} push{es} {the dobj/him}. '

    /* default acknowledgment for switching on/off */
    okayTurnOn = '{You/he} turn{s} {the dobj/him} on. '
    okayTurnOff = '{You/he} turn{s} {the dobj/him} off. '

    /* the key (iobj) does not fit the lock (dobj) */
    keyDoesNotFitLock = '{You/he} tr{ies} {the iobj/he}, but {it iobj/he}
                         do{es} not fit the lock. '

    /* acknowledge entering "follow" mode */
    okayFollowMode = '"Okay, I will follow {the dobj/him}." '

    /* extinguishing a candle */
    okayExtinguishCandle = '{You/he} extinguish{es} {the dobj/him}. '
;

/* ------------------------------------------------------------------------ */
/*
 *   Listers
 */

/*
 *   The basic "room lister" object - this is the object that we use by
 *   default with showList() to construct the listing of the portable
 *   items in a room when displaying the room's description.  
 */
roomLister: Lister
    /* show the prefix/suffix in wide mode */
    showListPrefixWide(itemCount, pov, parent) { "{You/he} see{s} "; }
    showListSuffixWide(itemCount, pov, parent) { " here. "; }

    /* show the tall prefix */
    showListPrefixTall(itemCount, pov, parent) { "{You/he} see{s}:"; }
;

/*
 *   The basic room lister for dark rooms 
 */
darkRoomLister: Lister
    showListPrefixWide(itemCount, pov, parent)
        { "In the darkness {you/he} can see "; }

    showListSuffixWide(itemCount, pov, parent) { ". "; }

    showListPrefixTall(itemCount, pov, parent)
        { "In the darkness {you/he} see{s}:"; }
;

/*
 *   Standard inventory lister for actors - this will work for the player
 *   character and NPC's as well.  
 */
actorInventoryLister: InventoryLister
    showListPrefixWide(itemCount, pov, parent)
        { "\^<<parent.nameIs>> carrying "; }
    showListSuffixWide(itemCount, pov, parent)
        { ". "; }

    showListPrefixTall(itemCount, pov, parent)
        { "\^<<parent.nameIs>> carrying:"; }
    showListContentsPrefixTall(itemCount, pov, parent)
        { "<<parent.aName>>, who <<parent.verbToBe>> carrying:"; }

    showListEmpty(pov, parent)
        { "\^<<parent.nameIs>> empty-handed. "; }
;

/*
 *   Special inventory lister for non-player character descriptions - long
 *   form lister.  This is used to display the inventory of an NPC as part
 *   of the full description of the NPC.
 *   
 *   This long form lister is meant for actors with lengthy descriptions.
 *   We start the inventory listing on a new line, and use the actor's
 *   full name in the list preface.  
 */
actorHoldingDescInventoryListerLong: actorInventoryLister
    showListPrefixWide(itemCount, pov, parent)
        { "<.p>\^<<parent.nameIs>> carrying "; }

    /* 
     *   don't show anything as part of the actor's description when the
     *   actor is empty-handed 
     */
    showListEmpty(pov, parent) { }
;

/* short form of non-player character description inventory lister */
actorHoldingDescInventoryListerShort: actorInventoryLister
    showListPrefixWide(itemCount, pov, parent)
        { "\^<<parent.itIs>> carrying "; }
    showListEmpty(pov, parent) { }
;

/*
 *   standard inventory lister for items being worn by an actor
 */
actorWearingLister: WearingLister
    showListPrefixWide(itemCount, pov, parent)
        { "\^<<parent.nameIs>> wearing "; }
    showListSuffixWide(itemCount, pov, parent)
        { ". "; }

    showListPrefixTall(itemCount, pov, parent)
        { "\^<<parent.nameIs>> wearing:"; }
    showListContentsPrefixTall(itemCount, pov, parent)
        { "<<parent.aName>>, wearing:"; }
    
    showListEmpty(pov, parent) { }
;

/*
 *   special inventory lister for items being worn by an NPC, used for the
 *   full description of an NPC 
 */
actorHoldingDescWearingLister: actorWearingLister
;

/*
 *   Base contents lister for things.  This is used to display the
 *   contents of things shown in room and inventory listings; we subclass
 this for various purposes
 */
class BaseThingContentsLister: Lister
    showListPrefixWide(itemCount, pov, parent)
        { "\^<<parent.nameVerb('contain')>> "; }
    showListSuffixWide(itemCount, pov, parent)
        { ". "; }
    showListPrefixTall(itemCount, pov, parent)
        { "\^<<parent.nameVerb('contain')>>:"; }
    showListContentsPrefixTall(itemCount, pov, parent)
        { "<<parent.aName>>, which contain<<parent.verbEndingS>>:"; }
;

/* 
 *   Contents lister for things.  This is used to display the second-level
 *   contents lists for things listed in the top-level list for a room
 *   description, inventory listing, or object description.  
 */
thingContentsLister: ContentsLister, BaseThingContentsLister
;

/*
 *   Contents lister for openable things.
 */
openableContentsLister: thingContentsLister
    showListEmpty(pov, parent)
    {
        "\^<<parent.itIsContraction>> <<parent.openDesc>>. ";
    }
    showListPrefixWide(itemCount, pov, parent)
    {
        "\^<<parent.itIsContraction>> <<parent.openDesc>>, and
        contain<<parent.verbEndingS>> ";
    }
;

/*
 *   Contents lister for descriptions of things - this is used to display
 *   the contents of a thing as part of the long description of the thing
 *   (in response to an "examine" command); it differs from a regular
 *   thing contents lister in that we use a pronoun to refer to the thing,
 *   rather than its full name, since the full name was probably just used
 *   in the basic long description.  
 */
thingDescContentsLister: DescContentsLister, BaseThingContentsLister
    showListPrefixWide(itemCount, pov, parent)
        { "\^<<parent.itVerb('contain')>> "; }
;

/*
 *   Contents lister for descriptions of things whose contents are
 *   explicitly inspected ("look in").  This differs from a regular
 *   contents lister in that we explicitly say that the object is empty if
 *   it's empty.
 */
thingLookInLister: DescContentsLister, BaseThingContentsLister
    showListEmpty(pov, parent)
    {
        /* 
         *   Indicate that the list is empty, but make this a default
         *   descriptive report.  This way, if we report any special
         *   descriptions for items contained within the object, we'll
         *   suppress this default description that there's nothing to
         *   describe, which is clearly wrong if there are
         *   specially-described contents. 
         */
        gMessageParams(parent);
        defaultDescReport('{You/he} see{s} nothing in {the parent/him}. ');
    }
;

/*
 *   Contents lister for newly-revealed objects after opening a container. 
 */
openingLister: BaseThingContentsLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(itemCount, pov, parent)
        { "Opening <<parent.theNameObj>> reveals "; }
;

/*
 *   Base class for contents listers for a surface 
 */
class BaseSurfaceContentsLister: Lister
    showListPrefixWide(itemCount, pov, parent)
        { "On <<parent.theNameObj>> <<itemCount == 1 ? 'is' : 'are'>> "; }
    showListSuffixWide(itemCount, pov, parent)
        { ". "; }
    showListPrefixTall(itemCount, pov, parent)
        { "On <<parent.theNameObj>> <<itemCount == 1 ? 'is' : 'are'>>:"; }
    showListContentsPrefixTall(itemCount, pov, parent)
        { "<<parent.aName>>, on which <<itemCount == 1 ? 'is' : 'are'>>:"; }
;

/*
 *   Contents lister for a surface 
 */
surfaceContentsLister: ContentsLister, BaseSurfaceContentsLister
;

/*
 *   Contents lister for explicitly looking in a surface 
 */
surfaceLookInLister: DescContentsLister, BaseSurfaceContentsLister
    showListEmpty(pov, parent)
    {
        /* show a default message indicating the surface is empty */
        gMessageParams(parent);
        defaultDescReport('{You/he} see{s} nothing on {the parent/him}. ');
    }
;

/*
 *   Contents lister for a surface, used in a long description. 
 */
surfaceDescContentsLister: DescContentsLister, BaseSurfaceContentsLister
    showListPrefixWide(itemCount, pov, parent)
        { "On <<parent.theNameObj>> <<itemCount == 1 ? 'is' : 'are'>> "; }
;

/*
 *   Contents lister for room parts
 */
roomPartContentsLister: surfaceContentsLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    /* the part I'm listing */
    part_ = nil
;

/*
 *   contents lister for room parts, used in a long description 
 */
roomPartDescContentsLister: surfaceDescContentsLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    part_ = nil
;

/*
 *   Look-in lister for room parts.  Most room parts act like surfaces
 *   rather than containers, so base this lister on the surface lister.  
 */
roomPartLookInLister: surfaceLookInLister
    isListed(obj)
    {
        /* list the object if it's listed in the room part */
        return obj.isListedInRoomPart(part_);
    }

    part_ = nil
;
                         

/*
 *   Contents lister for a generic in-line list entry. 
 */
inlineListingContentsLister: ContentsLister
    showListEmpty(pov, parent) { }
    showListPrefixWide(cnt, pov, parent)
        { " (which contains "; }
    showListSuffixWide(itemCount, pov, parent)
        { ")"; }
;

/*
 *   Contents lister for keyring list entry.  This is used to display a
 *   keyring's contents in-line with the name of the keyring,
 *   parenthetically. 
 */
keyringListingContentsLister: inlineListingContentsLister
    showListPrefixWide(cnt, pov, parent)
        { " (with "; }
    showListSuffixWide(cnt, pov, parent)
        { " attached)"; }
;

/*
 *   Contents lister for "examine <keyring>" 
 */
keyringExamineContentsLister: DescContentsLister
    showListEmpty(pov, parent)
        { "\^<<parent.nameIs>> empty. "; }
    showListPrefixWide(cnt, pov, parent)
        { "Attached to <<parent.theNameObj>> <<cnt == 1 ? 'is' : 'are'>> "; }
    showListSuffixWide(itemCount, pov, parent)
        { ". "; }
;

/*
 *   Lister for actors aboard a traveler.  This is used to list the actors
 *   on board a vehicle when the vehicle arrives or departs a location.  
 */
aboardVehicleLister: Lister
    showListPrefixWide(itemCount, pov, parent)
        { " (carrying "; }
    showListSuffixWide(itemCount, pov, parent)
        { ")"; }

    /* list anything whose isListedAboardVehicle returns true */
    isListed(obj) { return obj.isListedAboardVehicle; }
;

/*
 *   Finish Options lister.  This lister is used to offer the player
 *   options in finishGame(). 
 */
finishOptionsLister: Lister
    showListPrefixWide(cnt, pov, parent)
    {
        "<.p>Would you like to ";
    }
    showListSuffixWide(cnt, pov, parent)
    {
        /* end the question, add a newline, and show the ">" prompt */
        "?\n&gt;";
    }
    isListed(obj) { return true; }
    showListItem(obj, options, pov, infoTab)
    {
        /* obj is a FinishOption object; show its description */
        obj.desc;
    }
    
    showListSeparator(options, curItemNum, totalItems)
    {
        /* 
         *   for the last separator, show "or" rather than "and"; for
         *   others, inherit the default handling 
         */
        if (curItemNum + 1 == totalItems)
        {
            if (totalItems == 2)
                " or ";
            else
                ", or ";
        }
        else
            inherited(options, curItemNum, totalItems);
    }
;

/*
 *   Equivalent list state lister.  This shows a list of state names for a
 *   set of otherwise indistinguishable items.  We show the state names in
 *   parentheses, separated by commas only (i.e., no "and" separating the
 *   last two items); we use this less verbose format so that we blend
 *   into the larger enclosing list more naturally.
 *   
 *   The items to be listed are EquivalentStateInfo objects.  
 */
equivalentStateLister: Lister
    showListPrefixWide(cnt, pov, parent) { " ("; }
    showListSuffixWide(cnt, pov, parent) { ")"; }
    isListed(obj) { return true; }
    showListItem(obj, options, pov, infoTab)
    {
        "<<spellIntBelow(obj.stateCount, 100)>> <<obj.name>>";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        if (curItemNum < totalItems)
            ", ";
    }
;

/* in case the exits module isn't included in the build */
property destName_, destIsBack_, others_, enableHyperlinks;

/*
 *   Basic room exit lister.  This shows a list of the apparent exits from
 *   a location.
 *   
 *   The items to be listed are DestInfo objects.  
 */
class ExitLister: Lister
    showListPrefixWide(cnt, pov, parent)
    {
        if (cnt == 1)
            "The only obvious exit leads ";
        else
            "Obvious exits lead ";
    }
    showListSuffixWide(cnt, pov, parent) { ". "; }
    isListed(obj) { return true; }
    showListItem(obj, options, pov, infoTab)
    {
        /* 
         *   Show the back-to-direction prefix, if we don't know the
         *   destination name but this is the back-to direction: "back to
         *   the north" and so on. 
         */
        if (obj.destIsBack_ && obj.destName_ == nil)
            say(obj.dir_.backToPrefix + ' ');
        
        /* show the direction */
        showListItemDirName(obj, nil);

        /* if the destination is known, show it as well */
        if (obj.destName_ != nil)
        {
            /* 
             *   if we have a list of other directions going to the same
             *   place, show it parenthetically 
             */
            otherExitLister.showListAll(obj.others_, 0, 0);
            
            /* 
             *   Show our destination name.  If we know the "back to"
             *   destination name, show destination names in the format
             *   "east, to the living room" so that they are consistent
             *   with "west, back to the dining room".  Otherwise, just
             *   show "east to the living room".  
             */
            if ((options & hasBackNameFlag) != 0)
                ",";

            /* if this is the way back, say so */
            if (obj.destIsBack_)
                " back";

            /* show the destination */
            " to <<obj.destName_>>";
        }
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /* 
         *   if we have a "back to" name, use the "long" notation - this is
         *   important because we'll use commas in the directions with
         *   known destination names 
         */
        if ((options & hasBackNameFlag) != 0)
            options |= ListLong;

        /* 
         *   for a two-item list, if either item has a destination name,
         *   show a comma or semicolon (depending on 'long' vs 'short' list
         *   mode) before the "and"; for anything else, use the default
         *   handling 
         */
        if (curItemNum == 1
            && totalItems == 2
            && (options & hasDestNameFlag) != 0)
        {
            if ((options & ListLong) != 0)
                "; and ";
            else
                ", and ";
        }
        else
            inherited(options, curItemNum, totalItems);
    }

    /* show a direction name, hyperlinking it if appropriate */
    showListItemDirName(obj, initCap)
    {
        local dirname;
        
        /* get the name */
        dirname = obj.dir_.name;

        /* capitalize the first letter, if desired */
        if (initCap)
            dirname = dirname.substr(1,1).toUpper() + dirname.substr(2);

        /* show the name with a hyperlink or not, as configured */
        if (libGlobal.exitListerObj.enableHyperlinks)
            say(aHref(dirname, dirname));
        else
            say(dirname);
    }

    /* this lister shows destination names */
    listerShowsDest = true

    /* 
     *   My special options flag: at least one object in the list has a
     *   destination name.  The caller should set this flag in our options
     *   if applicable. 
     */
    hasDestNameFlag = ListerCustomFlag(1)
    hasBackNameFlag = ListerCustomFlag(2)
    nextCustomFlag = ListerCustomFlag(3)
;

/*
 *   Show a list of other exits to a given destination.  We'll show the
 *   list parenthetically, if there's a list to show.  The objects to be
 *   listed are Direction objects.  
 */
otherExitLister: Lister
    showListPrefixWide(cnt, pov, parent) { " (or "; }
    showListSuffixWide(cnt, pov, parent) { ")"; }
    isListed(obj) { return true; }
    showListItem(obj, options, pov, infoTab)
    {
        if (libGlobal.exitListerObj.enableHyperlinks)
            say(aHref(obj.name, obj.name));
        else
            say(obj.name);
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /* 
         *   simply show "or" for all items (these lists are usually
         *   short, so omit any commas) 
         */
        if (curItemNum != totalItems)
            " or ";
    }
;

/*
 *   Show room exits as part of a room description, using the "verbose"
 *   sentence-style notation.  
 */
lookAroundExitLister: ExitLister
    showListPrefixWide(cnt, pov, parent)
    {
        /* add a paragraph break before the exit listing */
        "<.roompara>";

        /* inherit default handling */
        inherited(cnt, pov, parent);
    }    
;

/*
 *   Show room exits as part of a room description, using the "terse"
 *   notation. 
 */
lookAroundTerseExitLister: ExitLister
    showListPrefixWide(cnt, pov, parent)
    {
        "<.roompara><.parser>Obvious exits: ";
    }
    showListItem(obj, options, pov, infoTab)
    {
        /* show the direction name */
        showListItemDirName(obj, true);
    }
    showListSuffixWide(cnt, pov, parent)
    {
        "<./parser> ";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /* just show a comma between items */
        if (curItemNum != totalItems)
            ", ";
    }

    /* this lister does not show destination names */
    listerShowsDest = nil
;

/*
 *   Show room exits in response to an explicit request (such as an EXITS
 *   command).  
 */
explicitExitLister: ExitLister
    showListEmpty(pov, parent)
    {
        "There are no obvious exits. ";
    }
;

/*
 *   Show room exits in the status line (used in HTML mode only)
 */
statuslineExitLister: ExitLister
    showListEmpty(pov, parent)
    {
        "<br><b>Exits:</b> <i>None</i><br>";
    }
    showListPrefixWide(cnt, pov, parent)
    {
        "<br><b>Exits:</b> ";
    }
    showListSuffixWide(cnt, pov, parent)
    {
        "<br>";
    }
    showListItem(obj, options, pov, infoTab)
    {
        "<a plain href='<<obj.dir_.name>>'><<obj.dir_.name>></a>";
    }
    showListSeparator(options, curItemNum, totalItems)
    {
        /* just show a space between items */
        if (curItemNum != totalItems)
            " &nbsp; ";
    }

    /* this lister does not show destination names */
    listerShowsDest = nil
;

