-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001-2005  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.Characters.Latin_1;
with Ada.Command_Line;
with Ada.Integer_Text_IO;
with Ada.IO_Exceptions;
with Ada.Strings;
with Ada.Strings.Fixed;
with Ada.Text_IO;
with Version_ID;

package body Misc is

   -- Two subprograms to make writing strings easier.
   procedure Character_IO_Put (F : in Character_IO.File_Type;
                               S : in String) is
   begin
      for I in S'First..S'Last loop
         Character_IO.Write(F, S(I));
      end loop;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Character_IO_Put");
         raise;
   end Character_IO_Put;

   -- Two subprograms to make writing strings easier.
   procedure Character_IO_Put_Line (F : in Character_IO.File_Type;
                                    S : in String) is
   begin
      Character_IO_Put(F, S);
      Character_IO.Write(F, Ada.Characters.Latin_1.LF);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Character_IO_Put_Line");
         raise;
   end Character_IO_Put_Line;

   -- How to handle errors and debugging.
   procedure Error (The_Error : in String) is
   begin
      if Ada.Text_IO.Is_Open(Result_File) then
         Ada.Text_IO.Put_Line(Result_File,
                              "Topal: Fatal error: " & The_Error);
      end if;
      Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                           "Topal: Fatal error: " & The_Error);
      raise Panic;
   end Error;

   procedure ErrorNE (The_Error : in String) is
   begin
      if Ada.Text_IO.Is_Open(Result_File) then
         Ada.Text_IO.Put_Line(Result_File,
                              "Topal: Error: " & The_Error);
      end if;
      Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                           "Topal: Error: " & The_Error);
   end ErrorNE;

   procedure Debug (Message : in String) is
   begin
      if Config.Debug then
         if Ada.Text_IO.Is_Open(Result_File) then
            Ada.Text_IO.Put_Line(Result_File,
                                 "Topal: Debug: " & Message);
         end if;
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Topal: Debug: " & Message);
      end if;
   end Debug;

   -- Strings to integers.

   function String_To_Integer (S : String) return Integer is
      use Ada.Integer_Text_IO;
      use Ada.IO_Exceptions;
      L : Positive;
      N : Integer;
   begin
      Get(S, N, L);
      return N;
   exception
      when Data_Error =>
         raise String_Not_Integer;
      when End_Error =>
         raise String_Not_Integer;
      when others =>
         Ada.Text_IO.Put_Line("*** Problem in String_To_Integer. ***");
         raise;
   end String_To_Integer;

   function String_To_Integer (S : UBS) return Integer is
   begin
      return String_To_Integer(ToStr(S));
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.String_To_Integer (B)");
         raise;
   end String_To_Integer;

   -- Throw away leading blanks from a string.
   function Trim_Leading_Spaces (S : String) return String is
      use Ada.Strings.Fixed;
   begin
      return S(Ada.Strings.Fixed.Index_Non_Blank(S)..S'Last);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Trim_Leading_Spaces");
         raise;
   end Trim_Leading_Spaces;

   -- Create our own temporary file names.
   function Temp_File_Name (Tail : String) return String is
   begin
      return ToStr(Topal_Directory)
        & "/temp"
        & Trim_Leading_Spaces(Integer'Image(Our_PID))
        & Tail;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Temp_File_Name");
         raise;
   end Temp_File_Name;

   -- An `unbounded' Get_Line.
   function Unbounded_Get_Line (File : in Ada.Text_IO.File_Type)
                                return UBS is

      use Ada.Text_IO;

      function More_Input return UBS is
         Input : String (1 .. 1024);
         Last  : Natural;
         use type UBS;
      begin
         Get_Line(File, Input, Last);
         if Last < Input'Last then
            return ToUBS(Input(1..Last));
         else
            return ToUBS(Input(1..Last)) & More_Input;
         end if;
      end More_Input;

   begin
      return More_Input;
   exception
      when Ada.IO_Exceptions.End_Error =>
         -- Just let it through and let the caller sort it out.
         raise;
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Unbounded_Get_Line (A)");
         raise;
   end Unbounded_Get_Line;

   function Unbounded_Get_Line return UBS is
   begin
      return Unbounded_Get_Line(Ada.Text_IO.Standard_Input);
   exception
      when Ada.IO_Exceptions.End_Error =>
         -- Just let it through and let the caller sort it out.
         raise;
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Unbounded_Get_Line (B)");
         raise;
   end Unbounded_Get_Line;

   -- Open and close the result file.
   procedure Open_Result_File (Resultfile : in String) is
   begin
      Debug("Creating result file with name `" & Resultfile & "'");
      Ada.Text_IO.Create(File => Result_File,
                         Mode => Ada.Text_IO.Append_File,
                         Name => Resultfile);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Open_Result_File");
         raise;
   end Open_Result_File;

   procedure Close_Result_File is
   begin
      if Ada.Text_IO.Is_Open(Result_File) then
         Ada.Text_IO.Close(Result_File);
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Close_Result_File");
         raise;
   end Close_Result_File;

   procedure Disclaimer is
      use Ada.Text_IO;
   begin
      Put_Line("Topal " & Version_ID.Release
               & "  Copyright (C) 2001-2005 Phillip J. Brooke");
      Put_Line("Topal comes with ABSOLUTELY NO WARRANTY; "
               & "for details see the file `COPYING'.");
      Put_Line("This is free software, and you are welcome to redistribute "
               & "it under certain ");
      Put_Line("conditions; again, see the file `COPYING'.");
      New_Line;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Disclaimer");
         raise;
   end Disclaimer;

   function Value_Nonempty (V : in UBS) return UBS is
   begin
      if ToStr(V) = "" then
         raise Need_Nonempty_String;
      end if;
      return V;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Value_Nonempty (A)");
         raise;
   end Value_Nonempty;

   function Value_Nonempty (V : UBS) return String is
   begin
      return ToStr(Value_Nonempty(V));
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Misc.Value_Nonempty (B)");
         raise;
   end Value_Nonempty;

   -- Given a string, A, we want to split it up.  Really, we would
   -- like to properly honour bash-style quoting.
   -- At the moment, we'll simply do a space-separated run.
   -- Then added `stuffing'.  `"' will group things as an argument (i.e., stop the
   -- search).  `"' can be included literally by stuffing: `""'.
   function Split_Arguments (A : UBS) return UBS_Array is
      BA : UBS_Big_Array;
      AS : String := ToStr(A);
      TI : Integer := 0;
      use UAP;

      -- Recurse:
      -- Given a string, extract one token from it.
      -- Then recurse with the rest of the string.
      procedure Grab_Next_Token (A : in String) is
         I        : Natural;
         No_More  : Boolean := False;
         T        : UBS;
         Quoted   : Boolean := False;
         use type UBS;
      begin
         Debug("Grab_Next_Token invoked with `"
               & A & "'");
         -- Only do this if we're actually been given something.
         if A'Length /= 0 then
            -- Find first non-blank.
            I := A'First;
        Start_Loop:
            loop
               if A(I) = ' ' then
                  -- Advance.
                  I := I + 1;
                  -- Check for termination without finding new token.
                  if I > A'Last then
                     No_More := True;
                     exit Start_Loop;
                  end if;
               else
                  -- Start of a new token.
                  exit Start_Loop;
               end if;
            end loop Start_Loop;
                if not No_More then
                   -- Copy character by character until we find a space (unless
                   -- we're quoted!).
               Copy_Loop:
                   loop
                      if I > A'Last then
                         exit Copy_Loop;
                      elsif (not Quoted) and then A(I) = ' ' then
                         I := I + 1;
                         -- Finished.
                         exit Copy_Loop;
                      elsif A(I) = '"' then
                         -- If the next character is a ", then copy just one.
                         -- Otherwise, toggle Quoted.
                         if I + 1 <= A'Last and then A(I + 1) = '"' then
                            -- Literal copy of ".
                            T := T & '"';
                            I := I + 2;
                         else
                            I := I + 1;
                            Quoted := not Quoted;
                         end if;
                      else
                         T := T & A(I);
                         I := I + 1;
                      end if;
                   end loop Copy_Loop;
                       -- Trap silliness.
                       if Quoted then
                          Error("Misc.Split_Arguments.Grab_Next_Token: String `" & A
                            & "' ended inside `""'.");
                       end if;
                       -- Finished.
                       TI := TI + 1;
                       Set(BA, TI, T);
                       -- Recurse.
                       Grab_Next_Token(A(I .. A'Last));
                end if;
         end if;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
              "Exception raised in Misc.Split_Arguments.Grab_Next_Token");
            raise;
      end Grab_Next_Token;

   begin
      Debug("Split_Arguments invoked with `"
        & AS
        & "'");
      Create(BA, 100);
      Grab_Next_Token(AS);
      declare
         RA : UBS_Array(0..TI-1);
      begin
         for I in 1 .. TI loop
            RA(I-1) := Value(BA, I);
         end loop;
         return RA;
      end;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
           "Exception raised in Misc.Split_Arguments");
         raise;
   end Split_Arguments;

   -- Get the basename of a filename.
   function Basename (S : String) return String is
      -- Index of last (if any) `/'.
      I : Integer;
   begin
      I := Ada.Strings.Fixed.Index(Source  => S,
                                   Pattern => "/",
                                   Going   => Ada.Strings.Backward);
      if I = 0 then
         -- Already a basename.
         return S;
      else
         return S(I + 1 .. S'Last);
      end if;
   end Basename;

   -- Basename.
   function Command_Basename return String is
   begin
      return Basename(Ada.Command_Line.Command_Name);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Command_Line_Wrapper.Command_Basename");
         raise;
   end Command_Basename;

end Misc;
