/*****************************************************************************
** FILE IDENTIFICATION
**
**   Name:          wdq2wav.cpp
**   Purpose:       Converts a channel of WinDAQ file to .WAV format
**   Programmer:    Kevin Rosenberg <kevin@rosenberg.net>
**   Date Started:  Jan 2003
**
**  Copyright (c) 2003 Kevin Rosenberg
**
**  $Id$
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License (version 2) as
**  published by the Free Software Foundation.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

#include <stdlib.h>
#include <wdq2wav.h>

const char* g_szIdStr = "$Id$";

bool g_quiet = false;
bool g_verbose = false;
bool g_debug = false;
bool g_ignore_zero = false;
bool g_dont_demean = false;


#ifdef WIN32
#define lseek _lseek
#define close _close
#define open _open
#define read _read
#define write _write
#define O_BINARY _O_BINARY
#define O_RDONLY _O_RDONLY
#define O_WRONLY _O_WRONLY
#define O_RDWR _O_RDRW
#define O_TRUNC _O_TRUNC
#define O_CREAT _O_CREAT
const int g_fileMode = _S_IWRITE | _S_IREAD;
struct fpos_t std::_Fpz = {0,0};
#else
const int g_fileMode = S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH;
// Define as NULL for non-Windows platforms
#ifndef O_BINARY
#define O_BINARY 0
#endif
#endif


void
error_msg (const char *msg)
{
  std::cerr << msg << std::endl;
}

void
info_msg (const char* msg)
{
  std::cout << msg << std::endl;
}

void
info_msg_sans_newline (const char* msg)
{
  std::cout << msg;
}

const char*
fileBasename (const char* filename)
{
  const char* pslash = strrchr (filename, '/');
  const char* pbackslash = strrchr (filename, '\\');
  const char* p = filename;
  if (pbackslash && (! pslash || pbackslash >= pslash))
        p = pbackslash+1;
  else if (pslash && (! pbackslash || pslash >= pbackslash))
    p = pslash+1;

  return p;
}

char *
str_rm_tail (char *str, const char* const charlist)
{
  size_t i;

  for (i = strlen(str) - 1; i >= 0; i--)
    if (strchr (charlist, str[i]) != NULL)
      str[i] = 0;
    else
      break;            /* found non-specified char, all done */

  return (str);
}

char *
str_wrm_tail (char *str)
{
  return (str_rm_tail(str, "\b\t\n\r"));
}


void
usage (const char* progname)
{
  std::cout << "usage: " << fileBasename (progname) << " [OPTIONS] <wdq-file> <channel-number> <wav-file>\n";
  std::cout << "OPTIONS\n";
  std::cout << "  -p   Play channel through audio system\n";
  std::cout << "  -q   Supress all messages\n";
  std::cout << "  -z   Scale output without preserving zero point\n";
  std::cout << "  -m   Do not demean the data (don't subtract the mean value from each sample)\n";
  std::cout << "  -v   Verbose mode\n";
  std::cout << "  -d   Debug mode\n";
  std::cout << "  -r   Print program version and exit\n";
  std::cout << "  -h   Print this help message\n";
}


int
main (int argc, char *argv[])
{
  int c;
  bool play = false;

  const char* progname = argv[0];

  while ((c = getopt (argc, argv, "rqvzmdph")) != -1) {
    switch (c) {
    case 'r':
        std::cout << "Version " << g_szIdStr << std::endl;
                exit(0);
        break;
    case 'q':
      g_quiet = true;
      break;
    case 'm':
      g_dont_demean = true;
      break;
    case 'z':
      g_ignore_zero = true;
      break;
    case 'v':
      g_verbose = true;
      break;
    case 'd':
      g_debug = true;
      break;
    case 'p':
      play = true;
      break;
    case 'h':
      usage (progname);
      return (0);
    case '?':
    default:
        usage (progname);
        return (1);
      }
    }

  if (g_verbose || g_debug)
          std::cout << "Version " << g_szIdStr << std::endl;

  argc -= optind;
  argv += optind;
  if (argc > 3) {
      std::cerr << "Too many parameters\n";
      usage (progname);
      return (1);
    }

  char wdq_fname[MAX_INPUT_STR];
  if (argc >= 1)
      strncpy (wdq_fname, argv [0], MAX_INPUT_STR);
  else {
    std::cout << "Enter input WinDAQ filename: ";
      std::cin.getline (wdq_fname, MAX_INPUT_STR);
    }

    char channel_buf [MAX_INPUT_STR];
    if (argc >= 2)
      strncpy (channel_buf, argv[1], MAX_INPUT_STR);
    else {
      std::cout << "Enter channel number: ";
      std::cin.getline (channel_buf, MAX_INPUT_STR);
    }

    char *channel_endptr;
    int channel = static_cast<int>(strtol (channel_buf, &channel_endptr, 10));
    if (*channel_endptr != 0) {
      std::ostringstream os;
      os << "Error: Channel " << channel_buf << " is not an integer";
      error_msg (os.str().c_str());
      usage (progname);
      return (1);
    }

    char wav_fname[MAX_INPUT_STR];
    if (argc >= 3)
      strncpy (wav_fname, argv[2], MAX_INPUT_STR);
    else {
      std::cout << "Enter output wav filename: ";
      std::cin.getline (wav_fname, MAX_INPUT_STR);
    }

    if (! wdq2wav (wdq_fname, channel, wav_fname, play))
      return 1;

    return 0;
}

bool
wdq2wav (const char* wdq_fname, const int channel, const char *wav_fname, bool play)
{
  WindaqFile wdq (wdq_fname);

  if (! wdq.ReadHeader()) {
    if (wdq.m_error.size()) {
      std::ostringstream os;
      os << "Error reading file " << wdq_fname << ": " << wdq.m_error.c_str();
      error_msg (os.str().c_str());
    } else {
      std::ostringstream os;
      os << "Error reading file " << wdq_fname;
      error_msg (os.str().c_str());
    }
    return false;
  }

  if (wdq.any_packed_channels()) {
          std::ostringstream os;
          os << "File contains 'packed' channels." << std::endl;
          os << "Convert to 'Advanced CODAS headers' before processing with wdq2wav.";
          error_msg (os.str().c_str());
          return false;
  }

  if (! g_quiet || g_verbose || g_debug) {
    std::ostringstream os1;
        os1 << "File: " << wdq_fname;
    info_msg (os1.str().c_str());
        std::ostringstream os;
        os << "Legacy Format: ";
        if (wdq.m_bLegacy_format)
                os << "Yes";
        else
                os << "No";
        info_msg(os.str().c_str());
    std::ostringstream os2;
    time_t time = wdq.m_time_acq_start;
    struct tm* tm = gmtime (&time);
    os2 << "  Time File Creation: " << asctime(tm);
    info_msg_sans_newline (os2.str().c_str());
    std::ostringstream os3;
    time = wdq.m_time_acq_stop;
    tm = gmtime (&time);
    os3 << "  Time File Written: " << asctime(tm);
    info_msg_sans_newline (os3.str().c_str());
    std::ostringstream os4;
    os4 << "  Samples: " << wdq.m_nSamples <<
      ", Channels: " << wdq.m_nChannels <<
      ", Sample Rate: " << wdq.m_sample_rate;
    info_msg (os4.str().c_str());
  }

  WindaqChannel wdq_channel (wdq, channel);
  if (! wdq_channel.m_valid) {
    error_msg ("Error reading data from channel");
    return false;
  }

  if (! g_quiet || g_verbose || g_debug) {
    std::ostringstream os1;
    os1 << "Channel " << channel;
    info_msg (os1.str().c_str());
    std::ostringstream os2;
    os2 << "  Units: " << wdq_channel.m_units.c_str();
    info_msg (os2.str().c_str());
    std::ostringstream os3;
    os3 << "  Raw minimum: " << wdq_channel.m_min_raw_data <<
      ", maximum: " << wdq_channel.m_max_raw_data;
    info_msg (os3.str().c_str());
  }

  if (g_debug) {
    std::ostringstream os4;
    os4 << "  Scaled minimum: " << wdq_channel.m_min_scaled_data <<
      ", maximum: " << wdq_channel.m_max_scaled_data;
    info_msg (os4.str().c_str());
    std::ostringstream os5;
    os5 << "  Slope " <<  wdq_channel.m_slope <<
      ", Intercept " <<  wdq_channel.m_intercept;
    info_msg (os5.str().c_str());
  }

  WavFile wav (wdq_channel, wav_fname);

  if (! wav.m_valid) {
    error_msg ("Error extracting wav from channel");
    return false;
  }

  if (! wav.WriteFile ()) {
    error_msg ("Error writing file");
    return false;
  }

  if (play)
    wav.Play();

  return true;
}


WindaqFile::WindaqFile (const char* fname)
  : m_valid(false), m_fd(0), m_nChannels(0), m_nSamples(0), m_sample_rate(0),
    m_strFile (fname)
{
}

WindaqFile::~WindaqFile ()
{
  if (m_fd != 0)
    close (m_fd);
}

bool read_int1 (int fd, unsigned char& n)
{
  unsigned char tmp1;
  if (read (fd, &tmp1, 1) != 1)
    return false;

  n = tmp1;
  return true;
}

bool read_int2 (int fd, unsigned short int& n)
{
  unsigned char tmp1;
  unsigned int tmp2;
  if (read (fd, &tmp1, 1) != 1)
    return false;
  tmp2 = tmp1;
  if (read (fd, &tmp1, 1) != 1)
    return false;

  n = tmp2 + (tmp1 * 256);
  return true;
}

bool read_int4 (int fd, unsigned int& n)
{
  unsigned int tmp4;
  unsigned short int tmp2;
  if (! read_int2 (fd, tmp2))
    return false;
  tmp4 = tmp2;
  if (! read_int2 (fd, tmp2))
    return false;

  n = tmp4 + (tmp2 * 65536);
  return true;
}

bool read_float8 (int fd, double& f)
{
  unsigned char buf[8];
  if (read (fd, &buf, 8) != 8)
    return false;

#if WORDS_BIG_ENDIAN
  unsigned char c;
  c = buf[0]; buf[0] = buf[7]; buf[7] = c;
  c = buf[1]; buf[1] = buf[6]; buf[6] = c;
  c = buf[2]; buf[2] = buf[5]; buf[5] = c;
  c = buf[3]; buf[3] = buf[4]; buf[4] = c;
#endif

  f = *(reinterpret_cast<double*>(buf));

  return true;
}

bool
WindaqFile::ReadHeader ()
{
  m_valid = false;
  if ((m_fd = open (m_strFile.c_str(), O_RDONLY | O_BINARY)) < 0) {
    m_error = "Unable to open file";
    return false;
  }

  lseek (0, 0, SEEK_SET);
  unsigned short int element1;
  if (! read_int2 (m_fd, element1))
    return false;

  short unsigned int byte1 = (element1 & 0xFF00) >> 8;
  short unsigned int byte2 = element1 & 0xFF;
  if (byte1 == 0 || byte1 == 1) {
          m_bLegacy_format = false;
          m_sr_denom = m_sr_numer = 0;
  } else {
          m_sr_denom = (element1 & 0x7fff) >> 5;
          m_sr_numer = (element1 & 0x8000) << 1;
      m_bLegacy_format = true;
  }
  unsigned short int element2;
  if (! read_int2 (m_fd, element2))
    return false;

  if (m_bLegacy_format)
          m_sr_numer |= element2;

  unsigned char element3;
  if (! read_int1 (m_fd, element3))
    return false;
  m_channel_offset = element3;
  if (g_debug)
          std::cout << "Channel offset: " << m_channel_offset << std::endl;

  unsigned char element4;
  if (! read_int1 (m_fd, element4))
    return false;
  m_nBytes_channel_header = element4;
  if (g_debug)
          std::cout << "Channel header bytes: " << m_nBytes_channel_header << std::endl;

  unsigned short int element5;
  if (! read_int2 (m_fd, element5))
    return false;
  m_nHeader_bytes = element5;
  if (g_debug)
          std::cout << "Header bytes: " << m_nHeader_bytes << std::endl;

  m_nMaxChannels = (m_nHeader_bytes - 112) / 36;
  if (m_nMaxChannels >= 144)
        m_nChannels = byte2 & 0xFF;
  else
    m_nChannels = byte2 & 0x1F;

  unsigned int element6;
  if (! read_int4 (m_fd, element6))
    return false;
  m_nData_bytes = element6;
  if (g_debug)
          std::cout << "Data bytes: " << m_nData_bytes << std::endl;

  m_nSamples = (m_nData_bytes / m_nChannels) / 2;

  lseek (m_fd, 28, SEEK_SET);
  double element13;
  if (! read_float8 (m_fd, element13))
          return false;
  m_time_between_channel_samples = element13;

  if (m_bLegacy_format)
     m_sample_rate = (double) m_sr_numer / (double) (m_sr_denom * m_nChannels);
  else
         m_sample_rate = (double) m_nChannels / m_time_between_channel_samples;

  lseek (m_fd, 36, SEEK_SET);
  if (! read_int4 (m_fd, m_time_acq_start))
    return false;

  if (! read_int4 (m_fd, m_time_acq_stop))
    return false;

  lseek (m_fd, 100, SEEK_SET);
  unsigned short int element27;
  if (! read_int2 (m_fd, element27))
          return false;

  m_bHires = (element27 & 0x0001) ? true : false;
  if (g_debug) {
          std::cout << "High resolution: ";
          if (m_bHires)
                  std::cout << "Yes";
          else
                  std::cout << "No";

          std::cout << std::endl;
  }

  // Verify Windaq signature
  lseek (m_fd, m_nHeader_bytes - 2, SEEK_SET);
  unsigned short int element35;
  if (! read_int2 (m_fd, element35))
    return false;

  if (element35 != 0x8001) {
    std::ostringstream os;
        m_error = "Incorrect signagure: file is not a valid WinDAQ file";
    return false;
  }

  m_valid = true;
  return true;
}

bool
WindaqFile::any_packed_channels ()
{
        for (int iChannel = 0; iChannel < m_nChannels; iChannel++)
                if (is_channel_packed (iChannel))
                        return true;

        return false;
}

bool
WindaqFile::is_channel_packed (const int channel)
{
  long iStart = m_channel_offset + channel * m_nBytes_channel_header;

  lseek (m_fd, iStart + 31, SEEK_SET);
  unsigned char iReadings_per_data_point;
  if (! read_int1 (m_fd, iReadings_per_data_point))
        return false;

  if (iReadings_per_data_point > 1)
          return true;

  return false;
}

WindaqChannel::WindaqChannel (WindaqFile& wdq, const int channel)
  : r_wdq(wdq), m_data(0), m_slope(0), m_intercept (0), m_channel(channel),
    m_valid(false)
{
  if (wdq.m_valid) {
    if (channel >= 1 && channel <= wdq.m_nChannels) {
      if (read_channel_data())
        m_valid = true;
    } else {
      std::ostringstream os;
      os << "Channel " << channel << " is invalid, valid range 1-" <<
        wdq.m_nChannels;
      error_msg (os.str().c_str());
    }
  }
}

WindaqChannel::~WindaqChannel ()
{
  if (m_data)
    delete m_data;
}

bool
WindaqChannel::read_channel_data ()
{
  int fd = r_wdq.m_fd;

  m_data = new signed short int [r_wdq.m_nSamples * 2];

  lseek (fd, r_wdq.m_channel_offset + 8 +
         (m_channel - 1) * r_wdq.m_nBytes_channel_header,
         SEEK_SET);
  if (! read_float8 (fd, m_slope))
    return false;

  if (! read_float8 (fd, m_intercept))
    return false;

  char units[7];
  units[6] = 0;
  if (read (fd, units, 6) != 6) {
    error_msg ("Error reading file");
    return false;
  }
  m_units = units;

  long int row_bytes = 2 * r_wdq.m_nChannels;

  signed short int *sample_row = new signed short int [row_bytes];

  signed short int* psample = &sample_row[m_channel - 1];

  lseek (fd, r_wdq.m_nHeader_bytes, SEEK_SET);
  unsigned long int i;
  signed short int data_max = 0, data_min = 0;
  double total_data = 0;
  for (i = 0; i < r_wdq.m_nSamples; i++) {
    if (read (fd, sample_row, row_bytes) != row_bytes) {
      std::ostringstream os;
      os << "Error reading file at " << i;
      error_msg (os.str().c_str());
      delete sample_row;
      return false;
    }
    signed short int v = *psample;

#if WORDS_BIG_ENDIAN
    unsigned char* p = reinterpret_cast<unsigned char*>(&v);
    unsigned char c = p[0]; p[0] = p[1]; p[1] = c;
#endif

    signed short int value = v;
        if (! r_wdq.m_bHires)
                value >>= 2;

    m_data[i] = value;
    total_data += value;

    if (i == 0) {
      data_max = value;
      data_min = value;
    } else {
      if (value > data_max)
        data_max = value;
      else if (value < data_min)
        data_min = value;
    }
  }

  m_max_raw_data = data_max;
  m_min_raw_data = data_min;
  m_max_scaled_data = (m_slope * data_max) + m_intercept;
  m_min_scaled_data = (m_slope * data_min) + m_intercept;

  if (! g_dont_demean) {
    double dmean = total_data / static_cast<double>(r_wdq.m_nSamples);
    int mean = nearest<int>(dmean);
    std::cout << "Removing mean: " << (dmean * m_slope) + m_intercept <<
      " " << m_units << std::endl;

    for (i = 0; i < r_wdq.m_nSamples; i++)
      m_data[i] -= mean;
  }

  delete sample_row;
  return true;
}


WavFile::WavFile (WindaqChannel& wdq_channel, const char* fname)
  : m_valid(false), m_data(0), m_nSamples(0), m_strFile(fname), m_fd(0)
{
  if (wdq_channel.m_valid) {
    m_nSamples = wdq_channel.r_wdq.m_nSamples;
    m_nChannels = 1;
    m_nBitsPerSample = 16;
    m_nBytesPerSample = 2;
    m_rate = wdq_channel.r_wdq.m_sample_rate;

    double data_offset = 0, data_scale = 0;
    if (g_ignore_zero) {
      data_offset = -wdq_channel.m_min_scaled_data;
      if (wdq_channel.m_max_scaled_data != wdq_channel.m_min_scaled_data)
        data_scale = 65535. / (wdq_channel.m_max_scaled_data -
                               wdq_channel.m_min_scaled_data);
    } else {
      double max_value = fabs(wdq_channel.m_max_scaled_data);
      if (fabs (wdq_channel.m_min_scaled_data) > max_value)
        max_value = fabs (wdq_channel.m_min_scaled_data);
      if (max_value != 0.)
        data_scale = 32767. / max_value;
    }

    if (g_debug) {
      std::ostringstream os;
      os << "  Wav data_scale: " << data_scale << ", data_offset: " << data_offset;
      info_msg (os.str().c_str());
    }

    m_nHeaderBytes = 44;
    m_nDataBytes = m_nSamples * m_nBytesPerSample * m_nChannels;
    m_nFileBytes = m_nHeaderBytes + m_nDataBytes;

    unsigned int nHeaderShortInts = m_nHeaderBytes / sizeof(signed short int);

    m_data = new signed short int [m_nSamples + nHeaderShortInts];
    signed short int* input = wdq_channel.m_data;
    signed short int* output = &m_data[nHeaderShortInts];
    double slope = wdq_channel.m_slope;
    double intercept = wdq_channel.m_intercept;

    if (! fill_header ())
      return;

    unsigned long int i;

    for (i = 0; i < m_nSamples; i++) {
      double value = input[i];
      value = (slope * value) + intercept;
      if (g_ignore_zero) {
        value = (value + data_offset) * data_scale;
        value += 0.5 - 32768;
      } else {
        value = value * data_scale;
      }

      signed short int v = static_cast<signed short int>(value);
#if WORDS_BIG_ENDIAN
      unsigned char* p = reinterpret_cast<unsigned char*>(&v);
      unsigned char c = p[0]; p[0] = p[1]; p[1] = c;
#endif
      output[i] = v;
    }
  }

  m_valid = true;
}


WavFile::~WavFile ()
{
  if (m_fd != 0)
    close (m_fd);

  if (m_data != NULL)
    delete m_data;
}

void
put_int4 (char* p, unsigned int n)
{
  *p = n & 0xFF;
  *(p+1) = 0xFF & (n >> 8);
  *(p+2) = 0xFF & (n >> 16);
  *(p+3) = 0xFF & (n >> 24);
}

void
put_int2 (char* p, unsigned short int n)
{
  *p = n & 0xFF;
  *(p+1) = 0xFF & (n >> 8);
}

bool
WavFile::fill_header ()
{
  char* pData = reinterpret_cast<char*> (m_data);

  strncpy (pData, "RIFF", 4);

  // Length of file after 8 byte header
  put_int4 (pData + 4, 36 + m_nDataBytes);

  strncpy (pData + 8, "WAVEfmt ", 8);

  // Length of header block
  put_int4 (pData + 16, 0x10);

  // Always 1
  put_int2 (pData + 20, 1);

  /* Number of channels */
  put_int2 (pData + 22, m_nChannels);

  // Sample Rate
  put_int4 (pData + 24, static_cast<int> (m_rate + 0.5));

  // Bytes per second
  put_int4 (pData + 28, static_cast<int> (m_rate * m_nBytesPerSample + 0.5));

  // Bytes per sample
  put_int2 (pData + 32, m_nBytesPerSample * m_nChannels);

  // Bits per sample
  put_int2 (pData + 34, m_nBitsPerSample);

  strncpy (pData + 36, "data", 4);

  put_int4 (pData + 40, m_nDataBytes);

  return true;
}

bool
WavFile::WriteFile ()
{
  if (! m_valid)
    return false;

  if (m_fd == 0)
    if ((m_fd = open (m_strFile.c_str(), O_WRONLY | O_BINARY | O_TRUNC | O_CREAT, g_fileMode)) == 0) {
      std::ostringstream os;
      os << "Error opening output file " << m_strFile.c_str();
      error_msg (os.str().c_str());
      return false;
    }

  lseek (m_fd, 0, SEEK_SET);
  if (write (m_fd, m_data, m_nFileBytes) != m_nFileBytes) {
    error_msg ("Error writing file");
    return false;
  }

  if (close (m_fd) < 0)
    error_msg ("Error closing output file");

  m_fd = 0;
  return true;
}

#ifdef WIN32
#include <windows.h>
#include <mmsystem.h>
#elif defined(LINUX)
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#endif

bool
WavFile::Play ()
{
#ifdef WIN32
  if (PlaySound ((LPCSTR) m_data, 0, SND_MEMORY | SND_NODEFAULT))
    return true;
#elif defined(LINUX)
  int fd;
  if ((fd = open ("/dev/dsp",O_WRONLY)) == -1) {
    error_msg ("Error opening /dev/dsp");
    return false;
  }

  int format = AFMT_S16_LE;
  if (ioctl (fd, SNDCTL_DSP_SETFMT, &format) == -1) {
    error_msg ("Error setting DSP format");
    close(fd); return false;
  }
  if (format != AFMT_S16_LE) {
    error_msg ("DSP Format not set");
    close(fd); return false;
  }

  unsigned int channels = m_nChannels;
  if (ioctl (fd, SNDCTL_DSP_CHANNELS, &format) == -1) {
    error_msg ("Error setting number of channels");
    close(fd); return false;
  }
  if (channels != m_nChannels) {
    error_msg ("Number of channels not set");
    close(fd); return false;
  }

  unsigned int speed = static_cast<int>(m_rate + 0.5);
  if (ioctl (fd, SNDCTL_DSP_SPEED, &speed) == -1) {
    error_msg ("Error setting sample rate");
    close(fd); return false;
  }
  if (speed != m_rate && ! g_quiet) {
    std::ostringstream os;
    os << "Warning: Sample rate set to " << speed << ", not " << m_rate;
    error_msg (os.str().c_str());
  }

  if (write (fd, reinterpret_cast<char*>(m_data) + m_nHeaderBytes, m_nDataBytes) !=
      m_nDataBytes) {
    error_msg ("Error writing audio samples");
    close(fd); return false;
  }

  close (fd);
  return true;
#else
#endif

  return false;
}


