/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
*/
package gov.nasa.worldwind.servers.wms;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.Socket;
import java.util.*;
import java.text.SimpleDateFormat;

/**
 *
 * @author brownrigg
 * @version $Id: WMSHttpServletResponse.java 5011 2008-04-10 16:53:54Z rick $
 */
public class WMSHttpServletResponse {
    
    /** Creates a new instance of WMSHttpServletResponse */
    public WMSHttpServletResponse(Socket s) throws IOException {
        theSocket = s;
        theOutputStream = new HttpOutputStream(s.getOutputStream());
        headers = new HashMap<String, String>();
    }
    
    public void setStatus(int status) {
        httpStatus = status;
    }
    
    public void setContentLength(int length) {
        contentLength = length;
    }
    
    public void setContentType(String type) {
        contentType = type;
    }
    
    // Note: overwrites any existing header of this name...
    public void addHeader(String header, String value) {
        headers.put(header, value);
    }
    
    public OutputStream getOutputStream() throws IOException {
        return theOutputStream;
    }
    
    public PrintWriter getWriter() throws IOException {
        return null;
    }
    
    public boolean isCommitted() {
        return committed;
    }
    
    public void flushBuffer() throws IOException {
        getOutputStream().flush();
    }
    
    private void writeHttpResponseHead() throws IOException {
        OutputStreamWriter out = new OutputStreamWriter(theSocket.getOutputStream());
        String statusText = statusMsgs.get(httpStatus);
        if (statusText == null)
            statusText = statusMsgs.get(UNKNOWN_STATUS);
        String status = "HTTP/1.1 " + Integer.toString(httpStatus) + " " + statusText + "\r\n";
        out.write(status);

        // include these "special" headers and write all of them...
        addHeader("Content-Type", contentType);
        // omit contentLength if unknown...
        if (contentLength > 0)
            addHeader("Content-Length", Integer.toString(contentLength));

        Date now = new Date(System.currentTimeMillis());
        String nowStr = getTimeStamp(now);
        addHeader("Date", nowStr);

        addHeader("Server", SERVER_STRING);

        Set<String> keys = headers.keySet();
        for (String key : keys) {
            String val = headers.get(key);
            out.write(key + ": " + val + "\r\n");
        }
        out.write("\r\n");
        out.flush();
        committed = true;
    }

    private String getTimeStamp(Date date) {
       SimpleDateFormat formatter = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
       formatter.setTimeZone(TimeZone.getTimeZone("GMT"));
       return formatter.format(date);
    }
    
    /*
     * This inner class wraps the socket's OutputStream. We override the write()
     * methods so that we can guarantee that the HTTP headers get written before
     * any other output.
     */
    private class HttpOutputStream extends OutputStream {
        public HttpOutputStream(OutputStream o) {
            out = o;
        }
        
        public void write(byte[] b) throws IOException {
            if (!isCommitted())
                writeHttpResponseHead();
            out.write(b);
        }
        
        public void write(byte[] b, int len, int off) throws IOException {
            if (!isCommitted())
                writeHttpResponseHead();
            out.write(b, len, off);
        }
        
        public void write(int b) throws IOException {
            if (!isCommitted())
                writeHttpResponseHead();
            out.write(b);
        }
        
        public void flush() throws IOException {
            if (!isCommitted())
                writeHttpResponseHead();
            out.flush();
        }
        
        private OutputStream out;
    }
    
    private Socket theSocket = null;
    private String contentType = "text/html";
    private int    contentLength = 0;
    private int    httpStatus = 0;
    private boolean committed = false;
    private Map<String, String> headers;
    private HttpOutputStream theOutputStream;
    
    // HTTP status codes...
    static public final int UNKNOWN_STATUS = -1;
    static public final int OK = 200;
    static public final int NO_CONTENT = 204;
    static public final int BAD_REQUEST = 400;
    static public final int NOT_FOUND = 404;
    static public final int SERVER_ERROR = 500;
    static private Map<Integer, String> statusMsgs = new HashMap<Integer, String>();
    {
        statusMsgs.put(OK, "OK");
        statusMsgs.put(NO_CONTENT, "No Content");
        statusMsgs.put(BAD_REQUEST, "Bad Request");
        statusMsgs.put(NOT_FOUND, "Not Found");
        statusMsgs.put(SERVER_ERROR, "Internal Server Error");
        statusMsgs.put(UNKNOWN_STATUS,  "WMSResponse");   // for unknown status codes...
    }

    static private final String SERVER_STRING = "NASA WorldWind WMS";
    // hide this...
    private WMSHttpServletResponse() {}
}
