/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap;

import xnap.cmdl.CommandLine;
import xnap.gui.SplashWindow;
import xnap.gui.XNapFrame;
import xnap.net.*;
import xnap.util.*;

import gnu.getopt.Getopt;
import java.lang.reflect.*;
import java.beans.*;
import java.net.*;
import java.io.*;
import java.util.*;
import org.apache.log4j.Level;

/**
 * Starts the main application.
 */
public class XNap {

    //--- Constant(s) ---

    /**
     * Bind-zone-file-style number to track current release date. This has to 
     * be increased on every release.
     *
     * Format: YYYYMMDDNN
     */
    public static final long RELEASE_NR = 2003011801;

    /**
     * Current version.
     *
     * format: '$MAJOR.$MINOR[-$EXTRA(1-9)]'
     * $EXTRA can be 'alpha', 'beta' or 'pre'
     *
     * examples: 1.3-alpha1, 2.1-pre1, 2.4
     *
     * Security fix related releases should append 'r(1-9)' to the version,
     * like this: 2.4r1
     */
    public static final String VERSION = "2.4-pre5";

    /**
     * The minimal JDK version required to run the commandline version
     */
    public static final String REQUIRED_CMDL_JDK = "1.2.0";

    /**
     * The minimal JDK version required to run the gui version
     */
    public static final String REQUIRED_GUI_JDK = "1.3.0";

    /**
     * This list contains all languages the XNap interface has been
     * translated into. Add new languages here (see TRANSLATION file).
     */
    public static final Locale[] LANGUAGES = {
	Locale.CHINESE, Locale.ENGLISH, Locale.FRENCH, Locale.GERMAN, 
	Locale.ITALIAN, Locale.JAPANESE, new Locale("nl", "", ""), 
	new Locale("es", "", ""),
	//new Locale("ru", "", ""),
    };

    //--- Data field(s) ---

    private static ResourceBundle resources;

    //--- Method(s) ---

    /**
     * Brings up the Command Line (Cmdl) or Graphic User Interface (GUI).
     *
     * @param args foo
     */
    public static void main(String[] argv)
    {
	Level level = Level.OFF;
	boolean debug = false;
	boolean startCmdl = false;
	boolean showHelp  = false;
	boolean jdkCheck = true;

	Getopt g = new Getopt("xnap", argv, "cd::hnv");

	int c;
	while ((c = g.getopt()) != -1) {
	    //System.out.println((char)c + " [" + g.getOptarg() + "]");
	    switch (c) {
	    case 'c':
		startCmdl = true;
		break;
	    case 'd':
		debug = true;
		String arg = g.getOptarg();
		level = (arg != null) ? Level.toLevel(arg) : Level.ALL;
		break;
	    case 'h':
		showHelp = true;
		break;
	    case 'n':
		jdkCheck = false;
		break;
	    case 'v':
		System.out.println(XNap.VERSION);
		System.exit(0);
		break;
	    case '?':
		System.exit(1);
		break;
	    }
	}

	File debugPrefs = new File(FileHelper.getHomeDir() + "debug.prefs");
	Debug.init(debugPrefs, level);

	if (!debug) {
	    File f = new File(FileHelper.getHomeDir() + "error.log");
	    Debug.setErrorFile(f);
	}

	// here we go...
	Preferences prefs = Preferences.getInstance();
	prefs.read();

	initLocale(prefs.getLanguage());

// 	try {
// 	    System.err.println("enc: " + (new InputStreamReader(new FileInputStream("/etc/hosts"))).getEncoding());
// 	}
// 	catch (Exception e) {
// 	}

	if (showHelp) {
	    showHelp();
	    System.exit(0);
	}
	else if (startCmdl) {
	    if (jdkCheck)
		checkJDK(REQUIRED_CMDL_JDK);

	    init();

	    CommandLine cmdl = new CommandLine(argv);
	    cmdl.run();
	}
	else {
	    if (jdkCheck) {
		checkJDK(REQUIRED_GUI_JDK);
	    }

	    init();

	    if (prefs.getShowSplash()) {
		SplashWindow.showSplashWindow();
	    }
	    
	    XNapFrame f = new XNapFrame();
	    f.setVisible(true);

	    if (prefs.getShowSplash()) {
		SplashWindow.closeSplashWindow();
	    }
	}
    }

    private static void init()
    {
	JarClassLoader.getInstance().init();
    }

    /**
     * Sets default locale and loads resources.
     */
    public static void initLocale(String language)
    {
	for (int i = 0; i < LANGUAGES.length; i++) {
	    if (language.equals(LANGUAGES[i].getLanguage())) {
		Locale.setDefault(LANGUAGES[i]);
		break;
	    }
	}

        try {
	    resources = ResourceBundle.getBundle("xnap.resources.XNap",
						 Locale.getDefault());
        }
	catch (MissingResourceException e) {
            System.err.println("resources/XNap.properties not found");
            System.exit(1);
	}
    }

    public static final ResourceBundle getMessagesBundle()
    {
	return resources;
    }

    
    public static final void checkJDK(String reqVer)
    {
	String jdkVer = System.getProperty("java.version");

	if (VersionParser.compare(jdkVer, reqVer) < 0) {
 	    System.out.println(XNap.tr("Your java version") + "\t: " + jdkVer);
	    System.out.println(XNap.tr("Required version") + "\t: " + reqVer);
	    System.out.println(XNap.tr("If XNap runs fine, you can safely ignore this message.\n") 
			       + XNap.tr("Otherwise you probably need to update your jdk from\nhttp://java.sun.com/products/jdk/1.3/jre/index.html."));
	    System.out.println(tr("See http://xnap.sourceforge.net for more information.\n") 
			   + "(" + XNap.tr("use -n to suppress this message") + ")\n");
	}
    }
    
    public static final void showHelp()
    {
	System.out.println(XNap.tr("Usage: xnap [options]"));
	System.out.println(" -c\t " + XNap.tr("start in commandline mode"));
	System.out.println(" -d[level]\t " + XNap.tr("show debug messages"));
	System.out.println(" -h\t " + XNap.tr("show this help"));
	System.out.println(" -n\t" + XNap.tr("do not check jdk version"));
    }

    /*
     * DO NOT REMOVE THIS COMMENT
     *
     * special hack to allow for translation of strings that are not 
     * present in XNap code
     *
     * XNap.tr("Exit");
     * XNap.tr("Hide XNap");
     * XNap.tr("Show XNap");
     */

    /**
     * 
     */

    /**
     * Returns <code>text</code> translated into the currently selected
     * language. Every user-visible string in the program must be wrapped
     * into this function.  
     */
    public static final String tr(String text)
    {
	try {
	    return getMessagesBundle().getString(text);
	}
	catch (MissingResourceException e) {
	    System.err.println("missing translation key: \"" + text + "\"");
	    //e.printStackTrace(System.err);
	    return text;
	}
    }

    /**
     * Returns <code>text</code> translated into the currently selected
     * language, and appends as many whitespaces as denoted in padding.
     */
    public static final String tr(String text, int padding)
    {
	String s = tr(text);
	if (padding <= 0) {
	    return s;
	}
	StringBuffer sb = new StringBuffer(s.length() + padding * 2);
	append(sb, " ", padding);
	sb.append(s);
	append(sb, " ", padding);
	return sb.toString();
    }

    /**
     * Returns <code>text</code> translated into the currently selected
     * language, and prepends/appends as many whitespaces as denoted in
     * lpadding/rpadding.  
     */
    public static final String tr(String text, int lpadding, int rpadding)
    {
	String s = tr(text);
	StringBuffer sb = new StringBuffer(s.length() + lpadding + rpadding);
	append(sb, " ", lpadding);
	sb.append(s);
	append(sb, " ", rpadding);
	return sb.toString();
    }

    public static final void append(StringBuffer sb, String s, int count)
    {
	for (int i = 0; i < count; i++) {
	    sb.append(s);
	}
    }

}
