/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package xnap.gui;

import xnap.*;
import xnap.gui.tree.*;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

public class DirectoryChooser extends JDialog implements TreeSelectionListener
{

    //--- Constant(s) ---

    public static final int APPROVE_OPTION = 1;
    public static final int CANCEL_OPTION = 2;

    //--- Data field(s) ---

    private OkayAction acOkay = new OkayAction();
    private JTree jtDirectories;
    private FileTreeModel dtmDirectories;
    private File selectedDirectory = null;

    //--- Constructor(s) ---

    public DirectoryChooser()
    {
	// center
	dtmDirectories = new FileTreeModel(XNap.tr("Directories"),
					   File.listRoots());
	jtDirectories = new JTree(dtmDirectories);
	jtDirectories.setRootVisible(false);
	jtDirectories.setCellRenderer(new FileCellRenderer());
	jtDirectories.putClientProperty("JTree.lineStyle", "Angled");
	jtDirectories.addTreeSelectionListener(this);

        JScrollPane jspDirectories = new JScrollPane(jtDirectories);

	// bottom
	JPanel jpButtons = new JPanel(new FlowLayout(FlowLayout.RIGHT));

	JButton jbHome = new JButton(new HomeAction());
	jpButtons.add(jbHome);

	JButton jbOkay = new JButton(acOkay);
	jpButtons.add(jbOkay);

	JButton jbCancel = new JButton(new CancelAction());
	jpButtons.add(jbCancel);

	// set me up
	getContentPane().setLayout(new BorderLayout());
	getContentPane().add(jspDirectories, "Center");
	getContentPane().add(jpButtons, "South");

	pack();
    }

    //--- Method(s) ---

    public File getSelectedDirectory()
    {
	return selectedDirectory;
    }

    public void setSelectedDirectory(File dir)
    {
	LinkedList files = new LinkedList();
	File parent = dir;
	while (parent != null) {
	    files.addFirst(new FileNode(parent));
	    parent = parent.getParentFile();
	}
  	Object[] path = new Object[files.size() + 1];
  	path[0] = dtmDirectories.getRoot();
  	System.arraycopy(files.toArray(), 0, path, 1, path.length - 1);
	TreePath tp = new TreePath(path);
	jtDirectories.setSelectionPath(tp);
	jtDirectories.scrollPathToVisible(tp);
    }

    public int showChooseDialog(Component c)
    {
	if (c != null) {
	    setLocationRelativeTo(c);
	}

	setTitle(XNap.tr("Choose directory"));
	setModal(true);
	pack();
	show();

	return (selectedDirectory == null ? CANCEL_OPTION : APPROVE_OPTION);
    }

    public void valueChanged(TreeSelectionEvent e)
    {
	acOkay.setEnabled(jtDirectories.getLastSelectedPathComponent()
			  instanceof File);
    }

    public class HomeAction extends AbstractAction {
	
	public HomeAction()
	{
	    putValue(Action.NAME, XNap.tr("home"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("gohome.png"));
            putValue(Action.SHORT_DESCRIPTION,
		     XNap.tr("Jump to home directory"));
            putValue(Action.MNEMONIC_KEY, new Integer('H'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    setSelectedDirectory(new File(System.getProperty("user.home")));
        }

    }

    public class OkayAction extends AbstractAction {
	
	public OkayAction()
	{
	    putValue(Action.NAME, XNap.tr("OK"));
            putValue(Action.SHORT_DESCRIPTION,
		     XNap.tr("Close dialog selecting directory"));
            putValue(Action.MNEMONIC_KEY, new Integer('O'));
	    
	    this.setEnabled(false);
        }

        public void actionPerformed(ActionEvent event) 
	{
	    Object f = jtDirectories.getLastSelectedPathComponent();
		
	    if (f instanceof File) {
		selectedDirectory = (File)f;
		dispose();
	    }
        }

    }

    public class CancelAction extends AbstractAction {
	
	public CancelAction()
	{
	    putValue(Action.NAME, XNap.tr("Cancel"));
            putValue(Action.SHORT_DESCRIPTION,
		     XNap.tr("Close dialog discarding changes"));
            putValue(Action.MNEMONIC_KEY, new Integer('C'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    selectedDirectory = null;
	    dispose();
        }

    }

}
