/*
 *  Java Napster version x.yz (for current version number as well as for
 *  additional information see version.txt)
 *
 *  Previous versions of this program were written by Florian Student
 *  and Michael Ransburg available at www.weblicity.de/jnapster and
 *  http://www.tux.org/~daneel/content/projects/10.shtml respectively.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.net;

import xnap.util.*;
import xnap.net.*;
import xnap.plugin.nap.net.msg.ExceptionListener;
import xnap.plugin.nap.net.msg.MessageHandler;
import xnap.plugin.nap.net.msg.client.SearchRequestMessage;
import xnap.plugin.nap.net.msg.server.SearchResponseMessage;
import xnap.plugin.nap.util.NapPreferences;

import java.io.IOException;
import java.util.*;

/**
 * The methods that are only called by the <code>Server</code> class don't
 * have access modifiers.
 */
public class Search extends AbstractSearch implements ExceptionListener {

    //--- Data Field(s) ---
    
    /**
     * Stores all results in list for search cache.
     */
    protected LinkedList results = new LinkedList();
    protected Server server;
    protected String failedMsg = null;
    protected SearchRequestMessage requestMsg;

    protected LinkedList peers = new LinkedList();
    protected Object peerLock = new Object();

    //--- Constructor(s) ---

    public Search(SearchFilter filter, int priority, int maxResults, 
		  Server server) 
    {
	super(filter, priority, maxResults);

	this.server = server;
	this.requestMsg = new SearchRequestMessage(filter, maxResults);
    }

    //--- Method(s) ---

    void add(SearchResponseMessage msg)
    {
	User user = server.getUser(msg.nick);
	SearchResult result 
	    = new SearchResult(msg.filesize, msg.bitrate, msg.frequency, 
			       msg.length, user, msg.filename, msg.md5);

	results.add(result);
	add(result);

	synchronized (peerLock) {
	    for (Iterator i = peers.iterator(); i.hasNext();) {
		((Search)i.next()).add(msg);
	    }
	}
    }

    void addPeer(Search s)
    {
	synchronized (peerLock) {
	    peers.add(s);
	}
    }

    void failed(String msg)
    {
	failedMsg = msg;

	synchronized(peerLock) {
	    for (Iterator i = peers.iterator(); i.hasNext();) {
		((Search)i.next()).failed(msg);
	    }
	}
    }

    void finished()
    {
	finished = true;

	synchronized (peerLock) {
	    for (Iterator i = peers.iterator(); i.hasNext();) {
		((Search)i.next()).finished();
	    }
	}
    }

    public Server getServer()
    {
	return server;
    }

    void start()
    {
	requestMsg.setExceptionListener(this);
	MessageHandler.send(server, requestMsg);
    }

    public synchronized int available() throws IOException
    {
	if (failedMsg != null) {
	    throw new IOException(failedMsg);
	}
	
	return super.available();
    }

    public void close()
    {
	if (!finished) {
	    Debug.log("nap search: aborted");
	    server.removeSearch(this);
	}
	finished = true;
    }
    
    public void connect() throws IOException
    {
	server.addSearch(this);
    }

    public void exceptionThrown(Exception e)
    {
	failed(e.getMessage());
    }

    public LinkedList getResults()
    {
	return results;
    }

    public String getRequest()
    {
	return requestMsg.getData(server.getVersion());
    }

    public boolean equals(Object obj)
    {
	if (obj instanceof Search) {
	    Search s = (Search)obj;
	    return (getRequest().equals(s.getRequest()) 
		    && getServer().equals(s.getServer()));
	}
	
	return false;
    }

}

