/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.util;

import xnap.net.AbstractChannel;
import xnap.net.IUser;
import xnap.plugin.nap.net.Server;
import xnap.plugin.nap.net.User;
import xnap.plugin.nap.net.msg.ExceptionListener;
import xnap.plugin.nap.net.msg.MessageHandler;
import xnap.plugin.nap.net.msg.client.PrivateMessage;
import xnap.plugin.nap.net.msg.server.GlobalMessage;
import xnap.plugin.nap.net.msg.server.MessageListener;
import xnap.plugin.nap.net.msg.server.ServerMessage;

import java.io.IOException;

/**
 * Manages a list of channels.
 */
public class PrivateChannel extends AbstractChannel 
    implements MessageListener, ExceptionListener
{

    //--- Constant(s) ---

    /**
     * Time to wait for global error message.
     */
    public static final int ERROR_WAIT = 3 * 1000;

    //--- Data field(s) ---

    protected long lastSent;
    protected User user;
    protected Server server;

    //--- Constructor(s) ---

    public PrivateChannel(User user)
    {
	super(user.getServer(), user.toString(), "", 1);

	this.user = user;
	this.server = user.getServer();
    }

    //--- Method(s) ---

    public boolean canChangeTopic()
    {
	return false;
    }

    public void changeTopic(String newTopic)
    {
    }

    public void close()
    {
	MessageHandler.unsubscribe(GlobalMessage.TYPE, this);
	MessageHandler.removePrivateChannel(this);
    }

    public void exceptionThrown(Exception e) 
    {
	errorReceived(e.getMessage());
    }

    public User getUser()
    {
	return user;
    }

    public IUser[] getUsers()
    {
	return new IUser[] { user };
    }

    public void join()
    {
	MessageHandler.subscribe(GlobalMessage.TYPE, this);
	infoReceived("Talking to " + user.toString());
    }

    public void messageReceived(String message)
    {
	messageReceived(user, message);
    }

    public void sendMessage(String message) throws IOException
    {
	PrivateMessage msg = new PrivateMessage(user.getName(), message);
	msg.setExceptionListener(this);
	MessageHandler.send(server, msg);
	lastSent = System.currentTimeMillis();

	// self copy
	messageReceived(server.getUser(), message);
    }
    

    public void messageReceived(ServerMessage msg) 
    {
	if (msg.getServer() != server) {
	    return;
	}

	if (msg instanceof GlobalMessage 
	    && System.currentTimeMillis() - lastSent < ERROR_WAIT) {	    
	    StringBuffer sb = new StringBuffer();
	    sb.append(server.getHost());
	    sb.append(": ");
	    sb.append(((GlobalMessage)msg).message);

	    errorReceived(sb.toString());
	}
    }

}

