/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.util;

import xnap.XNap;
import xnap.net.*;

import java.io.*;
import java.util.*;

/**
 * Complete conversion of SearchFilter.Data to SearchFilterData in 2.5.
 */
public class SearchFilter implements Cloneable, Serializable {

    //--- Constant(s) ---
    
    // values correspond to the indices of the compare vector
    public static final int COMPARE_NOT_ACTIVE = 0;
    public static final int COMPARE_AT_LEAST   = 1;
    public static final int COMPARE_AT_BEST    = 2;
    public static final int COMPARE_EQUAL_TO   = 3;

    public static final Object[] COMPARES = {
	"", XNap.tr("at least"), XNap.tr("at best"), XNap.tr("equal to")

    };

    // if the format is changed, getBitrateFromIndex() needs to be adjusted
    public static final Object[] BITRATES = { 
	"", "320 kbps", "256 kbps", "192 kbps", "160 kbps", "128 kbps", 
	"112 kbps", "96 kbps", "80 kbps", "64 kbps", "56 kbps", 
	"48 kbps", "32 kbps", "24 kbps", "20 kbps" 
    };

    public static final Object[] LINK_SPEEDS = { 
	"unknown", "14.4 kbps", "28.8 kbps", "33.6 kbps", "56.7 kbps", 
	"64K ISDN", "128K ISDN", "Cable", "DSL", "T1", "T3 or greater"
    };

    // values correspond to the indices of the media array
    public static final int MEDIA_ANYTHING  = 0;
    public static final int MEDIA_AUDIO     = 1;
    public static final int MEDIA_VIDEO     = 2;
    public static final int MEDIA_IMAGES    = 3;
    public static final int MEDIA_SOFTWARE  = 4;
    public static final int MEDIA_DOCUMENTS = 5;
    	
    public static final Object[] media = {
	XNap.tr("Anything"), XNap.tr("Audio"), XNap.tr("Video"),
	XNap.tr("Images"), XNap.tr("Software"), XNap.tr("Documents")
    };

    /* these should be made configurable in the future, they are ordered
       according to frequenzy of occurrence. */
    public static final String [] audio = {
        "mp3", "ogg", "wav", "mid", "midi", "ram", "ra", "wma", "sd2f", "sd2",
	"wax", "au", "aif", "aiff", "mpa", "rmi", "aifc", "snd"
    };

    public static final String [] video = {
        "avi", "mpg", "mpeg", "asf", "qt", "mov", "3iv", "div", "xiv", "rm",
	"mpe", "swf", "wmv", "mp2v", "mlv", "mpv", "wm", "mp4", "asx"
    };

    public static final String [] images = {
        "jpg", "jpeg", "png", "gif", "jpe", "tif", "tiff", "bmp", "jp2",
	"ps", "psp", "eps", "xpm",  "tga", "xcf"
    };

    public static final String [] software = {
	"exe", "zip", "gz", "hqx", "tar", "tgz", "z", "rmj", "lqt", "iso",
	"rpm", "cue", "iso", "image", "bin", "dmg", "com", "deb", "pkg", "app",
	"vob", "ifo", "jar", "rom", "smc", "n64", "v64", "gba", "gb", "mod",
	"7z", "fcd", "ccd", "rar", "ace", "sit", "sitx", "smi", "img", "cdr", 
	"gzip"
    };

    public static final String [] documents = {
        "html", "htm", "txt", "pdf", "ps", "rtf", "doc", "xml",
        "tex", "dcr", "lit", "dvi", "zip", "log", "bat", "sh"
    };

    public static final String[][] mediaTypes = {
	audio, video, images, software, documents
    };

    //--- Data field(s) ---

    private Data data;
    private String[] path = null;

    //--- Constructor(s) ---

    public SearchFilter(String searchText, int bitrateCompare, int bitrate,
			int filesizeCompare, long filesize, int mediaType)
    {
	this.data = new Data
	    (searchText, bitrateCompare, bitrate, filesizeCompare, filesize,
	     mediaType);
    }

    public SearchFilter(String searchText, int bitrateCompare, int bitrate,
			int mediaType)
    {
	this(searchText, bitrateCompare, bitrate, COMPARE_NOT_ACTIVE, 0, 
	     mediaType);
    }

    public SearchFilter(String searchText)
    {
	this(searchText, COMPARE_NOT_ACTIVE, 0, MEDIA_ANYTHING);
    }

    public SearchFilter()
    {
	this("");
    }

    public SearchFilter(Data data)
    {
	this.data = data;
    }

    //--- Method(s) ---

    public int getBitrate()
    {
	return data.bitrate;
    }

    public void setBitrate(int newValue)
    {
	data.bitrate = newValue;
    }

    public int getBitrateCompare()
    {
	return data.bitrateCompare;
    }

    public void setBitrateCompare(int newValue)
    {
	data.bitrateCompare = newValue;
    }

    public Data getData()
    {
	return data;
    }

    public long getFilesize()
    {
	return data.filesize;
    }

    public void setFilesize(long newValue)
    {
	data.filesize = newValue;
    }

    public int getFilesizeCompare()
    {
	return data.filesizeCompare;
    }

    public void setFilesizeCompare(int newValue)
    {
	data.filesizeCompare = newValue;
    }

    public int getMediaType()
    {
	return data.mediaType;
    }

    public void setMediaType(int newValue)
    {
	data.mediaType = newValue;
    }

    public String[] getPath()
    {
	return path;
    }

    public void setPath(String[] newValue)
    {
	path = newValue;
    }

    public String getSearchText()
    {
	return data.searchText;
    }

    public void setSearchText(String newValue)
    {
	data.searchText = newValue;
    }

    public Object clone()
    {
	SearchFilter s = new SearchFilter
	    (getSearchText(), getBitrateCompare(), getBitrate(), 
	     getFilesizeCompare(), getFilesize(), getMediaType());

	return s;
    }

    public boolean matches(ISearchResult sr)
    {
	if (getSearchText().length() > 0) {
	    StringTokenizer t = new StringTokenizer(getSearchText(), " ");
	    String filename = sr.getFilename().toUpperCase();
	    
	    while (t.hasMoreTokens()) {
		String word = t.nextToken();
		// word should not be matched
		if (word.startsWith("-") && word.length() > 1) {
		    word = word.substring(1);
		    if (!(filename.indexOf(word.toUpperCase()) == -1)) {
			return false;
		    }
		}
		// word should be matched
		else {
		    if (filename.indexOf(word.toUpperCase()) == -1) {
			return false;
		    }
		}
	    }
	}

	if (! matches(getBitrateCompare(), sr.getBitrate(), getBitrate())) {
	    return false;
	}

	if (! matches(getFilesizeCompare(), sr.getFilesize(), getFilesize())) {
	    return false;
	}
	 
	String extension = FileHelper.extension(sr.getShortFilename());
	if (! matches(extension, getMediaType())) {
	    return false;
	}

	if (path != null && !Arrays.equals(path, sr.getPath())) {
	    return false;
	}
	
	return true;
    }

    public String toString() 
    {
	return getSearchText();
    }

    public static int getBitrateFromIndex(int index) 
    {
	if (index > 0) {
	    StringTokenizer t 
		= new StringTokenizer((String)BITRATES[index], " ");
	    return Integer.parseInt(t.nextToken());
	}
	else {
	    return 0;
	}
    }

    // see SearchFilterHelper
//      /**
//       * @return index of bitrate in array <code>SearchFilter.BITRATES</code>.
//       */
//      public static int getIndexFromBitrate(int bitrate) 
//      {
//  	for (int i = 0; i < SearchFilter.BITRATES.length; i++) {
//  	    if (bitrate == SearchFilter.getBitrateFromIndex(i)) {
//  		return i;
//  	    }
//  	}
//  	return 0;
//      }

    public static boolean matches(int mode, long v1, long v2)
    {
	//Debug.log("SearchFilter.matches: " + v1 + " " + compares[mode] 
	//          + " " + v2);
	
	switch (mode) {
	case COMPARE_AT_LEAST:
	    return v1 >= v2;
	case COMPARE_AT_BEST:
	    return v1 <= v2;
	case COMPARE_EQUAL_TO:
	    return v1 == v2;
	default: //COMPARE_NOT_ACTIVE
	    return true;
	}
    }

    /**
     * Compares extension to all elements in array pointed to by given index
     * and returns true if it finds one match, otherwise false.
     * @param index index to mediaTypes array.
     * @param ext the searchresult's file extension.
     * @exception ArrayIndexOutOfBoundsException
     */
    public boolean matches(String ext, int mediaType)
    {
	if (mediaType == MEDIA_ANYTHING) {
	    return true;
	}

	if (mediaType < 0 || mediaType > mediaTypes.length) {
	    return false;
	}

	// shifted by 1 because MEDIA_ANYTHING has no valid extensions
	return occursIn(mediaTypes[mediaType - 1], ext);
    }

    /**
     * Compares extension to all elements in array and returns true if it
     * finds one match, otherwise false.
     * @param array array of possible file extensions.
     * @param ext the searchresult's file extension.
     */
    public boolean occursIn(String[] array, String ext)
    {
	for (int i = 0; i < array.length; i++) {
	    //Debug.log(array[i] + "?" + ext);
	    if (array[i].equalsIgnoreCase(ext)) {
		return true;
	    }
	}
	return false;
    }

    /**
     * Should be removed in 2.5 release.
     *
     * @see xnap.util.SearchFilterData
     */
    public class Data implements Cloneable, Serializable {

	//--- Data field(s) ---
	
	protected int bitrateCompare;
	protected int bitrate;
	protected int filesizeCompare;
	protected long filesize;
	protected int mediaType;
	protected String searchText;

	//--- Constructor(s) ---
	
	public Data(String searchText, int bitrateCompare, int bitrate,
		    int filesizeCompare, long filesize, int mediaType)
	{
	    this.searchText = searchText;
	    this.bitrateCompare = bitrateCompare;
	    this.bitrate = bitrate;
	    this.filesizeCompare = filesizeCompare;
	    this.filesize = filesize;
	    this.mediaType = mediaType;
	}
	
    }

}
