/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.table;

import xnap.*;
import xnap.gui.util.SwingSynchronizedCache;
import xnap.net.*;
import xnap.plugin.INetworkPlugin;
import xnap.util.*;
import xnap.util.event.*;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

public class SearchTableModel extends AbstractDynamicTableModel
    implements Observer, ListListener {

    //--- Constant(s) ---

    public static final int FILENAME = 0;
    public static final int PATH = 1;
    public static final int FILESIZE = 2;
    public static final int BITRATE = 3;
    public static final int FREQUENCY = 4;
    public static final int LENGTH = 5;
    public static final int USER = 6;
    public static final int LINKSPEED = 7;
    public static final int PLUGIN = 8;
    public static final int STATUS = 9;
    public static final int SCORE = 10;
    public static final int TYPE = 11;
    
    protected Column columns[] = new Column[] {
	new Column(XNap.tr("Filename"), String.class, 
		   new StringCellRenderer()),
	new Column(XNap.tr("Path"), String.class, 
		   new StringCellRenderer()),
	new Column(XNap.tr("Filesize"), Long.class, 
		   new FilesizeCellRenderer()),
	new Column(XNap.tr("Bitrate"), Integer.class, 
		   new NumberCellRenderer()),
	new Column(XNap.tr("Frequency"), Double.class, 
		   new NumberCellRenderer(1)),
	new Column(XNap.tr("Length"), Integer.class, 
		   new TimeCellRenderer()),
	new Column(XNap.tr("User"), String.class),
	new Column(XNap.tr("Link Speed"), Integer.class, 
		   new LinkSpeedCellRenderer()),
	new Column(XNap.tr("Plugin"), String.class),
	new Column(XNap.tr("Status"), String.class, 
		   new StringCellRenderer()),
	new Column(XNap.tr("Availability"), Integer.class, 
		   new ScoreCellRenderer()),
	new Column(XNap.tr("Type"), String.class, 
		   new StringCellRenderer()),
    };

    //--- Data field(s) ---

    private ArrayList rows = new ArrayList();
    private SwingSynchronizedCache cache = new SwingSynchronizedCache(this);

    //--- Constructor(s) ---

    public SearchTableModel() 
    {
	super("search", Preferences.getInstance());
	
	setColumns(columns);
    }

    protected SearchTableModel(String table, Preferences prefs)
    {
	super(table, prefs);
    }

    //--- Method(s) ---

    public SearchResultContainer get(int i) 
    {
        return (SearchResultContainer)rows.get(mapToIndex(i));
    }

    public String getTableName()
    {
	return XNap.tr("Search Table");
    }

    /**
     * Should only be called once.
     */
    public void setFilteredData(EventVector data)
    {
	data.addListListener(cache);
    }

    public void elementAdded(ListEvent e)
    {
	if (e.getElement() instanceof Observable) {
	    Observable o = (Observable)e.getElement();
	    o.addObserver(this);
	}

	rows.add(e.getElement());
	fireTableRowsInserted(rows.size() - 1, rows.size() - 1);
    }

    public void elementRemoved(ListEvent e)
    {
	if (e.getID() == ListEvent.LIST_CLEARED) {
	    int size = rows.size();
	    if (size > 0) {
		for (Iterator i = rows.iterator(); i.hasNext();) {
		    Object item = i.next();
		    if (item instanceof Observable) {
			Observable o = (Observable)item;
			o.deleteObserver(this);
		    }
		}

		rows.clear();
		fireTableRowsDeleted(0, size - 1);
	    }
	}
	else {
	    int i = rows.indexOf(e.getElement());
	    if (i != -1) {
		if (e.getElement() instanceof Observable) {
		    Observable o = (Observable)e.getElement();
		    o.deleteObserver(this);
		}

		rows.remove(i);
		fireTableRowsDeleted(i, i);
	    }
	}
    }

    public int getRowCount() 
    {
        return rows.size();
    }
    
    public Object get(int i, int j) 
    {
        if (i >= rows.size()) {
	    return null;
	}

        SearchResultContainer sr = (SearchResultContainer)rows.get(i);

        if (sr == null) 
	    return null;

        switch (j) {
	case FILENAME:
	    return  sr.getShortFilename();
	case FILESIZE:
	    return new Long(sr.getFilesize());
	case BITRATE:
	    return new Integer(sr.getBitrate());
	case FREQUENCY:
	    return new Double((double)sr.getFrequency() / 1000);
	case LENGTH:
	    return new Integer(sr.getLength());
	case USER:
	    return sr.getUsername();
	case LINKSPEED:
	    return new Integer(sr.getLinkSpeed());
	case SCORE:
	    return new Integer(sr.getScore());
	case PATH:
	    return sr.getPath();
	case STATUS:
	    return sr.getStatus();
	case PLUGIN:
	    return sr.getPlugin();
	case TYPE:
	    return FileHelper.extension(sr.getShortFilename());
	default:
	    return "";
        }
    }

    /**
     * Observes the grouped results.
     */
    public void update(Observable o, Object arg) 
    {
        if (arg instanceof SearchResultContainer) {
	    SwingUtilities.invokeLater(new StatusChangeEventHandler(arg));
	}
    }

    private class StatusChangeEventHandler implements Runnable
    {

	Object item;

	public StatusChangeEventHandler(Object item)
	{
	    this.item = item;
	}

	public void run()
	{
	    int i = rows.indexOf(item);
	    if (i != -1) {
		fireTableRowsUpdated(i, i);
	    }
	}

    }
}
