/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.table;

import xnap.util.*;

import java.util.*;
import java.awt.*;
import javax.swing.*;
import javax.swing.table.*;

/**
 * Don't use this class directly. Use 
 * <code>TableHeaderListener.install()</code> instead.
 *
 * @see xnap.gui.table.TableHeaderListener
 */
public class SortButtonRenderer extends Object implements TableCellRenderer
{

    public static final Icon downIcon 
	= new BevelArrowIcon(BevelArrowIcon.DOWN, false);
    public static final Icon upIcon 
	= new BevelArrowIcon(BevelArrowIcon.UP, false);

    //--- Data Field(s) ---

    private Hashtable icons = new Hashtable();
    private TableCellRenderer cr;
    private JLabel l;
    private int selectedColumn = -1;

    //--- Constructor(s) ---
    
    public SortButtonRenderer(TableCellRenderer cr)
    {
	this.cr = cr;
	if (cr instanceof JLabel) {
	    l = (JLabel)cr;
	    l.setHorizontalTextPosition(SwingUtilities.LEFT);
	}
	else {
	    // avoid null pointer exception if cr is not instanceof JLabel
	    l = new JLabel();
	}
    }

    //--- Method(s) ---
 
    public Component getTableCellRendererComponent(JTable table, Object value,
                                         boolean isSelected, boolean hasFocus,
                                         int row, int column) 
    {
	l.setIcon(getIcon(column));

	Component c = cr.getTableCellRendererComponent(table, value,
						       isSelected, hasFocus,
						       row, column);

	if (column == selectedColumn) {
	    l.setBackground(UIManager.getColor("controlShadow"));
	}

        return c;
    }

    public void selectColumn(int col)
    {
	selectedColumn = col;
    }

    public void setSortedColumn(int col, boolean ascending) 
    {
        icons.clear();
	if (ascending) {
	    icons.put(new Integer(col), downIcon);
	}
	else {
	    icons.put(new Integer(col), upIcon);
	}
    }

    /**
     * Returns the icon of col.  
     */
    public Icon getIcon(int col) 
    {
        return (Icon)icons.get(new Integer(col));
    }
}

/**
 * Provides a small arrow shaped icon.
 */
class BevelArrowIcon implements Icon {
    
    public static final int UP    = 0;
    public static final int DOWN  = 1;
    
    private Color edge1;
    private Color edge2;
    private Color fill;
    private int direction;

    public BevelArrowIcon(int direction, boolean isPressedView) 
    {
	this.direction = direction;

	if (isPressedView) {
	    edge1 = UIManager.getColor("controlDkShadow");
	    edge2 = UIManager.getColor("controlLtHighlight");
	} 
	else {
	    edge1 = UIManager.getColor("controlShadow");
	    edge2 = UIManager.getColor("controlHighlight");
	}
    }

    public int getIconHeight() 
    {
	return 7;
    }
    
    public int getIconWidth() 
    {
	return 8;
    }
		
    public void paintIcon(Component c, Graphics g, int x, int y) 
    {
	switch (direction) {
	case DOWN:
	    g.setColor(edge2);
	    g.drawLine(x + 5, y + 7, x + 8, y);
	    g.setColor(edge1);
	    g.drawLine(x, y, x + 8, y);
	    g.drawLine(x, y, x + 4, y + 7);
	    break;
	case UP:
	    g.setColor(edge1);
	    g.drawLine(x, y + 6, x + 4, y);
	    g.setColor(edge2);
	    g.drawLine(x, y + 7, x + 8, y + 7);
	    g.drawLine(x + 5, y, x + 8, y + 7);
	    break;
	}
    }	

}

