/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package xnap.net;

import xnap.XNap;
import xnap.io.*;
import xnap.net.event.*;
import xnap.util.*;
import xnap.util.event.*;

import java.util.*;
import java.io.*;

public class SimpleUpload extends AbstractTransferContainer 
    implements IUploadContainer {

    //--- Constant(s) ---

    public static final int BUFFER_SIZE = 512;
    
    //--- Data field(s) ---

    private IUpload upload;

    //--- Constructor(s) ---

    public SimpleUpload(IUpload upload)
    {
	this.upload = upload;
    }

    //--- Method(s) ---
   
    public boolean equals(Object o)
    {
    	if (o instanceof SimpleUpload) {
	    return getUpload().equals(((SimpleUpload)o).getUpload());
	}
	return false;
    }

    public File getFile()
    {
	return upload.getFile();
    }

    public IUser getUser()
    {
	return upload.getUser();
    }

    public IUpload getUpload()
    {
	return upload;
    }

    public void reject(String reason)
    {
	setStatus(STATUS_FAILED, reason);
    }

    public void run()
    {
	getUser().modifyLocalUploadCount(1);
	setStatus(STATUS_CONNECTING);

	try {
	    upload.connect();
	} 
	catch (IOException e) {
	    logger.warn("upload connect failed", e);
	    setStatus(STATUS_FAILED, e.getMessage());
	}

	if (!isDone()) {
	    File file = getFile();
	    if (file == null) {
		setStatus(STATUS_FAILED, XNap.tr("Invalid request"));
	    }
	    else {
		upload(file);
	    }
	}

	getUser().modifyLocalUploadCount(-1);
	died();
    }

    private void upload(File file)
    {
	setStatus(STATUS_UPLOADING);
	startTransfer();

	try {
	    //byte[] buf = new byte[BUFFER_SIZE];
	    byte[] buf = new byte[ThrottledOutputStream.getBlocksize()];
	    logger.debug("using a blocksize of " 
			 + ThrottledOutputStream.getBlocksize() + " bytes");
	    int c;

	    BufferedInputStream in 
		= new BufferedInputStream(new FileInputStream(file));

	    long skipped = in.skip(upload.getOffset());
	    if (skipped != upload.getOffset()) {
		throw new IOException("Could not set offset in file: " 
				      + skipped + " / " + upload.getOffset());
	    }
	    totalBytesTransferred = skipped;

	    while (!die) {
		c = in.read(buf, 0, buf.length);
		if (c <= 0) {
		    break;
		}

		upload.write(buf, c);

		totalBytesTransferred += c;
		bytesTransferred += c;
	    }

	    if (!die) {
		setStatus(STATUS_SUCCESS);
	    }
	}
	catch (IOException e) {
	    logger.debug("upload failed", e);
	    setStatus(STATUS_FAILED);
	}
	finally {
	    upload.close();
	}
    }
    
}
