/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.util;

import java.util.*;

/**
 * Provides a set of static methods that help with string parsing and modifying.
 */
public class StringHelper
{
    //--- Constant(s) ---

    public static final String ALPHABET_LOWER = "abcdefghijklmnopqrstuvwxyz";
    public static final String ALPHABET_UPPER = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    public static final String NUMBERS_INT = "0123456789";
    public static final String NUMBERS_DECIMAL = NUMBERS_INT + ".";
    public static final String MONEY_USD = "$" + NUMBERS_DECIMAL;
    public static final String ALPHA_NUM 
	= ALPHABET_LOWER + ALPHABET_UPPER + NUMBERS_INT;
    public static final String EMAIL = ALPHA_NUM + ".@_-";
    public static final String REGULAR_STRING 
	= EMAIL + "[]$%/\\(){}?+-*,;.<>|_^~#";
    public static final String ANYTHING = null;

    //--- Data field(s) ---

    //--- Constructor(s) ---

    //--- Method(s) ---

    /**
     * Converts <code>ip</code> from an integer value to a dotted string
     * representation.
     *
     * @see xnap.util.StringHelper#parseIP(int)
     */
    public static String parseIP(String ip)
    {
	// int would do (it's 32-bit) but the range is too small
	return parseIP(Long.parseLong(ip));
    }

    /**
     * Converts <code>ip</code> from an integer value to a dotted string
     * representation.
     */
    public static String parseIP(long ip)
    {
	StringBuffer sb = new StringBuffer(4 * 3 + 3);
	
	sb.append(ip & 0xFF);
	sb.append(".");
	sb.append((ip >> 8) & 0xFF);
	sb.append(".");
	sb.append((ip >> 16) & 0xFF);
	sb.append(".");
	sb.append((ip >> 24) & 0xFF);

	return sb.toString();
    }

    /**
     * Replaces all occurences of <code>oldChars</code> in <code>s</code> by
     * <code>newChars</code>.
     *
     * @return the modified instance of <code>s</code>
     */
    public static String replaceAll(String s, String oldChars, String newChars)
    {
	StringBuffer sb = new StringBuffer();
	int i = 0;
	while (true) {
	    int j = s.indexOf(oldChars, i);
	    if (j != -1) {
		sb.append(s.substring(i, j));
		sb.append(newChars);
		i = j + oldChars.length();
	    }
	    else {
		sb.append(s.substring(i));
		break;
	    }
	}
	return sb.toString();
    }

    /**
     * Returns a random lower case letter-only string with <code>length</code>
     * characters.
     */
    public static String randomString(int length) 
    {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < length; i++) {
            sb.append((char)Math.round(Math.random() * 25 + 97));
        }

        return sb.toString();
    }

    public static String[] toArray(String value, String separators)
    {
	StringTokenizer st = new StringTokenizer(value, separators);
	String[] array = new String[st.countTokens()];
	for (int i = 0; i < array.length; i++) {
	    array[i] = st.nextToken();
	}
	return array;
    }

    /**
     * Returns a new <code>String</code> that has the first letter of value
     * set to upper case.
     */
    public static String toFirstUpper(String value)
    {
	if (value.length() > 1) {
	    return Character.toUpperCase(value.charAt(0)) + value.substring(1);
	}
	else {
	    return value.toUpperCase();
	}
    }

    public static int[] toIntArray(String value, String separators)
    {
	StringTokenizer st = new StringTokenizer(value, separators);
	int[] array = new int[st.countTokens()];
	for (int i = 0; i < array.length; i++) {
	    try {
		array[i] = Integer.parseInt(st.nextToken());
	    }
	    catch (NumberFormatException e) {
	    }
	}
	return array;
    }

    public static LinkedList toList(String value, String separators)
    {
	StringTokenizer st = new StringTokenizer(value, separators);
	LinkedList list = new LinkedList();
	
	while (st.hasMoreTokens()) {
	    list.add(st.nextToken());
	}
	return list;
    }

    public static String toString(int[] array, String separator)
    {
	StringBuffer sb = new StringBuffer();
	for (int i = 0; i < array.length; i++) {
	    sb.append(array[i]);
	    sb.append(separator);
	}
	return sb.toString();
    }

    public static String toString(String[] array, String separator)
    {
	StringBuffer sb = new StringBuffer();
	for (int i = 0; i < array.length; i++) {
	    sb.append(array[i]);
	    sb.append(separator);
	}
	return sb.toString();
    }

    public static String toString(Collection c, String separator)
    {
	StringBuffer sb = new StringBuffer();
	for (Iterator i = c.iterator(); i.hasNext();) {
	    sb.append(i.next().toString());
	    sb.append(separator);
	}
	return sb.toString();
    }

    /**
     * Removes all characters from <code>s</code> that are not letters. 
     * Returns a new String which can be for instance used as search text.
     *
     * @return a stripped instance of s
     */
    public static String stripExtra(String s)
    {
	StringBuffer sb = new StringBuffer();
	boolean newWord = false;
	char[] chars = s.toCharArray();
	for (int i = 0; i < chars.length; i++) {
	    if (Character.isLetter(chars[i])) {
		if (newWord) {
		    sb.append(" ");
		    newWord = false;
		}
		sb.append(chars[i]);
	    }
	    else {
		newWord = true;
	    }
	}

	return sb.toString();
    }

    /**
     * Returns the index of the first digit in <code>s</code>.
     *
     * @return -1, if <code>s</code> does not contain digits; the index of
     *         the first digit, otherwise
     * @see java.lang.String#indexOf(String)
     */
    public static int indexOfDigit(String s)
    {
	for (int i = 0; i < s.length(); i++) {
	    if (Character.isDigit(s.charAt(i))) {
		return i;
	    }
	}
	    
	return -1;
    }

}
