/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 *  hostif.h - Platform dependent interface for supporting 
 *             the vmx86 device driver. 
 */


#ifndef _HOSTIF_H_
#define _HOSTIF_H_

#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMCORE
#include "includeCheck.h"

#include "vmx86.h"
#include "vcpuset.h"

#ifdef linux
#include "cpuid_info.h"
#endif

/*
 * Host-specific definitions. 
 */
#if !defined(linux) && !defined(WINNT_DDK)
#error "ONLY Linux or NT defined for now "
#endif


#ifdef linux
#   define IRQ_HOST_INTR1_BASE 0x20
#   define IRQ_HOST_INTR2_BASE 0x28

/* See arch/i386/kernel/irq.h in the kernel source --hpreg */
#   define IRQ0_TRAP_VECTOR 0x51
#endif

#ifdef WINNT_DDK
#   define IRQ_HOST_INTR1_BASE 0x30
#   define IRQ_HOST_INTR2_BASE 0x38
#endif

EXTERN int   HostIF_Init(VMDriver *vm);
EXTERN MPN   HostIF_LookupUserMPN(VMDriver *vm, void *addr);
EXTERN void *HostIF_MapCrossPage(VMDriver *vm, void *addr, LA *linearAddr);
EXTERN void  HostIF_InitFP(VMDriver *vm);
EXTERN void  HostIF_InitEvent(VMDriver *vm);

EXTERN void *HostIF_AllocKernelMem(size_t size, int nonPageable);
EXTERN void  HostIF_FreeKernelMem(void *ptr);

EXTERN void *HostIF_AllocPage(void);
EXTERN void  HostIF_FreePage(void *ptr);

/*
 * XXX Those are currently only needed on Windows, but they should go away, as
 *     those functions are _not_ part of the platform-specific interface that
 *     the cross-platform code uses --hpreg
 */
EXTERN void *HostIF_AllocLowPages(VMDriver *vm, MPN* mpns, size_t numPages);
EXTERN int   HostIF_FreeLowPages(VMDriver *vm, void *ptr);

EXTERN int   HostIF_CopyFromUser(void *dst, const void *src, unsigned int len);
EXTERN int   HostIF_CopyToUser(void *dst, const void *src, unsigned int len);
EXTERN MPN   HostIF_LockPage(VMDriver *vm, void *addr, Bool allowMultipleMPNsPerVA);
EXTERN int   HostIF_UnlockPage(VMDriver *vm, void *addr);
EXTERN int   HostIF_UnlockPageByMPN(const VMDriver *vm, MPN mpn, void* va);
EXTERN Bool  HostIF_IsLockedByMPN(VMDriver *vm, MPN mpn);
EXTERN int   HostIF_FreeAllResources(VMDriver *vm);
EXTERN uint64 HostIF_ReadUptime(void);
EXTERN uint64 HostIF_UptimeFrequency(void);
EXTERN unsigned int HostIF_EstimateLockedPageLimit(const VMDriver *vm, 
 						   unsigned int lockedPages);
EXTERN void  HostIF_WaitForFreePages(unsigned int timeoutMs);
EXTERN Bool  HostIF_IsAnonPage(VMDriver *vm, MPN mpn);
EXTERN void  HostIF_InitGlobalLock(void);
EXTERN void  HostIF_GlobalLock(int callerID);
EXTERN void  HostIF_GlobalUnlock(int callerID);
#ifdef VMX86_DEBUG
Bool HostIF_GlobalLockIsHeld(void);
#endif
EXTERN void  HostIF_VMLock(VMDriver *vm, int callerID);
EXTERN void  HostIF_VMUnlock(VMDriver *vm, int callerID);
#ifdef VMX86_DEBUG
Bool HostIF_VMLockIsHeld(VMDriver *vm);
#endif

EXTERN MA    HostIF_APICBase(VMDriver *vm, Bool setVMPtr, Bool probe);
EXTERN uint8 HostIF_APIC_ID(void);

EXTERN void  HostIF_GetCPUID(int index, uint32* regs);
EXTERN uint64 HostIF_RDMSR(int index);
EXTERN void  HostIF_WRMSR(int index, uint64 qword);

EXTERN int   HostIF_SemaphoreWait(VMDriver *vm, Vcpuid vcpuid,
                                  int fd, int timeoutms);
EXTERN int   HostIF_SemaphoreSignal(int fd);
EXTERN void  HostIF_SemaphoreForceWakeup(VMDriver *vm, Vcpuid vcpuid);
EXTERN Bool  HostIF_IPI(VMDriver *vm, VCPUSet vcs, Bool all);

#ifdef WIN32
EXTERN void  HostIF_SetUserCallHandle(VMDriver *vm, int h);
EXTERN int   HostIF_SyncReadTSCs(uint64* TSCs, uintptr_t cpuSet);
EXTERN int   HostIF_SyncWriteTSCs(const uint32* TSCs, uintptr_t cpuSet);
EXTERN void  HostIF_SynchronizeTSCs(void);
EXTERN void  HostIF_SetMemoryParams(const VA64* params, int count);
#endif
EXTERN void  HostIF_UserCall(VMDriver *vm, Vcpuid vcpuid);
EXTERN Bool  HostIF_UserCallWait(VMDriver *vm, Vcpuid vcpuid, int timeoutms);
EXTERN void  HostIF_AwakenVcpu(VMDriver *vm, Vcpuid vcpuid);
EXTERN void  HostIF_AckUserCall(VMDriver *vm, Vcpuid vcpuid);
EXTERN uint32 HostIF_GetCurrentPCPU(void);
EXTERN unsigned int HostIF_NumOnlineLogicalCPUs(void);

#ifdef WIN32
#define HostIF_Reschedule()
EXTERN void HostIF_InitFastClock(void);
EXTERN void HostIF_StartFastClockThread(void);
EXTERN void HostIF_StopFastClockThread(void);
#else
EXTERN void HostIF_Reschedule(void);
#define HostIF_StartFastClockThread() do { } while (0)
#define HostIF_StopFastClockThread() do { } while (0)
#endif

#if defined(USE_PERFCTRS_HOSTED) && defined(linux)
EXTERN MA HostIF_APICEnable(VMDriver *vm);
EXTERN Bool HostIF_APICSetPerfCtr(VMDriver *vm, int gate);
EXTERN void HostIF_APICAck(VMDriver *vm);
#endif 

#ifdef linux
EXTERN void HostIF_GetAllCpuInfo(CPUIDResult *info);
EXTERN int HostIF_MarkLockedVARangeClean(const VMDriver *vm, VA uvAddr, 
                                         unsigned len, VA32 bv);
EXTERN void HostIF_PollListLock(int callerID);
EXTERN void HostIF_PollListUnlock(int callerID);
struct page;
EXTERN void *HostIF_MapUserMem(VA32 addr, size_t size, struct page **page);
EXTERN void HostIF_UnmapUserMem(struct page **page);
#endif

EXTERN int
HostIF_AllocLockedPages(VMDriver *vm, MPN32 *pmpn, 
			unsigned int numPages, Bool kernelMPNBuffer);
EXTERN int
HostIF_FreeLockedPages(VMDriver *vm, const MPN32 *pmpn, 
		       unsigned int numPages, Bool kernelMPNBuffer);
EXTERN int 
HostIF_GetLockedPageList(VMDriver* vm, MPN32* mpns, unsigned int numPages);
EXTERN int
HostIF_MapLockedPages(VMDriver *vm, VA32 va, const MPN32 *pmpn, 
                      unsigned int numPages);
EXTERN int
HostIF_UnmapLockedPages(VMDriver *vm, VA32 va, const MPN32 *pmpn, 
                        unsigned int numPages);

EXTERN int HostIF_ReadPage(MPN mpn, void* buf, Bool kernelBuffer);
EXTERN int HostIF_WritePage(MPN mpn, const void* buf, Bool kernelBuffer);
EXTERN void HostIF_FastClockLock(int callerID);
EXTERN void HostIF_FastClockUnlock(int callerID);
EXTERN int HostIF_SetFastClockRate(unsigned rate);
EXTERN MPN HostIF_COWAllocPage(void);
EXTERN void HostIF_COWFreePage(MPN mpn);

#endif // ifdef _HOSTIF_H_
