/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * staticEscape.c --
 *
 *    Buffer escaping, stolen from hpreg's buffer escaping 
 *    in lib/string, but modified to use bit vectors instead
 *    of arrays, and static buffers instead of dynbufs. [bac]
 *
 */

#ifdef sun
# include <string.h>
#endif

#include "vm_assert.h"
#include "staticEscape.h"


/*
 * Table to use to quickly convert an ASCII hexadecimal digit character into a
 * decimal number. If the input is not an hexadecimal digit character, the
 * output is -1 --hpreg
 */
static int const Hex2Dec[] = {
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9, -1, -1, -1, -1, -1, -1,
   -1, 10, 11, 12, 13, 14, 15, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, 10, 11, 12, 13, 14, 15, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
   -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
};


/*
 * Table to use to quickly convert a decimal number into an ASCII hexadecimal
 * digit character --hpreg
 */
static char const Dec2Hex[] = {
   '0', '1', '2', '3', '4', '5', '6', '7',
   '8', '9', 'A', 'B', 'C', 'D', 'E', 'F',
};


/*
 *-----------------------------------------------------------------------------
 *
 * StaticEscape_Do --
 *
 *    Escape a buffer --hpreg
 *
 *    You can calculate the required size of the output buffer as follows:    
 *    sizeBufOut >= (sizeIn + # of characters that will be escaped +
 *                   sizeof escByte + sizeof '\0')         -[krishnan]
 *
 * Results: 
 *    On success, the size (excluding the NUL terminator) of the
 *    escaped, NUL terminated buffer.
 *    On failure (bufOut not big enough to hold result), negative value.
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

int
StaticEscape_Do(char escByte,                   // IN
                EscBitVector const *bytesToEsc, // IN
                void const *bufIn,              // IN
                uint32 sizeIn,                  // IN
                uint32 sizeBufOut,              // IN
                char *bufOut)                   // OUT
{
   char const *buf;
   unsigned int startUnescaped;
   unsigned int index;
   char escSeq[3];
   int copySize;
   int outPos;

   /* Make sure we won't obviously overflow the bufOut [bac] */
   if (sizeIn > sizeBufOut) {
      return -1;
   }

   ASSERT(bytesToEsc);
   /* Unsigned does matter --hpreg */
   ASSERT(EscBitVector_Test(bytesToEsc, (unsigned char)escByte));
   buf = (char const *)bufIn;
   ASSERT(buf);

   escSeq[0] = escByte;
   startUnescaped = 0;
   outPos = 0;

   for (index = 0; index < sizeIn; index++) {
      /* Unsigned does matter --hpreg */
      unsigned char ubyte;

      ubyte = buf[index];
      if (EscBitVector_Test(bytesToEsc, ubyte)) {
         /* We must escape that byte --hpreg */

         escSeq[1] = Dec2Hex[ubyte >> 4];
	 escSeq[2] = Dec2Hex[ubyte & 0xF];
         copySize = index - startUnescaped;
         if (outPos + copySize + sizeof(escSeq) > sizeBufOut) {
            /*
             * Make sure that both the first chunk and the
             * escape sequence will fit in the bufOut. [bac]
             */
            return -1;
         }
         memcpy(&bufOut[outPos], &buf[startUnescaped], copySize);
         outPos += copySize;
         copySize = sizeof(escSeq);
         memcpy(&bufOut[outPos], escSeq, copySize);
         outPos += copySize;
         startUnescaped = index + 1;
      }
   }

   copySize = index - startUnescaped;
   if (outPos + copySize + 1 > sizeBufOut) {
      /* 
       * Make sure the terminating NUL will fit too, so we don't have
       * to check again below. [bac]
       */
      return -1;
   }
   memcpy(&bufOut[outPos], &buf[startUnescaped], copySize);
   outPos += copySize;
   memcpy(&bufOut[outPos], "", 1);

   return outPos; /* Size of the output buf, not counting NUL terminator */
}


/*
 *-----------------------------------------------------------------------------
 *
 * StaticEscape_Undo --
 *
 *    Unescape a buffer --hpreg
 *
 *    The unescaping is done in place in the input buffer, and
 *    thus can not fail.
 * 
 * Results:
 *    The size (excluding the NUL terminator) of the unescaped, NUL
 *    terminated buffer.
 *    
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

int
StaticEscape_Undo(char escByte,  // IN
                  void *bufIn,   // IN
                  uint32 sizeIn) // IN
{
   char *buf;
   unsigned int state;
   unsigned int startUnescaped;
   unsigned int index;
   int outPos;
   int copySize;
   int h = 0; /* Compiler warning --hpreg */
   int l;

   buf = (char *)bufIn;
   ASSERT(buf);

   outPos = 0;
   startUnescaped = 0;
   state = 0;

   for (index = 0; index < sizeIn; index++) {
      /* Unsigned does matter --hpreg */
      unsigned char ubyte;

      ubyte = buf[index];
      switch (state) {
      case 0: /* Found <byte> --hpreg */
         if (ubyte == escByte) {
            state = 1;
         }
         break;

      case 1: /* Found <escByte><byte> --hpreg */
         h = Hex2Dec[ubyte];
         state = h >= 0 ? 2 : 0;
         break;

      case 2: /* Found <escByte><hexa digit><byte> --hpreg */
         l = Hex2Dec[ubyte];
         if (l >= 0) {
            char escaped;

            escaped = h << 4 | l;

            copySize = index - 2 - startUnescaped;
            memmove(&buf[outPos], &buf[startUnescaped], copySize);
            outPos += copySize;
            memcpy(&buf[outPos], &escaped, 1);
            outPos++;

            startUnescaped = index + 1;
         }
         state = 0;
         break;

      default:
         NOT_IMPLEMENTED();
         break;
      }
   }

   /* Last unescaped chunk (if any) --hpreg */
   copySize = index - startUnescaped;
   memmove(&buf[outPos], &buf[startUnescaped], copySize);
   outPos += copySize;
   memcpy(&buf[outPos], "", 1);

   return outPos;
}


#if 0
/*
Tests for the staticEscape routines. [bac].

To use, copy this to hgfs/tests/testescape.c and compile thus:

gcc -g -I../../public/ -I../lib -I../../lib/public -o testescape ../lib/staticEscape.c testescape.c
 */


#include "escBitvector.h"
#include "staticEscape.h"

#include <stdio.h>

void
Panic(const char *fmt, ...)
{
   printf("PANIC\n");
   exit(1);
}

int
main(int argc, char **argv)
{
   uint32 NAME_MAX = 20;
   EscBitVector toEsc;
   char buf[NAME_MAX + 1];
   char *in;
   int i;
   int result;

   static struct {
      const char *in;
      const char *out;
   } tests[] = {
      { "", "", },
      { "b", "b", },
      { "bac", "bac", },
      { "/", "%2F", },
      { "%", "%25", },
      { "home/bac", "home%2Fbac", },
      { "////", "%2F%2F%2F%2F", },
      { "%/%%", "%25%2F%25%25", },
      { "/bac/", "%2Fbac%2F", },
      { "%bacbac%", "%25bacbac%25", },
      /*
       * The tests from here down should fail because the result is
       * too big to fit in the buffer.
       */
      { "this one is just too big", "", }, // input just to big
      { "first chunk too big/", "", },     // first chunk too big
      { "1st chunk okay/,fail", "", },     // first chunk okay, last chunk too big
   };

   /* Set up bit vector */
   EscBitVector_Init(&toEsc);
   EscBitVector_Set(&toEsc, (unsigned char)'%');
   EscBitVector_Set(&toEsc, (unsigned char)'/');

   /* Test bit vector */
   printf("chars to escape are: ");
   for (i = 0; i < 256; i++) {
      if (EscBitVector_Test(&toEsc, i)) {
         printf("%c ", (unsigned char)i);
      }
   }
   printf("\n\n");


   /* 
    * Test buffer escaping/unescaping.
    *
    * These first 10 tests should pass.
    */
   for (i = 0; i < 10; i++) {
      printf("Test %u: \"%s\", ", i, tests[i].in);
      result = StaticEscape_Do('%',
                               &toEsc,
                               tests[i].in,
                               strlen(tests[i].in),
                               NAME_MAX+1,
                               buf);

      printf("escaped buf is \"%s\". ", buf);
      if (result != strlen(tests[i].out) ||
         strcmp(buf, tests[i].out) != 0)
      {
         printf("escaping test %u failed: \"%s\"\n", i, buf);
         exit(1);
      }

      result = StaticEscape_Undo('%', buf, result);
      if (result != strlen(tests[i].in) ||
         strcmp(buf, tests[i].in) != 0)
      {
         printf("unescaping test %u failed: \"%s\"\n", i, buf);
         exit(1);
      }
      printf("\t\t\t...passed.\n");
   }

   /* These remaining tests should fail. See above. */

   for (i = 10; i < sizeof tests / sizeof tests[0]; i++) {
      printf("Test %u: \"%s\", ", i, tests[i].in);
      result = StaticEscape_Do('%',
                               &toEsc,
                               tests[i].in,
                               strlen(tests[i].in),
                               NAME_MAX+1,
                               buf);
      if (result >= 0) {
         printf("test %u failed to fail\n", i);
         exit(1);
      }

      printf("\t\t\t...passed.\n");
   }

   printf("ALL TESTS PASSED\n");
   exit(0);
}
#endif
