# -*- coding: UTF-8 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import os
import sys
import subprocess
import gettext
from optparse import OptionParser
from urllib2 import URLError

from qt import *
from kdeui import *
from kdecore import *
from RestrictedManager.RestrictedManagerCommon import RestrictedManagerCommon, get_opt_args, RepositoryError
from RestrictedManager.ManagerWindowKDE import ManagerWindowKDE
from RestrictedManager.core import fwurlretrieve,get_handlers
from RestrictedManager.FwHandlerBcm import FwHandlerBcm
from RestrictedManager.FwHandlerProgress import FwHandlerProgress
# encoding and translation fcs
def _(str):
    return unicode(gettext.gettext(str),'UTF-8')

def __(catalog,str):
    ''' searches a different text domain for the translation.
        useful when there are shared strings between
        applications.'''
    return unicode(gettext.dgettext(catalog, str), 'UTF-8')

def utf8(str):
  if isinstance(str, unicode):
      return str
  return unicode(str, 'UTF-8')

# testing if the app is run standalone or from KCM
standalone = hasattr(sys, "argv")

class RestrictedManagerKDE(RestrictedManagerCommon):
    ''' Restricted Manager class for KDE '''
    def __init__(self, parent = None, name = ""):
        # standalone check for KCMShell-specific code
        global standalone
        self.sipwrapper = parent
        self.sipname = name
        if standalone:
            (opts, args) = get_opt_args()
        else:
            opts = None
            args = None
        if name == "":
            appname = 'restricted-manager-kde'
        else:
            appname = name
        description = ''
        version     = ''
        if standalone:
            KCmdLineArgs.init ([sys.argv[0]], appname, description, version)
            a = KApplication()
        self.handlers = get_handlers()
        RestrictedManagerCommon.__init__(self,args,opts)
        self.mngr = None

    def install_package(self, handlerInstance, package):
        ''' uses adept_batch to install the package
            The handlerInstance is passed from the driver handler
            backend.'''

        # process main window events first
        KApplication.kApplication().processEvents()

        try:
            RestrictedManagerCommon.install_package(self, handlerInstance, package)
        except RepositoryError:
            return

        argv = ['/usr/bin/adept_batch', 'install', package]
        #if package_install_xid:
        #    argv += ['--parent-window-id', str(package_install_xid)]
        os.environ['DEBIAN_PRIORITY'] = 'critical'
        num = subprocess.call(argv, stdin=subprocess.PIPE)
        return num

    def remove_package(self, handlerInstance, package):
        ''' uses adept_batch to remove the package
            The handlerInstance is passed from the driver handler
            backend.' '''

        # process main window events first
        KApplication.kApplication().processEvents()
        argv = ['/usr/bin/adept_batch', 'remove', package]
        #if package_install_xid:
        #    argv += ['--parent-window-id', str(package_install_xid)]
        os.environ['DEBIAN_PRIORITY'] = 'critical'
        num = subprocess.call(argv, stdin=subprocess.PIPE)
        return num

    def set_title_label(self, label, in_use=False):
        ''' sets the KDE window label '''
        (bold, normal) = RestrictedManagerCommon.title_label(self, in_use)
        if label != None:
            label.setText('<b>' + bold + '</b><br><br>' + normal)

    def runme_as_root(self,extra_argv = []):
        '''Execute myself through kdesu.'''
        sudo_app = 'kdesu'
        argv = [sudo_app,  ' '.join([sys.argv[0]] + extra_argv)]
        os.execvp(sudo_app, argv)
    
    def show_dialog(self,type,message):
        ''' shows a KDE info or warning dialog '''
        if type == 'error':
          msg = KMessageBox.error(None, message, '')
        elif type == 'info':
          msg = KMessageBox.information(None, message, '')

    def exit(self,return_value = 0):
        ''' checks for standalone before closing the application
            so that systemsettings window does not close
            unexpectedly.'''
        if standalone:
            sys.exit(return_value)      

    def launch_notification(self, title, text, action_text):
        ''' creates a Notification() instance for KDE '''
        RestrictedManagerCommon.launch_notification(self, title, text, action_text)
        nt = self.Notification(self, title, text, action_text)

    def launch_confirm(self, handler, is_enabled, parent=None):
        ''' creates a ConfirmDialog instance for KDE '''
        kapp = KApplication.kApplication()
        kapp.processEvents()
        cdlg = self.ConfirmDialog(handler,is_enabled, parent)
        if cdlg.run() == 3:
            return True
        else:
            return False

    def launch_manager(self,iconify=False):
        ''' creates a ManagerWindow instance for KDE '''
        RestrictedManagerCommon.launch_manager(self)
        if standalone:
            a = KApplication.kApplication()
            self.mngr = self.ManagerWindow(self)
            a.setMainWidget(self.mngr)
            if not iconify:
                self.mngr.show()
                a.exec_loop()
        else:
            return self.ManagerWindow(self)


    class ManagerWindow(RestrictedManagerCommon.ManagerWindow, ManagerWindowKDE):
        ''' main window class for the KDE frontend '''
        def __init__(self,parent):
            self.parent = parent
            ManagerWindowKDE.__init__(self,parent.sipwrapper, parent.sipname)
            RestrictedManagerCommon.ManagerWindow.__init__(self)
            # check for being admin -- useful for greying the window out in systemsettings
            if os.getuid() != 0:
                self.treeview_drivers.setDisabled(True)
                if standalone:
                    self.setButtonApplyText(__("kdelibs","Administrator &Mode..."))
            else:
                if standalone:
                    self.showButtonApply(False)
            self.parent = parent
            self.setCaption(self.caption_text)
            # icon loading
            self.setIcon(KGlobal.iconLoader().loadIcon('memory',KIcon.NoGroup,KIcon.SizeLarge))
            if standalone:
                self.setButtonText(KDialogBase.Close,__('kdelibs','&Close'))
            else:
                self.setButtons(0)            
            icon = QPixmap(KGlobal.iconLoader().loadIcon('memory',KIcon.NoGroup,KIcon.SizeHuge))
            self.logo_image.setPixmap(icon)
            parent.set_title_label(self.label_heading, False)
            self.treeview_drivers.setColumnText(0, self.enabled_text)
            self.treeview_drivers.setColumnText(1, self.component_text)
            self.treeview_drivers.setColumnText(2, self.status_text)
            self.reloadListView()

        def reloadListView(self):
            ''' Reloads the list view with up-to-date information.
                Also changes the main window title.  '''
            item_driver = item_firmware = None
            self.handlers = self.parent.handlers
            self.listview_dict = {}
            self.treeview_drivers.clear()
            any_loaded = False
            for (name, handler) in self.parent.handlers.items():
                is_enabled = handler.is_enabled()
                is_loaded = handler.is_loaded()

                if handler._type == 'Driver':
                    if item_driver == None:
                        item_driver = QListViewItem(self.treeview_drivers, self.driver_text)
                        item_driver.setOpen(True)
                    item = QCheckListItem(item_driver,item_driver,'',QCheckListItem.CheckBox)
                elif handler._type == 'Firmware' and item_firmware == None:
                    if item_firmware == None:
                        item_firmware = QListViewItem(self.treeview_drivers, self.firmware_text)
                        item_firmware.setOpen(True)
                    item = QCheckListItem(item_firmware,item_firmware,'',QCheckListItem.CheckBox)
                self.listview_dict[item] = name
                if(handler.is_enabled() == True):
                    item.activate()
                item.setText(1,utf8(handler.description()))
                if(is_enabled != is_loaded) and handler.is_changed():
                    status, icon = self.restart_text, 'reload'
                    any_loaded = True
                elif is_loaded:
                    status, icon = self.in_use_text, 'button_ok'
                    any_loaded = True
                else:
                    status, icon = self.not_in_use_text, 'no'
                item.setPixmap(2,KGlobal.iconLoader().loadIcon(icon,KIcon.NoGroup,KIcon.SizeSmall))
                item.setText(2,status)
            self.parent.set_title_label(self.label_heading, any_loaded)

        def applied(self):
            ''' triggered when a Administrator Mode button in the standalone version
            is clicked. Used in Notification() for being able to run itself as root. '''
            self.parent.runme_as_root()

        def mouseClicked(self,clickedObject):
            ''' method called when a mouse clicks on the KListView, also when a key is pressed '''
            try:
                if clickedObject.state() == 2:
                    clickedState = True
                else:
                    clickedState = False
                if clickedObject == 0:
                    return
            except AttributeError:
                # user clicked on a button with no checkbox
                return

            # using self.listview_dict created at __init__() time to find the selected handler
            handler = self.handlers[self.listview_dict[clickedObject]]

            if isinstance(clickedObject,QCheckListItem) and handler.is_enabled() != clickedState:
                res = self.parent.launch_confirm(handler,handler.is_enabled())
                try:
                    if res == True:
                        if handler.is_enabled():
                            handler.disable()
                        else:
                            handler.enable()
                #except AttributeError:
                #    pass
                finally:
                    self.reloadListView()
            else:
                clickedObject.setSelected(True)


    class Notification(RestrictedManagerCommon.Notification):
        ''' Implements the notification pop-up and tray icon for KDE '''
        def __init__(self, parent, title, text, action_text):
            RestrictedManagerCommon.Notification.__init__(self, parent, title,
                text, action_text)
            # we have to init the tray before we run exec_loop
            # therefore we set the iconify parameter to True
            parent.launch_manager(iconify=True)
            
            tray = KSystemTray(parent.mngr,'restricted-manager-kde')
            tray.setPixmap(tray.loadIcon('memory'))
            tray.show()
            pop = KPassivePopup(tray)
            box = QVBox(pop)
            label = QLabel('<b>'+self.title+'</b><br><br>'+self.text+'<br><br>'+self.action_text, box)
            pop.setTimeout(14000)
            pop.setView(box)
            pop.show()
            KApplication.kApplication().exec_loop()


    class FWSelectDialog(RestrictedManagerCommon.FWSelectDialog, FwHandlerBcm):
        ''' Firmware selection dialog for KDE. FwHandlerBcm is the parent with the user-interface.'''

        class FWProgressDialog(FwHandlerProgress, RestrictedManagerCommon.FWSelectDialog.FWProgressDialog):
            ''' KDE dialog informing the user about the download progress.
               FwHandlerProgress is the parent with the user-interface.'''
            def __init__(self, parent = None):
                FwHandlerProgress.__init__(self)
                self.buttonCancel.setText(__('kdelibs','&Cancel'))
                self.buttonCancel.setIconSet(KGlobal.iconLoader().loadIconSet('button_cancel',KIcon.NoGroup,KIcon.SizeSmall))
                self.cancelClicked = False
                self.parent = parent
                self.dlProgress.setProgress(0)
                self.show()

            def dl_progress(self, blocks, block_size, total_size):
                ''' calculates the progress and processes the Kapplication events 
                    (moves the progress bar) '''
                if total_size > 0:
                    percentage = (blocks / float(total_size / block_size + 1))*100
                self.dlProgress.setProgress(percentage)
                kapp = KApplication.kApplication()
                kapp.processEvents()
                return self.cancelClicked

            def reject(self):
                ''' clicking Cancel stops the download at the next dl_progress() update'''
                self.cancelClicked = True


        def __init__(self, FSText = None, UrlText = None, FWCheckCommand = None):
            RestrictedManagerCommon.FWSelectDialog.__init__(self, FWCheckCommand)
            FwHandlerBcm.__init__(self)
            # load icons & translations
            self.helpIcon.setPixmap(KGlobal.iconLoader().loadIcon('help',KIcon.NoGroup,KIcon.SizeMedium))
            self.buttonOk.setText(__('kdelibs','&OK'))
            self.buttonOk.setIconSet(KGlobal.iconLoader().loadIconSet('button_ok',KIcon.NoGroup,KIcon.SizeSmall))
            self.buttonCancel.setText(__('kdelibs','&Cancel'))
            self.buttonCancel.setIconSet(KGlobal.iconLoader().loadIconSet('button_cancel',KIcon.NoGroup,KIcon.SizeSmall))
            self.response = False
            self.url = ''
            # insert the text
            self.setCaption(self.label_title)
            self.infoLabel.setText('<h3>%s</h3>%s' % (self.label_title, self.label))
            self.radioButtonLocal.setText(self.radio_local_text)
            self.radioButtonRemote.setText(self.radio_remote_text)
            self.locationGroup.setTitle(self.loc_label_text)
            # insert URLs
            if FSText is not None:
                self.localUrl.setURL(FSText)
            self.remoteUrl.setText(UrlText)
            # disable the OK button, so that the user has to click on one of the options
            self.buttonOk.setDisabled(True)
        
        def run(self):
            ''' triggers the exec loop for the firmare select dialog '''
            self.show()
            self.exec_loop()
            # return the values
            return self.response, self.url
        
        def localUrlClicked(self):
            ''' enable the OK button and disable the remote URL input '''
            self.remoteUrl.setDisabled(True)
            self.localUrl.setDisabled(False)
            self.buttonOk.setDisabled(False)
        
        def remoteUrlClicked(self):
            ''' enable the OK button and disable the local URL input '''
            self.localUrl.setDisabled(True)
            self.remoteUrl.setDisabled(False)
            self.buttonOk.setDisabled(False)
        
        def accept(self):
            ''' clicking OK in the dialog '''
            if(self.radioButtonRemote.isOn() == True):
                # remote file, fetch it
                dlurl = str(self.remoteUrl.text())
                progress_subdialog = self.FWProgressDialog(parent = self)
                progress_subdialog.firmwareName.setText('%s %s' % (self.downloading_text, dlurl))
                progress_subdialog.downloadLabel.setText('<h3>'+self.dl_firmware_text+'</h3>')
                try:
                    result, get, headers = fwurlretrieve(dlurl, reporthook = progress_subdialog.dl_progress)
                    if result == True:
                        self.url = get
                    progress_subdialog.done(0)
                except URLError:
                   progress_subdialog.done(0)
                
            elif(self.radioButtonLocal.isOn() == True):
                # local file
                self.url = self.localUrl.url()
            else:
                return
            # check the firmware
            if(self.check_firmware(self.url) == False):
                # firmware is fine, exit the loop
                self.response = True
                self.done(0)
            else:
                # firmware not fine, keep trying
                # TODO: use show_dialog() here
                KMessageBox.error(self, self.error_fw_invalid_text, '')
 
        def reject(self):
            ''' Triggered when clicking on Cancel in the firmware select dialog. '''
            self.response = False
            self.done(0)


    class ConfirmDialog(RestrictedManagerCommon.ConfirmDialog):
        ''' A confirmation pop-up dialog for KDE '''
        def __init__(self, handler, is_enabled, parent=None):
            RestrictedManagerCommon.ConfirmDialog.__init__(self,handler,is_enabled)
            self.handler = handler
            self.is_enabled = is_enabled
            self.parent = parent
            self.label = '<h3>%s</h3><b>%s</b><br><br>%s' % (self.frob, utf8(self.handler.description()), utf8(self.handler.rationale))

        def run(self):
            cancel = KGuiItem(__('kdelibs','&Cancel'))
            cancel.setIconName('button_cancel')
            # using a custom KMessageBox here, as we're changing the buttons
            # we have to replace '_' (GNOME shorcut key) with '&' (KDE shortcut key)
            ret = KMessageBox.questionYesNo(self.parent, self.label, self.frob, KGuiItem(self.button_label.replace('_', '&')), cancel)
            return ret

def create_RestrictedManagerKDE(parent, name):
    ''' KControl factory function '''
    return RestrictedManagerKDE(parent, name).launch_manager()
